#!/bin/bash
#: Title      : test_simimyoc
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../simimyoc$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f radiowater.fit ] || [ ! -f frames.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi


if [ ! -f simple.dat ]; then
  printf "\n Make simple data file \n"
  printf "1 100 0 0\n" > simple.dat
  printf "\n ok \n"
fi


if [ ! -f blood.tac ] || [ ! -f radiowater.tac ]; then
  rm -f myocardium.tac background.tac
  printf "\n Make BTAC in seconds \n"
  fit2dat --silent -c=0,600,1 radiowater.fit radiowater.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tactime -nogap radiowater.dat 32.5 radiowater.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacstudy --force radiowater.tac sim1 
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec radiowater.tac frames.dat blood.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi



if [ ! -f myocardium.tac ]; then
  rm -f tacs.tac
  printf "\n Simulate myocardial TTAC \n"
  b2t_h2o -nosub -fpt -voiname=myoc radiowater.tac 100 0.91 1 10 50 myocardium.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec myocardium.dat frames.dat myocardium.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f background.tac ]; then
  rm -f tacs.tac
  printf "\n Simulate background TTAC \n"
  b2t_h2o -nosub -fpt -voiname=bkg radiowater.tac 50 0.97 1 10 50 background.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec background.dat frames.dat background.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


if [ ! -f tacs.tac ]; then
  printf "\n Make TACs file \n"
  tacadd -ovr tacs.tac myocardium.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd tacs.tac blood.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd tacs.tac background.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren tacs.tac 2 blood__
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes tacs.svg tacs.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n Testing for general functionality \n"

printf "\n 1.0.0 \n"
printf " Test case: Default settings, reasonable data. \n"
printf " Expected result: No errors, reasonable image. \n\n"

rm -f output.v output.tif

$PROGRAM --verbose tacs.tac output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s output.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Set all options except -3d because that is slow. \n"
printf " Expected result: No errors, reasonable image. \n\n"

rm -f output2.v output2.tif

$PROGRAM --silent -dim=200 -pxlsize=0.5 -fwhm=4 -thickness=13 -diameter=40 -diamin=0.9 tacs.tac output2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s output2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Set all options including -3d for simple data. \n"
printf " Expected result: No errors, reasonable image. \n\n"

rm -f output2_3d.v output2_3d.tif

$PROGRAM -3D -dim=100 -pxlsize=1.0 -fwhm=4 -thickness=13 -diameter=40 -diamin=0.9 simple.dat output2_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s output2_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.0 \n"
printf " Test case: Simulate no heart beating and no resolution effect. \n"
printf " Expected result: No errors, reasonable image. \n\n"

rm -f output3.v output3.tif

$PROGRAM -fwhm=0 -diamin=1.0 tacs.tac output3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s output3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf " Testing for quantitativity \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case: Simulate 2D no heart beating and no resolution effect. \n"
printf " Expected result: Sum of 2D images must be correct. \n\n"

rm -f quant1.v quant1.tif quant1head.tac quant1head.ift

$PROGRAM -fwhm=0 -diameter=50 -thickness=10 -diamin=1 -dim=100 -pxlsize=1 simple.dat quant1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s quant1.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imghead -sum quant1.v quant1head.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift quant1head.tac > quant1head.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True sum must be:
# Act*(Area1-Area2)
# 100*Pi*((70/2) pot 2 - (50/2) pot 2) = 1,8850E+05
iftisval -abs=10 quant1head.ift "value[1][1]" 188600
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Simulate 2D beating but no resolution effect. \n"
printf " Expected result: Sum of 2D images must be correct. \n\n"

rm -f quant2.v quant2.tif quant2head.tac quant2head.ift

$PROGRAM -fwhm=0 -diameter=50 -thickness=10 -diamin=0.6 -dim=100 -pxlsize=1 simple.dat quant2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s quant2.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imghead -sum quant2.v quant2head.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift quant2head.tac > quant2head.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True sum must be:
# Act*(Area1-Area2)
# 100*Pi*((70/2) pot 2 - (50/2) pot 2) = 1,8850E+05
iftisval -abs=100 quant2head.ift "value[1][1]" 188600
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.2 \n"
printf " Test case: Simulate 2D beating and resolution effect. \n"
printf " Expected result: Sum of 2D images must be correct. \n\n"

rm -f quant3.v quant3.tif quant3head.tac quant3head.ift

$PROGRAM -fwhm=10 -diameter=50 -thickness=10 -diamin=0.6 -dim=100 -pxlsize=1 simple.dat quant3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s quant3.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imghead -sum quant3.v quant3head.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift quant3head.tac > quant3head.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True sum must be:
# Act*(Area1-Area2)
# 100*Pi*((70/2) pot 2 - (50/2) pot 2) = 1,8850E+05
iftisval -abs=200 quant3head.ift "value[1][1]" 188600
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n-------------------------------------------------------------------\n"

printf "\n 2.1.0 \n"
printf " Test case: Simulate 3D no heart beating and no resolution effect. \n"
printf " Expected result: Sum of 3D images must be correct. \n\n"

rm -f quant1_3d.v quant1_3d.tif quant1head_3d.tac quant1head_3d.ift

$PROGRAM -3d -fwhm=0 -diameter=50 -thickness=10 -diamin=1 -dim=100 -pxlsize=1 simple.dat quant1_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s quant1_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imghead -sum quant1_3d.v quant1head_3d.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift quant1head_3d.tac > quant1head_3d.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True sum must be:
# Act*(Vol1-Vol2) where Vol=(Pi*diameter pot 3)/6
# 100*Pi*((70 pot 3)-(50 pot 3))/6 = 1,141445E+07
iftisval -abs=2E4 quant1head_3d.ift "value[1][1]" 1.141445E+07
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Simulate 3D beating but no resolution effect. \n"
printf " Expected result: Sum of 3D images must be correct. \n\n"

rm -f quant2_3d.v quant2_3d.tif quant2head_3d.tac quant2head_3d.ift

$PROGRAM -3D -fwhm=0 -diameter=50 -thickness=10 -diamin=0.6 -dim=100 -pxlsize=1 simple.dat quant2_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s quant2_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imghead -sum quant2_3d.v quant2head_3d.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift quant2head_3d.tac > quant2head_3d.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True sum must be:
# Act*(Vol1-Vol2) where Vol=(Pi*diameter pot 3)/6
# 100*Pi*((70 pot 3)-(50 pot 3))/6 = 1,141445E+07
iftisval -abs=2E4 quant2head_3d.ift "value[1][1]" 1.141445E+07
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.2 \n"
printf " Test case: Simulate 3D beating and resolution effect. \n"
printf " Expected result: Sum of 3D images must be correct. \n\n"

rm -f quant3_3d.v quant3_3d.tif quant3head_3d.tac quant3head_3d.ift

$PROGRAM -3D -fwhm=10 -diameter=50 -thickness=10 -diamin=0.6 -dim=100 -pxlsize=1 simple.dat quant3_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s quant3_3d.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
imghead -sum quant3_3d.v quant3head_3d.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift quant3head_3d.tac > quant3head_3d.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True sum must be:
# Act*(Vol1-Vol2) where Vol=(Pi*diameter pot 3)/6
# 100*Pi*((70 pot 3)-(50 pot 3))/6 = 1,141445E+07
iftisval -abs=2E4 quant3head_3d.ift "value[1][1]" 1.141445E+07
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case: Optional beating phase image. \n"
printf " Expected result: Sum of image voxels must be correct. \n\n"

rm -f bif.v bif.tif bif.tac bif.ift

$PROGRAM -bif=bif.v -fwhm=10 -diameter=50 -thickness=10 -diamin=0.6 -dim=100 -pxlsize=1 simple.dat temp.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2tif -s bif.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
imghead bif.v bif.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclist -ift bif.tac > bif.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# True mean activity must be:
# Act*(Area1-Area2)/pxlnr
# 100*Pi*((70/2)^2 - (50/2)^2) / 100*100= 1,8850E+01
iftisval -abs=0.02 bif.ift "value[1][1]" 18.850
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 3.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption simple.dat temp.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM simple.dat temp.v extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.0.2 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM simple.dat  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n-------------------------------------------------------------------\n"

printf "\n 3.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.tac temp.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error in reading 'nonexistingfile.tac'" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

