#!/bin/bash
#: Title      : test_imgsuv
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../imgsuv$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f original.dft ] ; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

printf "\n making test data \n"

if [ ! -f calibrated.dft ] || [ ! -f calibrated2.dft ]; then
  cp -f original.dft calibrated.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -yset=kBq/cc calibrated.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f calibrated.dft calibrated2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacunit -yconv=Bq/cc calibrated2.dft
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f original.v ] || [ ! -f calibrated.v ] || [ ! -f calibrated2.v ]; then
  dft2img -scanner=HR+ -dim=8 original.dft original.v original.roi
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esetstrt original.v 2011-02-04 11:32:00
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgdecay -dc -i=C-11 original.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f original.v calibrated.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -u=kBq/cc calibrated.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f calibrated.v calibrated2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  imgunit -uc=Bq/cc calibrated2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f static.v ] || [ ! -f static2.v ]; then
  esplit calibrated.v  10 1 static.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  esplit calibrated2.v 10 1 static2.v
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: No command line options or arguments. \n"
printf " Expected result: Error. \n\n"

$PROGRAM 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Usage"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "See also"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption calibrated.v 0 0 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Valid arguments. \n"
printf " Expected result: Correctly identified by application. \n\n"

rm -f suv.v
$PROGRAM -d2 calibrated.v 0 0 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt petfile "calibrated.v"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt suvfile "suv.v"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt dose "200 MBq"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt weight "50 kg"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.3 \n"
printf " Test case: Too many arguments. \n"
printf " Expected result: Error. \n\n"

$PROGRAM calibrated.v 0 0 200 50 suv.v extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.4 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM  0 0 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM calibrated.v  0 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM calibrated.v 0 0 200 50  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Basic tests \n"
printf "===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: dynamic SUV calculation when data is in kBq/mL. \n"
printf " Expected result 1: Dynamic image in SUV units is made. \n"
printf " Expected result 2: SUVs are correct. \n\n"

rm -f calibrated.suv suv0-120.dft suv.v suv.dft
dftsuv -curve=calibrated.suv calibrated.dft 0 120 200 50 suv0-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM calibrated.v 0 0 200 50 suv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft suv.v original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y calibrated.suv suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case 1: static SUV from dynamic image when data is in kBq/mL. \n"
printf " Expected result 1: Static image in SUV units is made. \n"
printf " Expected result 2: SUVs are correct. \n\n"

rm -f suv.v suv.dft

$PROGRAM calibrated.v 0 120 200 50 suv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft suv.v original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# File for comparison was made in previous test
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv0-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case 1: static SUV from dynamic image when data is in Bq/mL. \n"
printf " Expected result 1: Static image in SUV units is made. \n"
printf " Expected result 2: SUVs are correct. \n\n"

rm -f suv.v suv.dft

$PROGRAM calibrated2.v 0 120 200 50 suv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft suv.v original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
# File for comparison was made in a previous test
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv0-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case 1: static SUV from dynamic image from shorter time range. \n"
printf " Expected result 1: Static image in SUV units is made. \n"
printf " Expected result 2: SUVs are correct. \n\n"

rm -f suv5-20.dft suv.v suv.dft
dftsuv calibrated.dft 5 20 200 50 suv5-20.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM calibrated2.v 5 20 200 50 suv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft suv.v original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv5-20.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.3.0 \n"
printf " Test case 1: static PID from dynamic image. \n"
printf " Expected result 1: Static image in PID units is made. \n"
printf " Expected result 2: PIDs are correct. \n\n"

rm -f pid90-120.dft pid.v pid.dft
dftsuv calibrated.dft 90 120 200 0 pid90-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM calibrated2.v 90 120 200 0 pid.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft pid.v original.roi pid.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y pid90-120.dft pid.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.4.0 \n"
printf " Test case 1: SUV from static image when user does not give time range. \n"
printf " Expected result 1: Static image in SUV units is made. \n"
printf " Expected result 2: SUVs are correct. \n\n"

rm -f suv60-120.dft suv.v suv.dft
dftsuv calibrated.dft 60 120 200 50 suv60-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM static.v 0 0 200 50 suv.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft suv.v original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv60-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.1 \n"
printf " Test case 1: SUV from static image when user gives some time range. \n"
printf " Expected result 1: Static image in SUV units is made. \n"
printf " Expected result 2: SUVs are correct. \n"
printf " Expected result 3: Warning on unused timerange. \n\n"

rm -f suv.v suv.dft

$PROGRAM static.v 30 120 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft suv.v original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv60-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

iftisval stdout.txt "Warning" "user-specified time range is not used."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.4.2 \n"
printf " Test case 1: PID from static image when units in Bq/cc. \n"
printf " Expected result 1: Static image in PID units is made. \n"
printf " Expected result 2: PIDs are correct. \n\n"

rm -f pid60-120.dft pid.v pid.dft
dftsuv calibrated2.dft 60 120 200 0 pid60-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM static2.v 0 0 200 0 pid.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft pid.v original.roi pid.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y pid60-120.dft pid.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Testing for error reporting \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: ECAT image does not contain calibration units. \n"
printf " Expected result: Error. \n\n"

rm -f suv.v

$PROGRAM original.v 0 120 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f suv.v ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid calibration unit in image."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: User gives too large time range. \n"
printf " Expected result: Error. \n\n"

rm -f suv.v

$PROGRAM calibrated.v 0 9120 200 50 suv.v 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ -f suv.v ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "integration time range oversteps data range."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Testing for other image formats than ECAT7 \n"
printf "===================================================================\n"

printf "\n 4.0.0 \n"
printf " Test case: dynamic SUV from dynamic NIfTI image with SIF. \n"
printf " Expected result: Results are correct, when units in kBq/mL. \n\n"

rm -f nifti/*
ecat2nii -o=nifti -sif calibrated.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftsuv -curve=calibrated.suv calibrated.dft 0 0 200 50 suv0-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f suv.dft
$PROGRAM nifti/calibrated 0 0 200 50 nifti/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft nifti/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y calibrated.suv suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case: static SUV from dynamic NIfTI image with SIF. \n"
printf " Expected result: Results are correct \n\n"

rm -f nifti/suv.*
rm -f suv.dft

$PROGRAM nifti/calibrated 0 120 200 50 nifti/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft nifti/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv0-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.2 \n"
printf " Test case: dynamic SUV from dynamic NIfTI image without SIF. \n"
printf " Expected result: Results are correct, just times are missing \n\n"

rm -f nifti/*.sif nifti/suv.*
rm -f suv.dft

$PROGRAM nifti/calibrated 0 0 200 50 nifti/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft nifti/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=n -y=y -xunit=n calibrated.suv suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.3 \n"
printf " Test case: static SUV from dynamic NIfTI image without SIF. \n"
printf " Expected result: Error. \n\n"

rm -f nifti/*.sif nifti/suv.*
rm -f suv.dft

$PROGRAM nifti/calibrated 0 120 200 50 nifti/suv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "nifti/calibrated does not contain frame times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.0.4 \n"
printf " Test case: static SUV from static NIfTI image with SIF. \n"
printf " Expected result: Results are correct, when units in kBq/mL. \n\n"

rm -f nifti/*
ecat2nii -o=nifti -sif static.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftsuv calibrated.dft 60 120 200 50 suv60-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f suv.dft
$PROGRAM nifti/static 0 0 200 50 nifti/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft nifti/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv60-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.5 \n"
printf " Test case: static SUV from static NIfTI image without SIF. \n"
printf " Expected result: Results are correct, just times are missing. \n\n"

rm -f nifti/*.sif nifti/suv.*
rm -f suv.dft 

$PROGRAM nifti/static 0 120 200 50 nifti/suv 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft nifti/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=n -y=y -xunit=n suv60-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n--------------------------------------------------------------\n\n"



printf "\n 4.1.0 \n"
printf " Test case: dynamic SUV from dynamic Analyze image with SIF. \n"
printf " Expected result: Results are correct, when units in kBq/mL. \n\n"

rm -f ana/*
ecat2ana -o=ana -sif calibrated.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftsuv -curve=calibrated.suv calibrated.dft 0 0 200 50 suv0-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f suv.dft
$PROGRAM ana/calibrated 0 0 200 50 ana/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y calibrated.suv suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.1 \n"
printf " Test case: static SUV from dynamic Analyze image with SIF. \n"
printf " Expected result: Results are correct \n\n"

rm -f ana/suv.*
rm -f suv.dft
dftsuv calibrated.dft 30 120 200 50 suv30-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM ana/calibrated 30 120 200 50 ana/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv30-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.2 \n"
printf " Test case: dynamic SUV from dynamic Analyze image without SIF. \n"
printf " Expected result: Results are correct, just times are missing \n\n"

rm -f ana/*.sif ana/suv.*
rm -f suv.dft

$PROGRAM ana/calibrated 0 0 200 50 ana/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=n -y=y -xunit=n calibrated.suv suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.3 \n"
printf " Test case: static SUV from dynamic Analyze image without SIF. \n"
printf " Expected result: Error. \n\n"

rm -f ana/*.sif ana/suv.*
rm -f suv.dft

$PROGRAM ana/calibrated 30 120 200 50 ana/suv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "ana/calibrated does not contain frame times."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.1.4 \n"
printf " Test case: static SUV from static Analyze image with SIF. \n"
printf " Expected result: Results are correct, when units in kBq/mL. \n\n"

rm -f ana/*
ecat2ana -o=ana -sif static.v
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftsuv calibrated.dft 60 120 200 50 suv60-120.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

rm -f suv.dft
$PROGRAM ana/static 0 0 200 50 ana/suv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=y -y=y -xunit=y suv60-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.5 \n"
printf " Test case: static SUV from static Analyze image without SIF. \n"
printf " Expected result: Results are correct, just times are missing. \n\n"

rm -f ana/*.sif ana/suv.*
rm -f suv.dft

$PROGRAM ana/static 90 120 200 50 ana/suv 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
img2dft ana/suv original.roi suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.001 -rel=0.1 -x=n -y=y -xunit=n suv60-120.dft suv.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



rm -f ana/* nifti/*

printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

