/** @file dft2res.c
 *  @brief Convert TAC files into result file format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Test with PMOD format.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts TAC files into result file format (*.res).",
  "This may be of use when regional values from parametric images need to",
  "be processed further.",
  " ",
  "Usage: @P [Options] tacfile(s)",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: rescoll, tacformat, dft2html, img2dft",
  " ",
  "Keywords: results, TAC, modelling, conversion, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret, fileNr=0, ffi=0;
  char  *cptr, tmp[FILENAME_MAX];
  char   dftfile[FILENAME_MAX], resfile[FILENAME_MAX];
  DFT    dft;
  RES    res;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dftfile[0]=resfile[0]=(char)0;
  dftInit(&dft); resInit(&res);
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(ffi<1) ffi=ai;
    fileNr++;
  }

  /* Is something missing? */
  if(fileNr<1) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("fileNr := %d\n", fileNr);
  }
  if(verbose>3) RESULT_TEST=verbose-3; else RESULT_TEST=0;


  /*
   *  Process one file at a time
   */
  for(ai=ffi; ai<argc; ai++) {

    strcpy(dftfile, argv[ai]);
    if(verbose>0 && fileNr>1) fprintf(stdout, "%s :\n", dftfile);

    /*
     *  Read DFT data
     */
    if(verbose>2) printf("reading '%s'\n", dftfile);
    if(dftRead(dftfile, &dft)) {
      fprintf(stderr, "Error in reading '%s': %s\n", dftfile, dfterrmsg);
      return(2);
    }

    /*
     *  Copy DFT data into RES structure
     */
    if(verbose>2) printf("converting\n");
    ret=dftToResult(&dft, &res, tmp);
    if(ret>0) {
      fprintf(stderr, "Error: %s.\n", tmp);
      dftEmpty(&dft); resEmpty(&res); return(3);
    } else if(ret<0) {
      fprintf(stderr, "Warning: %s.\n", tmp);
    }
    /* Set more header information */
    tpcProgramName("result", 1, 1, res.program, 256);
    cptr=strrchr(dftfile, '/'); if(cptr==NULL) cptr=strrchr(dftfile, '\\');
    if(cptr==NULL) cptr=dftfile; else cptr++; strcpy(res.datafile, cptr);

    /*
     *  Construct result filename
     */
    strcpy(resfile, dftfile); cptr=strrchr(resfile, '.');
    if(cptr!=NULL) *cptr=(char)0; 
    strcat(resfile, ".res");

    /*
     *  Write result file
     */
    if(verbose>0) fprintf(stdout, "  writing %s\n", resfile);
    ret=resWrite(&res, resfile, verbose-3);
    if(ret) {
      fprintf(stderr, "  Error (%d) in writing file %s\n", ret, resfile);
      dftEmpty(&dft); resEmpty(&res); return(11);
    }
     
    /* close file and clear TAC from memory */
    dftEmpty(&dft); resEmpty(&res);

  } /* next file */

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
