/** @file dftmax.c
 *  @brief Finds maximum value and its time in DFT file(s).
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
#include "libtpccurveio.h"
#include "libtpcmodel.h"
#include "libtpcmodext.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Finds the maximum (or minimum) value and its time in TAC file(s).",
  " ",
  "Usage: @P [Options] tacfile(s)",
  " ",
  "Options:",
  " -min",
  "     Find the minimum value, instead of maximum.",
  " -c[lean]",
  "     Only the maximum or minimum value of ALL specified files is printed.",
  " -x[=<y|N>]",
  "     Time of maximum is printed (y) or not (n, default).",
  " -f[=<y|N>]",
  "     Time frame (1..sampleNr) of maximum is printed (y) or not (n, default).",
  " -y[=<Y|n>]",
  "     Maximum value is printed (y, default) or not (n).",
  " -robust",
  "     Search is limited to the longest range above and below median", 
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1: Get maximum of all TACs into DOS variable 'MAXVAL':",
  "     @P -clean *.tac > max.txt",
  "     set /p MAXVAL=<max.txt",
  " ",
  "See also: tacpeak, tacunit, tacframe, tacrange, imgmax",
  " ",
  "Keywords: TAC, peak, max, min",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int    ai, help=0, version=0, verbose=1;
  int    ret, fileNr=0, firstfile=0;
  int    cleanOutput=0;
  int    robust_mode=0; // 0=normal, 1=robust
  int    mode=0; // 0=max, 1=min
  int    print_time=0;
  int    print_frame=0;
  int    print_value=1;
  int    mini, maxi;
  int    mins, maxs, allmins=-1, allmaxs=-1;
  char   dfile[FILENAME_MAX], *cptr=NULL, dftunit[128], dfttunit[128];
  DFT    dft;
  double tacmax=-1.0E22, allmax=-1.0E22;
  double tacmin=+1.0E22, allmin=+1.0E22;
  double xmin, xmax, allxmin=+1.0E22, allxmax=-1.0E22;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dftInit(&dft);
  dfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1;
    if(strncasecmp(cptr, "CLEAN", 1)==0) {
      cleanOutput=1; continue;
    } else if(strncasecmp(cptr, "ROBUST", 3)==0) {
      robust_mode=1; continue;
    } else if(strcasecmp(cptr, "MIN")==0) {
      mode=1; continue;
    } else if(strcasecmp(cptr, "MAX")==0) {
      mode=0; continue;
    } else if(strncasecmp(cptr, "TIME", 4)==0) {
      if(strlen(cptr)==4) {print_time=1; continue;}
      cptr+=4; if(*cptr=='=') {
        cptr++;
        if(*cptr=='Y' || *cptr=='y') {print_time=1; continue;}
        if(*cptr=='N' || *cptr=='n') {print_time=0; continue;}
      }
    } else if(strncasecmp(cptr, "X", 1)==0) {
      if(strlen(cptr)==1) {print_time=1; continue;}
      cptr+=1; if(*cptr=='=') {
        cptr++;
        if(*cptr=='Y' || *cptr=='y') {print_time=1; continue;}
        if(*cptr=='N' || *cptr=='n') {print_time=0; continue;}
      }
    } else if(strncasecmp(cptr, "F", 1)==0) {
      if(strlen(cptr)==1) {print_frame=1; continue;}
      cptr+=1; if(*cptr=='=') {
        cptr++;
        if(*cptr=='Y' || *cptr=='y') {print_frame=1; continue;}
        if(*cptr=='N' || *cptr=='n') {print_frame=0; continue;}
      }
    } else if(strncasecmp(cptr, "Y", 1)==0) {
      if(strlen(cptr)==1) {print_value=1; continue;}
      cptr+=1; if(*cptr=='=') {
        cptr++;
        if(*cptr=='Y' || *cptr=='y') {print_value=1; continue;}
        if(*cptr=='N' || *cptr=='n') {print_value=0; continue;}
      }
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]); 
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(firstfile==0) firstfile=ai;
    fileNr++;
  }
  /* Did we get all the information that we need? */
  if(fileNr==0) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  /* All x, frame, and y printing must not be turned off */
  if(!print_time && !print_frame && !print_value) {
    fprintf(stderr, "Error: user turned off both printings.\n"); 
    return(1);
  }

  /* In verbose mode print options */
  if(verbose>1) {
    printf("mode := %d\n", mode);
    printf("robust_mode := %d\n", robust_mode);
    printf("print_time := %d\n", print_time);
    printf("print_frame := %d\n", print_frame);
    printf("print_value := %d\n", print_value);
    printf("fileNr := %d\n", fileNr);
  }


  /*
   *  Process each file
   */
  fileNr=0;
  for(ai=firstfile; ai<argc; ai++) {
    strcpy(dfile, argv[ai]); 
    tacmax=-1.0E22; tacmin=+1.0E22;
    /* Read TAC file */
    ret=dftRead(dfile, &dft);
    if(ret) {
      if(verbose>1) printf("dfile := %s\n", dfile);
      if(verbose>1) printf("ret := %d\n", ret);
      if(fileNr>1) {fprintf(stderr, "Warning: %s\n", dfterrmsg); continue;}
      else {fprintf(stderr, "Error: %s\n", dfterrmsg); return(2);}
    }
    if(verbose>2) printf("frameNr:=%d\nvoiNr:=%d\n", dft.frameNr, dft.voiNr);
    /* Get maximum pixel value */
    if(robust_mode==0)
      ret=dftMinMaxTAC(&dft, -1, &xmin, &xmax, &tacmin, &tacmax, &mini, &maxi,
                       &mins, &maxs);
    else
      ret=dftRobustMinMaxTAC(&dft, -1, &xmin, &xmax, &tacmin, &tacmax,
                             &mini, &maxi, &mins, &maxs, verbose-3);
    if(ret!=0) {
      fprintf(stderr, "Warning: cannot find min/max in %s\n", dfile);
      if(verbose>1) printf("ret := %d\n", ret);
      continue;
    }
    strcpy(dftunit, dft.unit);
    strcpy(dfttunit, petTunit(dft.timeunit));
    if(tacmax>allmax) {allmax=tacmax; allxmax=xmax; allmaxs=maxs;}
    if(tacmin<allmin) {allmin=tacmin; allxmin=xmin; allmins=mins;}
    dftEmpty(&dft); fileNr++;
    if(cleanOutput) continue; // not printing individual max or min

    if((print_time+print_frame+print_value)==1) {
      /* Print the requested result on the same line with filename */
      fprintf(stdout, "%s : ", dfile); fflush(stdout);
      if(print_value) { 
        if(mode==0) fprintf(stdout, "%g %s\n", tacmax, dftunit );
        else fprintf(stdout, "%g %s\n", tacmin, dftunit );
      } else if(print_time) {
        if(mode==0) fprintf(stdout, "%g %s\n", xmax, dfttunit );
        else fprintf(stdout, "%g %s\n", xmin, dfttunit );
      } else {
        if(mode==0) fprintf(stdout, "%d\n", 1+maxs );
        else fprintf(stdout, "%d\n", 1+mins );
      }
    } else {
      /* Print first file name on its own line, and then the requested results */
      fprintf(stdout, "%s : \n", dfile);
      if(mode==0) {
        if(print_value) fprintf(stdout, "max_value := %g %s\n", tacmax, dftunit);
        if(print_time) fprintf(stdout, "max_time := %g %s\n", xmax, dfttunit);
        if(print_frame) fprintf(stdout, "max_frame := %d\n", 1+maxs);
        fprintf(stdout, "tac := %d\n", maxi+1);
      } else {
        if(print_value) fprintf(stdout, "min_value := %g %s\n", tacmin, dftunit);
        if(print_time) fprintf(stdout, "min_time := %g %s\n", xmin, dfttunit);
        if(print_frame) fprintf(stdout, "min_frame := %d\n", 1+mins);
        fprintf(stdout, "tac := %d\n", mini+1);
      }
    }
  } /* next file */


  /*
   *  Print the max or min of all files
   */
  if(cleanOutput) {
    if((print_time+print_frame+print_value)==1) {
      if(print_value) {
        if(mode==0) fprintf(stdout, "%g\n", allmax);
        else fprintf(stdout, "%g\n", allmin);
      } else if(print_time) {
        if(mode==0) fprintf(stdout, "%g\n", allxmax);
        else fprintf(stdout, "%g\n", allxmin);
      } else if(print_frame) {
        if(mode==0) fprintf(stdout, "%d\n", 1+allmaxs);
        else fprintf(stdout, "%d\n", 1+allmins);
      }
    } else {
      if(mode==0) {
        if(print_value) fprintf(stdout, "max_value := %g\n", allmax);
        if(print_time) fprintf(stdout, "max_time := %g\n", allxmax);
        if(print_frame) fprintf(stdout, "max_frame := %d\n", 1+allmaxs);
      } else {
        if(print_value) fprintf(stdout, "min_value := %g\n", allmin);
        if(print_time) fprintf(stdout, "min_time := %g\n", allxmin);
        if(print_frame) fprintf(stdout, "min_frame := %d\n", 1+mins);
      }
    }
    return(0);
  }

  /* Individual values were printed before, thus no reason to print
     max or min of all, if only one TAC was given */
  if(fileNr<1) return(0);
  if((print_time+print_frame+print_value)==1) {
    if(print_value) {
      if(mode==0) fprintf(stdout, "Maximum of all files : %g %s\n", allmax, dftunit );
      else fprintf(stdout, "Minimum of all files : %g %s\n", allmin, dftunit );
    } else if(print_time) {
      if(mode==0) fprintf(stdout, "Time at maximum of all files : %g %s\n", allxmax, dfttunit );
      else fprintf(stdout, "Time at minimum of all files : %g %s\n", allxmin, dfttunit );
    } else if(print_frame) {
      if(mode==0) fprintf(stdout, "Frame of maximum of all files : %d\n", 1+allmaxs);
      else fprintf(stdout, "Frame of minimum of all files : %d\n", 1+allmins);
    }
  } else {
    if(mode==0) {
      if(print_value) fprintf(stdout, "overall_max_value := %g\n", allmax);
      if(print_time) fprintf(stdout, "overall_max_time := %g\n", allxmax);
      if(print_frame) fprintf(stdout, "overall_max_frame := %d\n", 1+allmaxs);
    } else {
      if(print_value) fprintf(stdout, "overall_min_value := %g\n", allmin);
      if(print_time) fprintf(stdout, "overall_min_time := %g\n", allxmin);
      if(print_frame) fprintf(stdout, "overall_min_frame := %d\n", 1+mins);
    }
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
