/** @file asc2flat.c
    @brief Reads values from an ASCII file and writes those in a binary
           file as 4-byte floats.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Reads numerical data from an ASCII file and writes those in a binary file",
  "as 4-byte floats.",
  "Numerical data can be stored on one or more lines, separated by",
  "commas, tabs, semi-colons, or spaces.",
  "Comment lines starting with '#' or '//' are allowed (and ignored) in", 
  "the beginning of the file.",
  " ",
  "Usage: @P [Options] ASCII_file [binary_file]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: flat2nii, flat2img, simcirc, dft2img, simiart, imgadd, img2tif",
  " ",
  "Keywords: software testing, simulation, image, mask",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int           ai, help=0, version=0, verbose=1;
  unsigned int  vi, isComment;
  char         *cptr, datfile[FILENAME_MAX], binfile[FILENAME_MAX], buf[1024];
  FILE         *fp1, *fp2=NULL;
  float         f;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  datfile[0]=binfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!datfile[0]) {
      strlcpy(datfile, argv[ai], FILENAME_MAX); continue;
    } else if(!binfile[0]) {
      strlcpy(binfile, argv[ai], FILENAME_MAX); continue;
    }
    /* We should not be here */
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!datfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; try %s --help\n",
      argv[0]);
    return(1);
  }
  
  /*
   *  If output filename was not specified, then create it
   */
  if(!binfile[0]) {
    strlcpy(binfile, datfile, FILENAME_MAX); cptr=strrchr(binfile, '.');
    if(cptr!=NULL) *cptr=(char)0;
    strcat(binfile, ".bin");
  }
  
  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("datfile := %s\n", datfile);
    printf("binfile := %s\n", binfile);
  }

  /*
   *  Open ASCII file
   */
  if(verbose>0) fprintf(stdout, "opening ASCII file %s\n", datfile);
  // Open in binary mode, otherwise ftell() will not function consistently
  // in all platforms/compilers.
  if((fp1=fopen(datfile, "rb"))==NULL) {
    fprintf(stderr, "Error: cannot open %s\n", datfile);
    return(2);
  }
  
  /*
   *  Consume all comment lines from the beginning
   */
  if(verbose>1) printf("checking for comment lines\n");
  long int pos;
  vi=0;
  do {
    isComment=0; vi++;
    pos=ftell(fp1); //fgetpos(fp1, &pos);
    if(fgets(buf, 1023, fp1)==NULL) break;
    if(buf[0]=='\n') isComment=1;
    if(buf[0]=='\r') isComment=1;
    if(buf[0]=='#') isComment=1;
    if(buf[0]=='/' && buf[1]=='*') isComment=1;
    if(buf[0]=='/' && buf[1]=='/') isComment=1;
    if(verbose>1 && isComment) printf("ignoring line %d\n", vi);
  } while(isComment);
  fseek(fp1, pos, SEEK_SET); //fsetpos(fp1, &pos);

  /*
   *  Read the values in ASCII file and write them as binary floats
   */
  if(verbose>1) printf("reading data\n");
  vi=0;
  while(fscanf(fp1, "\t%f", &f)==1 || fscanf(fp1, " %f", &f)==1 ||
        fscanf(fp1, ",%f",  &f)==1 || fscanf(fp1, ";%f", &f)==1) {
    if(verbose>10) printf("vi=%d : %g\n", vi+1, f);
    /* Open output binary file, if this is the first value */
    if(vi==0) {
      if(verbose>0) fprintf(stdout, "opening binary file %s\n", binfile);
      fp2=fopen(binfile, "wb"); if(fp2==NULL) {
        fprintf(stderr, "Error: cannot open %s for write\n", binfile);
        fclose(fp1); return(11);
      }
    }
    /* Write */
    if(fwrite(&f, sizeof(float), 1, fp2) != 1) {
      fprintf(stderr, "Error: cannot write in %s\n", binfile);
      fclose(fp1); fclose(fp2); return(12);
    }
    vi++;
  }
  fclose(fp1); fclose(fp2);
  if(vi==0) {
    fprintf(stderr, "Error: no values could be read from %s\n", datfile);
    return(4);
  }
  if(verbose>=0) 
    fprintf(stdout, "%d float(s) were written in %s\n", vi, binfile);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
