/** @file convend.c
 *  @brief Conversion of byte order of numerical values stored in flat binary
 *  file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add tests.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "libtpcmisc.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Byte order conversion of numerical values stored in flat binary files,",
  "between big endian (Sun Sparc, Motorola, PowerPC) and little endian",
  "(PC/Intel) computers.",
  " ",
  "Usage: @P [Options] basetype flatfile outputfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Accepted base types are: char, short, int, long, float, and double.",
  " ",
  "See also: bigend, img2flat, flat2img, anabyteo",
  " ",
  "Keywords: image, byte order, big endian, little endian",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int       ai, help=0, version=0, verbose=1;
  int       size=0, wn;
  char      infile[FILENAME_MAX], outfile[FILENAME_MAX];
  char     *cptr, *base_type=NULL;
  long int  n;
  FILE     *fp, *fp2;
  unsigned short s[2];
  unsigned long l[2];
  double    d[2];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  infile[0]=outfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') { /* options */
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(cptr==NULL) continue;
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(size<=0) {
      base_type=argv[ai];
      if(!strcasecmp(base_type, "int")) size=sizeof(int);
      else if(!strcasecmp(base_type, "short")) size=sizeof(short int);
      else if(!strcasecmp(base_type, "int")) size=sizeof(int);
      else if(!strcasecmp(base_type, "long")) size=sizeof(long int);
      else if(!strcasecmp(base_type, "float")) size=sizeof(float);
      else if(!strcasecmp(base_type, "double")) size=sizeof(double);
      else if(!strcasecmp(base_type, "char")) {
        printf("No conversion is necessary for char type.\n"); return(0);
      } else {
        fprintf(stderr, "Error: illegal datatype %s.\n", base_type);
        return(1);
      }
      continue;
    } else if(!infile[0]) {
      strlcpy(infile, argv[ai], FILENAME_MAX); continue;
    } else if(!outfile[0]) {
      strlcpy(outfile, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }

  /* Is something missing? */
  if(!outfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(strcasecmp(infile, outfile)==0) {
    fprintf(stderr, "Error: same name for input and output file.\n");
    return(1);
  }


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("infile := %s\n", infile);
    printf("outfile := %s\n", outfile);
    printf("size := %d\n", size);
    if(verbose>2) 
      printf("double=%d, float=%d, long=%d, int=%d, short=%d, char=%d\n",
            (int)sizeof(double), (int)sizeof(float), (int)sizeof(long),
            (int)sizeof(int), (int)sizeof(short int), (int)sizeof(char));
  }

  /* Open datafile for read */
  if((fp=fopen(infile, "rb"))==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", infile); return(2);
  }
  /* Open outputfile for write */
  if((fp2=fopen(outfile, "wb"))==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", outfile);
    fclose(fp); return(11);
  }
  /* Read, convert and write one value at a time */
  n=0; wn=1;
  switch(size) {
    case 2:
      while(!feof(fp)) {
        if(fread((char*)s, size, 1, fp)!=1) break;
        swap((char*)s, (char*)(s+1), size);
        wn=fwrite((char*)(s+1), size, 1, fp2); if(wn==0) break;
        n++;
      }
      break;
    case 4:
      while(!feof(fp)) {
        if(fread((char*)l, size, 1, fp)!=1) break;
        swap((char*)l, (char*)(l+1), size);
        wn=fwrite((char*)(l+1), size, 1, fp2); if(wn==0) break; 
        n++;
      }
      break;
    case 8:
      while(!feof(fp)) {
        if(fread((char*)d, size, 1, fp)!=1) break;
        swap((char*)d, (char*)(d+1), size);
        wn=fwrite((char*)(d+1), size, 1, fp2); if(wn==0) break; 
        n++;
      }
      break;
  }
  fclose(fp); fclose(fp2);
  if(wn==0) {
    fprintf(stderr, "Error: cannot write data in %s\n", outfile);
    return(12);
  }
  if(verbose>0) fprintf(stdout, " %ld %ss converted\n", n, base_type);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
