/** @file absstime.c
    @brief Correct the sampling times in on-line blood sampler data file.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "tpcabss.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Correct the sample times in Allogg, Scanditronics, or GEMS ABSS on-line",
  "blood sampler data file by giving the correct sampling start time in format",
  "hh:mm:ss. If the new sampling start time is not given, program lists",
  "information on the current sample times on the screen.",
  " ",
  "Use this, if sampling was accidentally started later than the PET scan.",
  "In old Allogg files the sampler start time is given before the sampler",
  "data lines in format hhmmss. In new Allogg files individual sample time is",
  "stored for each sampling frame.",
  "In Scanditronics data files the correct sampling start time is in the file",
  "headers; however, in old Scanditronics data files the headers do not contain",
  "time, but it is read from the first sample data line.",
  "In GEMS data files the time in file headers is not the sampling start time,",
  "but time is read from the first sample data line.",
  "If new sampling time is given and it is different than the current sampling",
  "start time, then all sample times in the sampler data file are changed.",
  "Decay correction is not changed.",
  " ",
  "Notice that the original data file is overwritten by default.",
  " ",
  "Usage: @P [options] abssfile starttime",
  " ",
  "Options:",
  " -o=<filename>",
  "     Name for corrected file; by default the original file is overwritten.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example: List the start times from ABSS data and ECAT image, and",
  "set the ABSS start time based on the PET scanner start time.",
  "     @P us1328.bld",
  "     egetstrt us1328dy1.v",
  "     @P us1328.bld 13:07:45",
  " ",
  "See also: absszero, abssbkg, absscal, abssfch, abssexam, injdifft",
  " ",
  "Keywords: input, ABSS, blood, time, physical decay, calibration",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int     ai, help=0, version=0, verbose=1;
  char    abssfile[FILENAME_MAX], outfile[FILENAME_MAX];
  char   *cptr, startTime[256];
  int     ret;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  abssfile[0]=outfile[0]=startTime[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "O=", 2)==0 && strlen(cptr)>2) {
      strlcpy(outfile, cptr+2, FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(abssfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    if(strTimeValid(argv[ai])==0) {
      strlcpy(startTime, argv[ai], 9);
    } else {
      fprintf(stderr, "Error: invalid time '%s'.\n", argv[ai]);
      return(1);
    }
    ai++;
  }
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Did we get all the information that we need? */
  if(!abssfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }
  if(!outfile[0]) strcpy(outfile, abssfile);


  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("abssfile := %s\n", abssfile);
    printf("startTime := %s\n", startTime);
    printf("outfile := %s\n", outfile);
    fflush(stdout);
  }

  /*
   *  Read ABSS file
   */
  if(verbose>1) printf("reading %s\n", abssfile);
  TAC abss; tacInit(&abss);
  ret=tacRead(&abss, abssfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&abss); return(2);
  }
  if(verbose>1) {
    printf("sampleNr := %d\n", abss.sampleNr);
  }
  if(verbose>2) {
    printf("tacNr := %d\n", abss.tacNr);
  }
  if(verbose>1 || !startTime[0]) {
    printf("fileformat := %s\n", tacFormattxt(abss.format));
  }
  if(abss.format!=TAC_FORMAT_ABSS_ALLOGG && 
     abss.format!=TAC_FORMAT_ABSS_ALLOGG_OLD &&
     abss.format!=TAC_FORMAT_ABSS_GEMS &&
     abss.format!=TAC_FORMAT_ABSS_SCANDITRONICS)
  {
    fprintf(stderr, "Error: not valid ABSS format.\n");
    tacFree(&abss); return(2);
  }
  if(verbose>100) abssWrite(&abss, stdout, NULL);
  if(abss.tacNr<1 || abss.sampleNr<1) {
    fprintf(stderr, "Error: not valid ABSS format.\n");
    tacFree(&abss); return(2);
  }

  /* Print the sampler and sampling start times */
  char samplerStartTime[128], sampleStartTime[128], zeroTime[128];
  int ii;
  ii=iftFindKey(&abss.h, "sampler_start_time", 0);
  if(ii>=0) {
    strlcpy(samplerStartTime, abss.h.item[ii].value, 32);
    if(verbose>1)
      fprintf(stdout, "sampler_start_time := %s\n", samplerStartTime);
  } else {
    strcpy(samplerStartTime, "");
  }
  if(tacGetHeaderScanstarttime(&abss.h, sampleStartTime, NULL)!=TPCERROR_OK) {
    strcpy(sampleStartTime, "");
  }
  if(strDateTimeValid(sampleStartTime, NULL)!=0) {
    fprintf(stderr, "Error: invalid start time in %s\n", abssfile);
    tacFree(&abss); return(3);
  }
  strcpy(zeroTime, sampleStartTime);
  fprintf(stdout, "zero_time := %8.8s\n", zeroTime+11);
  if(fabs(abss.x1[0])>0.5) {
    /* we must compute the time at first sample start */
    struct tm starttm;
    if(strDateTimeRead(sampleStartTime, &starttm)==0) {
      tmAdd(abss.x1[0], &starttm);
      if(strftime(sampleStartTime, 32, "%Y-%m-%d %H:%M:%S", &starttm)==0)
        strcpy(sampleStartTime, "");        
    }
    if(strDateTimeValid(sampleStartTime, NULL)!=0) {
      fprintf(stderr, "Error: invalid start time in %s\n", abssfile);
      tacFree(&abss); return(3);
    }
  }
  fprintf(stdout, "date := %10.10s\n", sampleStartTime);
  fprintf(stdout, "first_sample_time := %8.8s\n", sampleStartTime+11);
  fflush(stdout);

  /* Print the sampling frame duration */
  fprintf(stdout, "sample_interval[s] := %g\n", abss.x2[0]-abss.x1[0]);
  /* Print the total sampling duration */
  fprintf(stdout, "sampling_time[s] := %g\n", 
          abss.x2[abss.sampleNr-1]-abss.x1[0]);
  fflush(stdout);


  /*
   *  If new start time was not specified, then stop here
   */
  if(!startTime[0]) {tacFree(&abss); return(0);}

  /*
   *  If new zero time is not different to the current one, then stop here
   */
  struct tm newtm, oldtm;
  char newZeroTime[128];
  strlcpy(newZeroTime, zeroTime, 12);
  strcat(newZeroTime, startTime);
  if(verbose>3) printf("newZeroTime := %s\n", newZeroTime);
  if(strDateTimeRead(newZeroTime, &newtm)!=0 || 
     strDateTimeRead(zeroTime, &oldtm)!=0)
  {
    fprintf(stderr, "Error: invalid start times.\n");
    tacFree(&abss); return(5);
  }
  double timeDiff=tmDifference(&newtm, &oldtm);
  if(verbose>1) {
    printf("required_start_time_change[s] := %g\n", timeDiff);
  }
  if(fabs(timeDiff)<1.0) {
    fprintf(stdout, "Note: start times are not different; file not changed.\n");
    tacFree(&abss); return(0);
  }


  /* Change the start time */
  ret=tacSetHeaderScanstarttime(&abss.h, newZeroTime);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: cannot set new start time.\n");
    tacFree(&abss); return(6);
  }

  /* Change also sample frame times for Allogg 1 */
  if(abss.format==TAC_FORMAT_ABSS_ALLOGG_OLD) {
    for(int i=0; i<abss.sampleNr; i++) {
      abss.x[i]-=timeDiff;
      abss.x1[i]-=timeDiff;
      abss.x2[i]-=timeDiff;
    }
  }



  /*
   *  Write the modified file
   */
  if(verbose>1) {
    printf("writing corrected data file in %s\n", outfile); fflush(stdout);}
  FILE *fp;
  fp=fopen(outfile, "w");
  if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file for writing.\n");
    tacFree(&abss); return(11);
  }
  ret=abssWrite(&abss, fp, &status);
  fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&abss); return(12);
  }
  if(verbose>0) {
    printf("corrected data written in %s\n", outfile); 
    fflush(stdout);
  }

  tacFree(&abss);  
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
