#!/bin/bash
#: Title      : test_arlkup
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../arlkup$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi


printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f blood.bld ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f dynamic.fra ] || [ ! -f static.fra ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_perf.tac ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f idi.tac ]; then
  printf "simulating image-derived input\n"
  simframe -sec blood.bld dynamic.fra idi.tac O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


if [ ! -f dynamic.tac ] || [ ! -f static.tac ]; then
  printf "simulating tissue data independently\n"
  b2t_h2o      -voiname=tac1 blood.bld   1 0.95 1 0.0 0.0 radiowater.tac
  b2t_h2o -add -voiname=tac2 blood.bld   5 0.95 1 0.0 0.0 radiowater.tac
  b2t_h2o -add -voiname=tac3 blood.bld  10 0.95 1 0.0 0.0 radiowater.tac
  b2t_h2o -add -voiname=tac4 blood.bld  20 0.95 1 0.0 0.0 radiowater.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec radiowater.tac dynamic.fra dynamic.tac O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe -sec radiowater.tac static.fra  static.tac O-15
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Default settings, reasonable noiseless data. \n"
printf " Test case 2: Calculated table is used to estimate regional perfusion.\n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output.lkup integral.tac output.tac output.ift

$PROGRAM blood.bld 0.95 30 15 250 output.lkup
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output.lkup > output.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift unit "mL/(dL*min)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift timeunit "sec*kBq/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftinteg dynamic.tac 15 250 integral.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclkup integral.tac output.lkup output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.2 correct_perf.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Image-derived input. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output.lkup output.tac

$PROGRAM idi.tac 0.95 30 15 250 output.lkup
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclkup integral.tac output.lkup output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.02 -rel=0.5 correct_perf.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case: Static scan. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output2.lkup integral2.tac output2.tac output2.ift

$PROGRAM -static=y blood.bld 0.95 30 0 180 output2.lkup
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacunit output.lkup > output2.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.ift unit "mL/(dL*min)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.ift timeunit "sec*kBq/mL"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
dftinteg static.tac 0 0 integral2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
taclkup integral2.tac output2.lkup output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=n -abs=0.01 -rel=0.2 correct_perf.tac output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "=====================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption blood.bld 0.8 30 0 180 output.lkup 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM blood.bld 0.8 30 0 180 output.lkup extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.2 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM blood.bld 0.8 30 0 180  1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.dat 0.8 30 0 180 output.lkup 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

