#!/bin/bash
#: Title      : test_bfmh2o
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../bfmh2o$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f blood.dat ] || [ ! -f frames.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_basicsim.res ] || [ ! -f correct_basicsim_ml.res ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_ptf.res ] || [ ! -f correct_vafixed.par ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f basicsim_fpt.tac ]; then
  printf "\n Simulate PET radiowater curves using b2t_h2o assuming that Vb=Va\n"
  printf " Flow per perfusable tissue\n"
  b2t_h2o -fpt      -nosub -voiname=f003v1 blood.dat   3 0.8 1 1 100 orig_fpt.tac
  b2t_h2o -fpt -add -nosub -voiname=f030v3 blood.dat  30 0.7 1 3 100 orig_fpt.tac
  b2t_h2o -fpt -add -nosub -voiname=f300v6 blood.dat 300 0.5 1 6 100 orig_fpt.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe orig_fpt.tac frames.dat basicsim_fpt.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes basicsim_fpt.svg basicsim_fpt.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f basicsim.tac ]; then
  printf "\n Simulate PET radiowater curves using b2t_h2o assuming that Vb=Va\n"
  printf " Flow per PET volume\n"
  b2t_h2o      -nosub -voiname=f003v1 blood.dat   3 0.8 1 1 100 orig.tac
  b2t_h2o -add -nosub -voiname=f030v3 blood.dat  30 0.7 1 3 100 orig.tac
  b2t_h2o -add -nosub -voiname=f300v6 blood.dat 300 0.5 1 6 100 orig.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe orig.tac frames.dat basicsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes basicsim.svg basicsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f nonphys.tac ]; then
  printf "\n Simulate nonphysiological PET radiowater curves\n"
  b2t_h2o -fpt      -nosub -voiname=f000v4 blood.dat   0 0.001 1 80 50 orig2.tac
  b2t_h2o -fpt -add -nosub -voiname=f050v0 blood.dat  50 2.0   1 0   0 orig2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe orig2.tac frames.dat nonphys.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes nonphys.svg nonphys.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f hetersim.tac ]; then
  printf "\n Simulate heterogeneous PET radiowater curves using b2t_h2o assuming that Vb=Va\n"
  printf " Flow per perfusable tissue\n"
  b2t_h2o -fpt -nosub blood.dat  2 0.8 1 5 100 orig_het1.tac
  b2t_h2o -fpt -nosub blood.dat 10 0.8 1 5 100 orig_het2.tac
  b2t_h2o -fpt -nosub blood.dat 50 0.8 1 5 100 orig_het3.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n Mean of TACs with different perfusion\n"
  taccalc orig_het1.tac + orig_het2.tac temp1.tac
  taccalc temp1.tac x 0.5 orig_het4.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc orig_het2.tac + orig_het3.tac temp1.tac
  taccalc temp1.tac x 0.5 orig_het5.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc orig_het1.tac + orig_het3.tac temp1.tac
  taccalc temp1.tac x 0.5 orig_het6.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n Combine TACs \n"
  tacadd -ovr orig_het.tac orig_het1.tac
  tacadd      orig_het.tac orig_het2.tac
  tacadd      orig_het.tac orig_het3.tac
  tacadd      orig_het.tac orig_het4.tac
  tacadd      orig_het.tac orig_het5.tac
  tacadd      orig_het.tac orig_het6.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren orig_het.tac 1 f002_v5
  tacren orig_het.tac 2 f010_v5
  tacren orig_het.tac 3 f050_v5
  tacren orig_het.tac 4 f006_v5_12
  tacren orig_het.tac 5 f030_v5_23
  tacren orig_het.tac 6 f026_v5_13
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n Simulate PET frames \n"
  simframe orig_het.tac frames.dat hetersim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes hetersim.svg hetersim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f ptfsim.tac ]; then
  printf "\n Simulate low perfusable tissue fraction\n"
  b2t_h2o -fpt      -nosub -voiname=f003v5 blood.dat   3 0.8 1 5 100 orig_ptf.tac
  b2t_h2o -fpt -add -nosub -voiname=f030v5 blood.dat  30 0.8 1 5 100 orig_ptf.tac
  b2t_h2o -fpt -add -nosub -voiname=f300v5 blood.dat 300 0.8 1 5 100 orig_ptf.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  taccalc orig_ptf.tac x 0.50 temp.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  cp -f temp.tac orig_ptf.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n Simulate PET frames \n"
  simframe orig_ptf.tac frames.dat ptfsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=yes ptfsim.svg ptfsim.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n 1.0.0 \n"
printf " Test case: Default settings, reasonable noiseless data. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output100.par output100.svg

$PROGRAM -d2 -svg=output100.svg blood.dat basicsim.tac 240 output100.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_basicsim.res output100.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output100.par fit_end 4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output100.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: Same but with option -fpt. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output101.par

$PROGRAM  -fpt blood.dat basicsim_fpt.tac 240 output101.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_basicsim.res output101.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output101.par fit_end 4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.3 \n"
printf " Test case: Reasonable limits for k2. \n"
printf " Expected result: Correct results calculated. \n\n"

rm -f output103.par

$PROGRAM -k2min=0.03 -k2max=7.5 blood.dat basicsim.tac 240 output103.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_basicsim.res output103.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.1.0 \n"
printf " Test case: Non-physiological data. \n"
printf " Expected result: Results can be whatever, but no error. \n\n"

rm -f output110.par output110.svg

$PROGRAM -svg=output110.svg blood.dat nonphys.tac 120 output110.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output110.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
if [ ! -f output110.svg ]; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output110.par fit_end 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "=====================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: PTF is low and perfusion calculated from k2. \n"
printf " Test case 2: Fit time can be limited. \n"
printf " Expected result: Correct flow results calculated. \n\n"

rm -f output200.par output200.svg

$PROGRAM -pH2O=0.8 -fpt -svg=output200.svg blood.dat ptfsim.tac 120 output200.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_ptf.res output200.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output200.par fit_end 2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output200.par pH2O 0.8
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Fit time is set to zero. \n"
printf " Expected result: All data is used. \n\n"

rm -f output210.par output210.svg

$PROGRAM -svg=output210.svg blood.dat basicsim.tac 0 output210.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_basicsim.res output210.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output210.par fit_end 6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.1.1 \n"
printf " Test case: Fit time is set to very large value. \n"
printf " Expected result: All data is used. \n\n"

rm -f output211.par

$PROGRAM blood.dat basicsim.tac 9999 output211.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 output210.par output211.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output211.par fit_end 6
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n 2.2.0 \n"
printf " Test case: Option -dL. \n"
printf " Expected result: Correct results per dL. \n\n"

rm -f output220.par

$PROGRAM -dL blood.dat basicsim.tac 240 output220.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_basicsim.res output220.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.2.1 \n"
printf " Test case: Option -mL. \n"
printf " Expected result: Correct results per mL. \n\n"

rm -f output221.par

$PROGRAM -mL blood.dat basicsim.tac 240 output221.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-3 -abs=0.01 -rel=1 correct_basicsim_ml.res output221.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 2.3.0 \n"
printf " Test case: Option -Va. \n"
printf " Expected result: Va is fixed to the given value. \n\n"

rm -f output230.par

$PROGRAM -Va=4 blood.dat basicsim.tac 0 output230.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.001 output230.par Vb 4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=3 -abs=0.01 -rel=1 correct_vafixed.par output230.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-2 -abs=0.1 -rel=2 correct_basicsim.par output230.par
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "=====================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption blood.dat basicsim.tac 0 output300.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM blood.dat basicsim.tac 0 output301.par extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.2 \n"
printf " Test case: Missing argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM blood.dat basicsim.tac 0 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "missing command-line argument; use option --help"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Input file does not exist. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexistingfile.dat basicsim.tac 0 output310.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM blood.dat nonexistingfile.tac 0 output310.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "cannot open file"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.1 \n"
printf " Test case: Wrong file format. \n"
printf " Expected result: Error. \n\n"

$PROGRAM output100.par basicsim.tac 0 output311.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM blood.dat output100.par 0 output311.par 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

