/** @file dcmdict.c
 *  @brief List the DICOM dictionary that is used by TPCCLIB.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the DICOM dictionary that is used by TPCCLIB.",
  "The data is printed in TSV format.",
  " ",
  "Usage: @P [options] [> dcmdict.tsv]",
  " ",
  "Options:",
  " -uid",
  "     Instead of dictionary, list the transfer syntax UIDs.",
  " -sop",
  "     Instead of dictionary, list the SOP UIDs.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: dcmlhdr, lmhdr, ana_lhdr",
  " ",
  "Keywords: DICOM, header, UID, dictionary",
  0};
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char *cptr;
  int   listWhat=0; // 0=dictionary, 1=transfer UIDs, 2=SOP UIDs

  /*
   *  Get arguments
   */
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "UID")==0) {
      listWhat=1; continue;
    } else if(strcasecmp(cptr, "SOP")==0) {
      listWhat=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  /* Non-options */
  for(; ai<argc; ai++) if(*argv[ai]!='-') {
    fprintf(stderr, "Error: invalid argument '%s'\n", argv[ai]);
    return(1);
  }
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}


  /* Print DICOM transfer syntax UIDs, if requested */
  if(listWhat==1) {
    unsigned int i=1;
    while(i<DCM_TRUID_INVALID) {
      printf("%s\t%s\n", dcmTrUIDString(i), dcmTrUIDDescr(i));
      i++;
    }
    fflush(stdout);
    return(0);
  }

  /* Print DICOM SOP UIDs, if requested */
  if(listWhat==2) {
    char *uid;
    unsigned int i=1;
    while(1) {
      uid=dcmSOPUID(i); if(strcasecmp(uid, "unknown")==0) break;
      printf("%s\t%s\n", uid, dcmSOPName(i));
      i++;
    }
    fflush(stdout);
    return(0);
  }

  /* Print DICOM dictionary contents */
  unsigned int n=dcmDictSize();
  if(verbose>1) {printf("Dictionary size := %u\n", n); fflush(stdout);}
  unsigned short int group, element;
  printf("Group\tElement\tVR\tDescription\n");
  for(unsigned int i=0; i<n; i++) {
    if(dcmDictIndexTag(i, &group, &element)) continue;
    printf("%04x\t%04x\t%s\t%s", group, element, dcmDictIndexVR(i), dcmDictIndexDescr(i));
    if(verbose>1) {
      unsigned int usi;
      char buf[4];
      memcpy(buf+2, &group, 2); memcpy(&buf, &element, 2); memcpy(&usi, buf, 4);
      printf("\t%u", usi);
    }
    printf("\n");
  }
  fflush(stdout);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
