/** @file dcmmlist.c
 *  @brief List DICOM PET image matrices.
 *  Reference: DICOM PS3.3 2017a C.8.9.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpctac.h"
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "List the matrices of a PET image in DICOM format.",
  " ",
  "NOT for production use!",
  " ",
  "Usage: @P [-Options] dicomfile",
  " ",
  "Options:",
  " -plane=<Y|n>",
  "     List (y, default) or do not list (n) the plane (slice) number.",
  " -frame=<Y|n>",
  "     List (y, default) or do not list (n) the frame number.",
  " -framestart=<Y|n>",
  "     List (y, default) or do not list (n) the frame start time.",
  " -framedur=<Y|n>",
  "     List (y, default) or do not list (n) the frame duration.",
  " -acqdate=<y|N>",
  "     List (y) or do not list (n, default) the acquisition date.",
  " -acqtime=<Y|n>",
  "     List (y, default) or do not list (n) the acquisition time.",
  " -filename=<Y|n>",
  "     List (y, default) or do not list (n) the file name.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: dcmlhdr, dcmframe",
  " ",
  "Keywords: image, DICOM",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  char  dcmfile[FILENAME_MAX];
  int   ret;
  int   listPlane=1; // 0=no, 1=yes
  int   listFrame=1; // 0=no, 1=yes
  int   listFrameStart=1; // 0=no, 1=yes
  int   listFrameDur=1; // 0=no, 1=yes
  int   listAcqDate=0; // 0=no, 1=yes
  int   listAcqTime=1; // 0=no, 1=yes
  int   listFilename=1; // 0=no, 1=yes


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  dcmfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr; cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "PLANE=", 6)==0) {
      if((listPlane=tpcYesNo(cptr+6))>=0) continue;
    } else if(strncasecmp(cptr, "FRAME=", 6)==0) {
      if((listFrame=tpcYesNo(cptr+6))>=0) continue;
    } else if(strncasecmp(cptr, "FRAMESTART=", 11)==0) {
      if((listFrameStart=tpcYesNo(cptr+11))>=0) continue;
    } else if(strncasecmp(cptr, "FRAMEDUR=", 9)==0) {
      if((listFrameDur=tpcYesNo(cptr+9))>=0) continue;
    } else if(strncasecmp(cptr, "ACQDATE=", 8)==0) {
      if((listAcqDate=tpcYesNo(cptr+8))>=0) continue;
    } else if(strncasecmp(cptr, "ACQTIME=", 8)==0) {
      if((listAcqTime=tpcYesNo(cptr+8))>=0) continue;
    } else if(strncasecmp(cptr, "FILENAME=", 9)==0) {
      if((listFilename=tpcYesNo(cptr+9))>=0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(dcmfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }
  /* Did we get all the information that we need? */
  if(!dcmfile[0]) {
    fprintf(stderr, "Error: missing command-line argument; use option --help\n");
    return(1);
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("dcmfile := %s\n", dcmfile);
    printf("listPlane := %d\n", listPlane);
    printf("listFrame := %d\n", listFrame);
    printf("listFrameStart := %d\n", listFrameStart);
    printf("listFrameDur := %d\n", listFrameDur);
    printf("listAcqDate := %d\n", listAcqDate);
    printf("listAcqTime := %d\n", listAcqTime);
    printf("listFilename := %d\n", listFilename);
  }

  /*
   *  Get list of DICOM files belonging to the same image
   */
  IFT fl; iftInit(&fl);
  ret=dcmFileList(dcmfile, &fl, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    return(2);
  }
  if(verbose>1) {printf("fileNr := %d\n", fl.keyNr); fflush(stdout);}
  if(verbose>6) iftWrite(&fl, stdout, NULL);

  /*
   *  Read matrix list
   */
  DCMML ml; dcmmlInit(&ml);
  ret=dcmMListRead(&fl, &ml, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&fl); dcmmlFree(&ml); return(3);
  }
  /* file list is not needed later */
  iftFree(&fl);

  /*
   *  Sort the list
   */
  dcmmlSortByPlane(&ml, NULL);

  /*
   *  Print matrix list
   */
  int col=0;
  if(listPlane) {if(col>0) {printf("\t");} printf("Plane"); col++;}
  if(listFrame) {if(col>0) {printf("\t");} printf("Frame"); col++;}
  if(listFrameStart) {if(col>0) {printf("\t");} printf("Start"); col++;}
  if(listFrameDur) {if(col>0) {printf("\t");} printf("Dur"); col++;}
  if(listAcqDate) {if(col>0) {printf("\t");} printf("AcqDate"); col++;}
  if(listAcqTime) {if(col>0) {printf("\t");} printf("AcqTime"); col++;}
  if(listFilename) {if(col>0) {printf("\t");} printf("File"); col++;}
  if(col==0) {
    fprintf(stderr, "Error: all prints turned off by user.\n");
    dcmmlFree(&ml); return(1);
  }
  printf("\n");
  for(unsigned int i=0; i<ml.nr; i++) {
    col=0;
    if(listPlane) {if(col>0) {printf("\t");} printf("%u", ml.m[i].plane); col++;}
    if(listFrame) {if(col>0) {printf("\t");} printf("%u", ml.m[i].frame); col++;}
    if(listFrameStart) {if(col>0) {printf("\t");} printf("%g", ml.m[i].frameStart); col++;}
    if(listFrameDur) {if(col>0) {printf("\t");} printf("%g", ml.m[i].frameDur); col++;}
    if(listAcqDate) {if(col>0) {printf("\t");} printf("%s", ml.m[i].acqDate); col++;}
    if(listAcqTime) {if(col>0) {printf("\t");} printf("%s", ml.m[i].acqTime); col++;}
    if(listFilename) {if(col>0) {printf("\t");} printf("%s", ml.m[i].filename); col++;}
    printf("\n");
  }



  dcmmlFree(&ml);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
