/** @file dcftime.c
 *  @brief Calculate decay correction factor based on times or dates and times.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcisotope.h"
/*****************************************************************************/

/*****************************************************************************/
/* Local functions */

/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Calculates the decay correction factor between specified times, optionally",
  "with dates.",
  " ",
  "Usage: @P [options] Isotope time0 time1 [> outputfile]",
  " ",
  "Options:",
  " -title=<Y|n>",
  "     Print title line (y, default), or do not print title (n).",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Examples:",
  "       @P Zr-89 2018-03-21 15:22:00 2018-03-27 15:46:00",
  "       @P C-11 11:08:13 12:16:52",
  " ",
  "See also: halflife, tacdecay, imgdecay, tactime, ecattime",
  " ",
  "Keywords: physical decay, decay correction, halflife, isotope",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int isotope=ISOTOPE_UNKNOWN;
  int printTitle=1;
  char datestr0[128], timestr0[128], datestr1[128], timestr1[128];


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  datestr0[0]=timestr0[0]=datestr1[0]=timestr1[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "TITLE=", 6)==0) {
      printTitle=tpcYesNo(cptr+6);
      if(printTitle>=0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; 

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) {
    isotope=isotopeIdentify(argv[ai]);
    if(isotope==ISOTOPE_UNKNOWN) {
      fprintf(stderr, "Error: invalid isotope '%s'\n", argv[ai]); return(1);}
    ai++;
  }
  if(ai<argc) {
    if(!strDateValid(argv[ai])) strlcpy(datestr0, argv[ai++], 128);
    else if(!strTimeValid(argv[ai])) strlcpy(timestr0, argv[ai++], 128);
    else {fprintf(stderr, "Error: invalid date/time '%s'\n", argv[ai]); return(1);}
  }
  if(ai<argc) {
    if(datestr0[0]) {
      if(!strTimeValid(argv[ai])) strlcpy(timestr0, argv[ai++], 128);
      else {fprintf(stderr, "Error: invalid time '%s'\n", argv[ai]); return(1);}
    } else {
      if(!strTimeValid(argv[ai])) strlcpy(timestr1, argv[ai++], 128);
      else {fprintf(stderr, "Error: invalid time '%s'\n", argv[ai]); return(1);}
    }
    if(timestr1[0] && ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}
  }
  if(ai<argc) {
    if(!strDateValid(argv[ai])) strlcpy(datestr1, argv[ai++], 128);
    else if(!strTimeValid(argv[ai])) strlcpy(timestr1, argv[ai++], 128);
    else {fprintf(stderr, "Error: invalid date/time '%s'\n", argv[ai]); return(1);}
  }
  if(ai<argc) {
    if(timestr1[0]) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}
    if(!strTimeValid(argv[ai])) strlcpy(timestr1, argv[ai++], 128);
    else {fprintf(stderr, "Error: invalid date/time '%s'\n", argv[ai]); return(1);}
  }
  if(ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}

  /* Check that we got what we need */
  if(isotope==ISOTOPE_UNKNOWN) {fprintf(stderr, "Error: missing isotope code.\n"); return(1);}
  if(!timestr1[0]) {fprintf(stderr, "Error: missing date/time.\n"); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n");
    printf("isotope := %s\n", isotopeName(isotope));
    printf("time0 := %s %s\n", datestr0, timestr0);
    printf("time1 := %s %s\n", datestr1, timestr1);
    printf("printTitle := %d\n", printTitle);
  }


  /*
   *  Calculate the difference in given times
   */
  char datetimestr0[128], datetimestr1[128];
  if(datestr0[0]) {
    strcpy(datetimestr0, datestr0); datetimestr0[10]=(char)0;
    if(datestr1[0]) {strcpy(datetimestr1, datestr1); datetimestr1[10]=(char)0;}
    else strcpy(datetimestr1, datestr0);
  } else {
    strcpy(datetimestr0, "2018-05-10");
    strcpy(datetimestr1, datetimestr0);
  }
  strcat(datetimestr0, " ");
  strcat(datetimestr1, " ");
  strlcat(datetimestr0, timestr0, 128);
  strlcat(datetimestr1, timestr1, 128);
  if(verbose>2) {
    printf("datetime0 := %s\n", datetimestr0);
    printf("datetime1 := %s\n", datetimestr1);
  }
  double tdiff=strDateTimeDifference(datetimestr1, datetimestr0)/60.0;
  if(verbose>2) {
    printf("time_difference := %g min\n", tdiff);
  }


  /*
   *  Calculate decay correction factor
   */
  double dcf=decayCorrectionFactorFromIsotope(isotope, tdiff, 0.0);
  if(isnan(dcf)) {
    fprintf(stderr, "Error: cannot calculate decay correction factor.\n");
    return(1);
  }


  /*
   *  Print result
   */
  if(printTitle) printf("time[min]\tDCF\n");
  printf("%g\t%g\n", tdiff, dcf);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
