/** @file csv2ift.c
 *  @brief Converts CSV file into IFT file for software testing.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Converts CSV file into IFT format, each CSV field as its own line.",
  " ",
  "Usage: @P [Options] csvfile iftfile",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: iftlist, iftisval, iftmatch",
  " ",
  "Keywords: CSV, IFT, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int   ai, help=0, version=0, verbose=1;
  int   ret;
  char  csvfile[FILENAME_MAX], iftfile[FILENAME_MAX];
  IFT   ift;
  CSV   csv;
  FILE *fp;


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  csvInit(&csv); iftInit(&ift);
  csvfile[0]=iftfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /* Process other arguments, starting from the first non-option */
  for(; ai<argc; ai++) {
    if(!csvfile[0]) {strcpy(csvfile, argv[ai]); continue;}
    if(!iftfile[0]) {strcpy(iftfile, argv[ai]); continue;}
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    return(1);
  }

  /* Check that we got the filenames */
  if(!iftfile[0]) {tpcPrintUsage(argv[0], info, stderr); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("csvfile := %s\n", csvfile);
    printf("iftfile := %s\n", iftfile);
  }

  /*
   *  Read the CSV file
   */
  if(verbose>1) printf("reading %s\n", csvfile);
  fp=fopen(csvfile, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", csvfile);
    return(2);
  }
  ret=csvRead(&csv, fp, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    csvFree(&csv); return(2);
  }
  if(verbose>2) {
    printf("field_nr := %d\n", csv.nr);
  }

  /*
   *  Fill IFT struct
   */
  char key[64];
  ret=TPCERROR_OK;
  for(int i=0; i<csv.nr; i++) {
    sprintf(key, "[%d][%d]", 1+csv.c[i].row, 1+csv.c[i].col);
    ret=iftPut(&ift, key, csv.c[i].content, 0, &status);
    if(ret!=TPCERROR_OK) break;
  }
  csvFree(&csv);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(3);
  }

  /*
   *  Write the IFT contents.
   */
  if(verbose>1) printf("writing modified IFT in %s\n", iftfile);
  fp=fopen(iftfile, "w"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile);
    iftFree(&ift); return(11);
  }
  ret=iftWrite(&ift, fp, &status); fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift); return(12);
  }
  iftFree(&ift);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
