/** @file iftvalc.c
 *  @brief Calculate with value in IFT file.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Simple artithmetic calculations on values in Interfile-type (IFT) files.",
  "Keys are case-insensitive. Only numerical values are accepted.",
  " ",
  "Usage: @P [options] file1 key1 operation file2 key2 [outputfile]",
  " ",
  "Options:",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "The following characters are accepted as operators: +, -, x, and :.",
  "Value(s) in file1 are modified using specified value in file2.",
  "If IFT does not contain keys, or all values in file1 are to be processed,",
  "then enter 'all' or 'none' as the key name.",
  "Contents of file1, including changed and preserved ones, are saved in",
  "outputfile, or if not given , then file1 is overwritten.",
  " ",
  "Example:",
  "     @P par.ift p1 x const.ift factor newpar.ift",
  " ",
  "See also: iftlist, iftisval, iftedit, iftadd, iftdel, parformat, taccalc",
  " ",
  "Keywords: header, IFT, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/** Do specified arithmetic operation between two double values.
    @return Returns the result, or NaN in case of an error.
 */
double doubleOperate(
  /** First value. */
  double v1,
  /** Second value. */
  double v2,
  /** Operation: 1=add, 2=subtract, 3=multiply, 4=divide. */
  int operation
) {
  if(operation==1) return(v1+v2);
  if(operation==2) return(v1-v2);
  if(operation==3) return(v1*v2);
  if(operation==4) return(v1/v2);
  return(nan(""));
}
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret;
  //char *cptr;
  char iftfile1[FILENAME_MAX], iftfile2[FILENAME_MAX], iftfile3[FILENAME_MAX];
  char key1[128], key2[128];
  int operation=0; // 1=+, 2=-, 3=*, 4=:


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  iftfile1[0]=iftfile2[0]=iftfile3[0]=(char)0;
  key1[0]=key2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // later arguments may start with '-'

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose;

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(iftfile1, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(key1, argv[ai++], 128);
  if(ai<argc) {
    if(strcasecmp(argv[ai], "DIV")==0) {
      operation=4;
    } else {
      switch(*argv[ai]) {
        case '+' : operation=1; break;
        case '-' : operation=2; break;
        case '*' : operation=3; break;
        case '.' : operation=3; break;
        case 'X' : operation=3; break;
        case 'x' : operation=3; break;
        case '/' : operation=4; break;
        case ':' : operation=4; break;
      }
    }
    if(operation==0) {
      fprintf(stderr, "Error: invalid operator.\n"); 
      return(1);
    }
    ai++;
  }
  if(ai<argc) strlcpy(iftfile2, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(key2, argv[ai++], 128);
  if(ai<argc) strlcpy(iftfile3, argv[ai++], FILENAME_MAX);
  if(ai<argc) {
    fprintf(stderr, "Error: too many arguments.\n");
    return(1);
  }

  /* Is something missing? */
  if(!key2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}
  /* If key is 'all' then set it to empty string */
  if(strcasecmp(key1, "ALL")==0 || strcasecmp(key1, "NONE")==0) key1[0]=(char)0;
  if(strcasecmp(key2, "ALL")==0 || strcasecmp(key2, "NONE")==0) key2[0]=(char)0;
  /* If output filename not given, then use input file */
  if(!iftfile3[0]) strcpy(iftfile3, iftfile1);

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++) printf("%s ", argv[ai]); 
    printf("\n"); fflush(stdout);
    printf("iftfile1 := %s\n", iftfile1);
    printf("key1 := %s\n", key1);
    printf("operation := %d\n", operation);
    printf("iftfile2 := %s\n", iftfile2);
    printf("key2 := %s\n", key2);
    printf("iftfile3 := %s\n", iftfile3);
  }


  /*
   *  Read IFT files
   */
  IFT ift1, ift2;
  iftInit(&ift1); iftInit(&ift2);
  int key_required; // If key was specified, then only values with keys are read

  if(verbose>1) printf("reading %s\n", iftfile1);
  FILE *fp=fopen(iftfile1, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile1);
    return(2);
  }
  if(key1[0]) key_required=1; else key_required=0;
  ret=iftRead(&ift1, fp, key_required, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift1); return(2);
  }
  if(verbose>2) printf("list size: %d item(s)\n", ift1.keyNr);

  if(verbose>1) printf("reading %s\n", iftfile2);
  fp=fopen(iftfile2, "r"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile2);
    return(3);
  }
  if(key2[0]) key_required=1; else key_required=0;
  ret=iftRead(&ift2, fp, key_required, 1, &status); fclose(fp);
  if(ret) {
    fprintf(stderr, "Error (%d): %s\n", ret, errorMsg(status.error));
    iftFree(&ift1); iftFree(&ift2); return(3);
  }
  if(verbose>2) printf("list size: %d item(s)\n", ift2.keyNr);


  /*
   *  If both lists have the same number of items, and keys were not given,
   *  then try to operate matching items.
   */
  if(ift1.keyNr==ift2.keyNr && !key1[0] && !key2[0]) {
    if(verbose>1) printf("operating items as pairs\n");
    double v1, v2, v3;
    int u1, u2;
    char newstr[128];
    int okNr=0;
    if(iftFindNrOfKeys(&ift1, "")==ift1.keyNr && 
       iftFindNrOfKeys(&ift2, "")==ift2.keyNr)
    {
      if(verbose>1) printf("files do not have key names\n");
      for(int li=0; li<ift1.keyNr; li++) {
        ret=iftGetDoubleWithUnit(&ift1, li, &v1, &u1);
        if(!ret) ret=iftGetDoubleWithUnit(&ift2, li, &v2, &u2);
        if(ret || isnan(v1) || isnan(v2)) {
          if(verbose>1) printf("invalid values in item %d\n", 1+li);
          continue;
        }
        v3=doubleOperate(v1, v2, operation);
        if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
        else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
        if(verbose>3) printf("new_value := '%s'\n", newstr);
        if(iftReplaceValue(&ift1, li, newstr, NULL)!=0) continue;
        okNr++;
      }
    } else {
      if(verbose>1) printf("files have key names, using matching keys\n");
      int lj;
      for(int li=0; li<ift1.keyNr; li++) {
        lj=iftFindKey(&ift2, ift1.item[li].key, 0);
        if(lj<0) continue;
        ret=iftGetDoubleWithUnit(&ift1, li, &v1, &u1);
        if(!ret) ret=iftGetDoubleWithUnit(&ift2, lj, &v2, &u2);
        if(ret || isnan(v1) || isnan(v2)) {
          if(verbose>1) printf("invalid values in item %d\n", 1+li);
          continue;
        }
        v3=doubleOperate(v1, v2, operation);
        if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
        else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
        if(verbose>3) printf("new_value := '%s'\n", newstr);
        if(iftReplaceValue(&ift1, li, newstr, NULL)!=0) continue;
        okNr++;
        /* make sure that the same key from ift2 is not used again */
        iftDelete(&ift2, lj);
      }
    }
    if(okNr<1) {
      fprintf(stderr, "Error: no valid calculations could be done.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(4);
    }
    if(okNr<ift1.keyNr && verbose>0)
      fprintf(stderr, "Notice: some lines omitted.\n");
    /* We're done */
    goto save_and_quit;
  }

  /*
   *  If keys were not given, check if ift2 has only one value to use for all.
   */
  if(!key1[0] && !key2[0]) {
    /* Count how many valid values we can find in file2 */
    int n=0, u, u2=0;
    double v, v2=nan("");
    for(int li=0; li<ift2.keyNr; li++) {
      ret=iftGetDoubleWithUnit(&ift2, li, &v, &u);
      if(ret==0 && !isnan(v)) {v2=v; u2=u; n++;}
    }
    if(verbose>2) printf("file2 contains %d valid values\n", n);
    if(n==1) {
      if(verbose>1) printf("using the single value in file2 for all file1 values\n");
      double v1, v3;
      int u1;
      char newstr[128];
      int okNr=0;
      for(int li=0; li<ift1.keyNr; li++) {
        ret=iftGetDoubleWithUnit(&ift1, li, &v1, &u1);
        if(ret || isnan(v1)) {
          if(verbose>1) printf("invalid values in item %d\n", 1+li);
          continue;
        }
        if(verbose>2 && u1!=u2)
          fprintf(stderr, "Warning: different units\n");
        v3=doubleOperate(v1, v2, operation);
        if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
        else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
        if(verbose>3) printf("new_value := '%s'\n", newstr);
        if(iftReplaceValue(&ift1, li, newstr, NULL)!=0) continue;
        okNr++;
      }
      if(okNr<1) {
        fprintf(stderr, "Error: no valid calculations could be done.\n");
        iftFree(&ift1); iftFree(&ift2);
        return(4);
      }
      if(okNr<ift1.keyNr && verbose>0)
        fprintf(stderr, "Notice: some lines omitted.\n");
      /* We're done */
      goto save_and_quit;
    }
  }

  /*
   *  If key names are given, then check that there is one matching key
   */
  if(key1[0]) {
    int n=iftFindNrOfKeys(&ift1, key1);
    if(n<1) fprintf(stderr, "Error: key '%s' not found in file1.\n", key1);
    if(n>1) fprintf(stderr, "Error: %d items matching key '%s' found in file1.\n", n, key1);
    if(n!=1) {
      iftFree(&ift1); iftFree(&ift2);
      return(5);
    }
  }
  if(key2[0]) {
    int n=iftFindNrOfKeys(&ift2, key2);
    if(n<1) fprintf(stderr, "Error: key '%s' not found in file2.\n", key2);
    if(n>1) fprintf(stderr, "Error: %d items matching key '%s' found in file2.\n", n, key2);
    if(n!=1) {
      iftFree(&ift1); iftFree(&ift2);
      return(6);
    }
  }

  /*
   *  Key given for file2 but not for file1; 
   *  use that one value in file2 for all items in file1.
   */
  if(key2[0] && !key1[0]) {
    if(verbose>1) printf("using the specified item in file2 for all file1 values\n");
    int li, u1, u2;
    double v1, v2, v3;
    li=iftFindKey(&ift2, key2, 0);
    ret=iftGetDoubleWithUnit(&ift2, li, &v2, &u2);
    if(ret || isnan(v2)) {
      fprintf(stderr, "Error: invalid value in file2.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(6);
    }
    char newstr[128];
    int okNr=0;
    for(li=0; li<ift1.keyNr; li++) {
      ret=iftGetDoubleWithUnit(&ift1, li, &v1, &u1);
      if(ret || isnan(v1)) {
        if(verbose>1) printf("invalid values in item %d\n", 1+li);
        continue;
      }
      if(verbose>2 && u1!=u2)
        fprintf(stderr, "Warning: different units\n");
      v3=doubleOperate(v1, v2, operation);
      if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
      else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
      if(verbose>3) printf("new_value := '%s'\n", newstr);
      if(iftReplaceValue(&ift1, li, newstr, NULL)!=0) continue;
      okNr++;
    }
    if(okNr<1) {
      fprintf(stderr, "Error: no valid calculations could be done.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(4);
    }
    if(okNr<ift1.keyNr && verbose>0)
      fprintf(stderr, "Notice: some lines omitted.\n");
    /* We're done */
    goto save_and_quit;
  }


  /*
   *  Key given for both files; 
   *  use those items in the files.
   */
  if(key2[0] && key1[0]) {
    if(verbose>1) printf("using the specified item in file1 and file2\n");
    int li, u1, u2;
    double v1, v2, v3;
    li=iftFindKey(&ift2, key2, 0);
    ret=iftGetDoubleWithUnit(&ift2, li, &v2, &u2);
    if(ret || isnan(v2)) {
      fprintf(stderr, "Error: invalid value in file2.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(6);
    }
    li=iftFindKey(&ift1, key1, 0);
    ret=iftGetDoubleWithUnit(&ift1, li, &v1, &u1);
    if(ret || isnan(v1)) {
      fprintf(stderr, "Error: invalid value in file1.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(5);
    }
    char newstr[128];
    if(verbose>2 && u1!=u2)
      fprintf(stderr, "Warning: different units\n");
    v3=doubleOperate(v1, v2, operation);
    if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
    else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
    if(verbose>3) printf("new_value := '%s'\n", newstr);
    if(iftReplaceValue(&ift1, li, newstr, NULL)!=0) {
      fprintf(stderr, "Error: no valid calculations could be done.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(4);
    }
    /* We're done */
    goto save_and_quit;
  }


  /*
   *  Key given only for file1;
   *  operate that, if only one value is found in file2,
   *  or if same key is found in file2.
   */
  if(key1[0] && !key2[0]) {
    if(verbose>1) printf("using the specified item in file1\n");
    int li, u1;
    double v1;
    li=iftFindKey(&ift1, key1, 0);
    ret=iftGetDoubleWithUnit(&ift1, li, &v1, &u1);
    if(ret || isnan(v1)) {
      fprintf(stderr, "Error: invalid value in file1.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(5);
    }
    /* Count how many valid values we can find in file2 */
    int n=0, u, u2=0, okNr=0;
    double v, v2=nan("");
    for(int lj=0; lj<ift2.keyNr; lj++) {
      ret=iftGetDoubleWithUnit(&ift2, lj, &v, &u);
      if(ret==0 && !isnan(v)) {v2=v; u2=u; n++;}
    }
    if(verbose>2) printf("file2 contains %d valid values\n", n);
    if(n==0) {
      fprintf(stderr, "Error: no valid values in file2.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(6);
    } else if(n==1) {
      if(verbose>1) printf("using the single value in file2\n");
      double v3;
      char newstr[128];
      if(verbose>2 && u1!=u2)
        fprintf(stderr, "Warning: different units\n");
      v3=doubleOperate(v1, v2, operation);
      if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
      else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
      if(verbose>3) printf("new_value := '%s'\n", newstr);
      if(iftReplaceValue(&ift1, li, newstr, NULL)==0)
        okNr++;
    } else {
      if(verbose>1) printf("trying to search same key in file2\n");
      int lj=iftFindKey(&ift2, key1, 0);
      if(lj<0) {
        fprintf(stderr, "Error: cannot find matching item in file2.\n");
        iftFree(&ift1); iftFree(&ift2);
        return(6);
      }
      ret=iftGetDoubleWithUnit(&ift2, lj, &v2, &u2);
      if(ret || isnan(v2)) {
        fprintf(stderr, "Error: invalid value in file2.\n");
        iftFree(&ift1); iftFree(&ift2);
        return(6);
      }
      double v3;
      char newstr[128];
      if(verbose>2 && u1!=u2)
        fprintf(stderr, "Warning: different units\n");
      v3=doubleOperate(v1, v2, operation);
      if(u1==UNIT_UNKNOWN) snprintf(newstr, 128, "%g", v3);
      else snprintf(newstr, 128, "%g %s", v3, unitName(u1));
      if(verbose>3) printf("new_value := '%s'\n", newstr);
      if(iftReplaceValue(&ift1, li, newstr, NULL)==0)
        okNr++;
    }
    if(okNr<1) {
      fprintf(stderr, "Error: no valid calculations could be done.\n");
      iftFree(&ift1); iftFree(&ift2);
      return(4);
    }
    /* We're done */
    goto save_and_quit;
  }

  /*
   *  Other cases not yet working
   */
  fprintf(stderr, "Error: no valid calculations could be done.\n");
  iftFree(&ift1); iftFree(&ift2);
  return(9);


  save_and_quit:
  /*
   *  Write the modified IFT contents.
   */
  if(verbose>1) printf("writing modified IFT in %s\n", iftfile3);
  fp=fopen(iftfile3, "w"); if(fp==NULL) {
    fprintf(stderr, "Error: cannot open file %s\n", iftfile3);
    iftFree(&ift1); iftFree(&ift2); return(11);
  }
  ret=iftWrite(&ift1, fp, &status); fclose(fp);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    iftFree(&ift1); iftFree(&ift2); return(12);
  }

  iftFree(&ift1); iftFree(&ift2);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
