/** @file imgsd.c
 *  @brief Make a map of standard deviation across image frames.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpccsv.h"
#include "tpctac.h"
#include "tpcimage.h"
#include "tpcstatist.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Make a map of standard deviation across image frames.",
  " ",
  "Usage: @P [options] imgfile sdmap",
  " ",
  "Options:",
  " -SNR",
  "     Calculate mean/SD map instead of SD map.",
  " -dif",
  "     Calculate mean-SD map instead of SD map.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: fvar4img, imginteg, imgfiltg, imgdysmo",
  " ",
  "Keywords: image, noise",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int  ai, help=0, version=0, verbose=1;
  int  ret;
  char imgfile[FILENAME_MAX], mapfile[FILENAME_MAX];
  int  mode=0; // 0=SD, 1=Mean/SD, 2=Mean-SD


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  imgfile[0]=mapfile[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strcasecmp(cptr, "SNR")==0) {
      mode=1; continue;
    } else if(strcasecmp(cptr, "SD")==0 || strcasecmp(cptr, "STD")==0) {
      mode=0; continue;
    } else if(strcasecmp(cptr, "DIF")==0) {
      mode=2; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Process other arguments, starting from the first non-option */
  if(ai<argc) strlcpy(imgfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) strlcpy(mapfile, argv[ai++], FILENAME_MAX);
  if(ai<argc) {fprintf(stderr, "Error: too many arguments: '%s'.\n", argv[ai]); return(1);}
  /* Is something missing? */
  if(!mapfile[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("imgfile := %s\n", imgfile);
    printf("mapfile := %s\n", mapfile);
    printf("mode := %d\n", mode);
    fflush(stdout);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;



  /*
   *  Read image data
   */
  if(verbose>1) {printf("reading %s\n", imgfile); fflush(stdout);}
  IMG img; imgInit(&img);
  ret=imgRead(&img, imgfile, &status);
  if(ret!=TPCERROR_OK) { // error
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), imgfile); fflush(stderr);
    return(2);
  }
  if(verbose>2) imgContents(&img, stdout);
  if(img.dimt<2) {
    fprintf(stderr, "Error: method is not applicable to static image.\n"); fflush(stderr);
    return(2);
  }
  if(imgNaNs(&img, 1)>0)
    if(verbose>0) fprintf(stderr, "Warning: missing pixel values.\n");


  /*
   *  Allocate memory for sd map
   */
  if(verbose>1) {printf("preparing SD map\n"); fflush(stdout);}
  IMG map; imgInit(&map);
  if(imgAllocate(&map, img.dimz, img.dimy, img.dimx, 1, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: invalid sample times in %s\n", imgfile); fflush(stderr);
    imgFree(&img); return(3);
  }
  imgCopyHeader(&img, &map);


  /*
   *  Pixel-by-pixel
   */
  long long pxlNr=img.dimx*img.dimy*img.dimz;
  if(verbose>0) {printf("processing %llu image pixels\n", pxlNr); fflush(stdout);}
  for(int zi=0; zi<img.dimz; zi++) {
    if(img.dimz>2 && verbose>0) {fprintf(stdout, "."); fflush(stdout);}
    for(int yi=0; yi<img.dimy; yi++) for(int xi=0; xi<img.dimx; xi++) {
      map.m[zi][yi][xi][0]=0.0;
      float mean, sd;
      if(fstatMeanSD(img.m[zi][yi][xi], img.dimt, &mean, &sd, NULL)!=0) continue;
      if(mode==0) {
        map.m[zi][yi][xi][0]=sd;
      } else if(mode==1) {
        float snr=mean/sd;
        if(isnormal(snr)) map.m[zi][yi][xi][0]=snr;
      } else {
        map.m[zi][yi][xi][0]=mean-sd;
      }
    }
  }
  if(img.dimz>2 && verbose>0) {fprintf(stdout, "\n"); fflush(stdout);}

  /* Dynamic data not needed after this */
  imgFree(&img);


  /* 
   *  Write the map
   */
  if(verbose>1) {printf("writing %s\n", mapfile); fflush(stdout);}
  ret=imgWrite(&map, mapfile, &status);
  if(ret!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error)); fflush(stderr);
    imgFree(&map);
    return(11);
  }
  if(verbose>0) {printf("  %s written.\n", mapfile); fflush(stdout);}

  imgFree(&map);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
