/** @file tacmove.c
    @brief Move TACs in time, but keeping TAC times.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcfileutil.h"
#include "tpcift.h"
#include "tpccsv.h"
#include "tpcisotope.h"
#include "tpctac.h"
#include "tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Move TAC(s) in time with interpolation, keeping the original sample times",
  "or time frames in TAC file, and cause only minimal bias in AUC.",
  " ",
  "Usage: @P [Options] tacfile delay",
  " ",
  "Options:",
  " -opposite",
  "     TACs are moved in opposite direction as specified in parameter file.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Delay time can be given as positive or negative value directly in the command",
  "line, or as dT or deltaT in a parameter file for certain TACs if file",
  "contains more than one TAC.",
  " ",
  "See also: tactime, tacframe, fitdelay, taccut, simframe, tacadd",
  " ",
  "Keywords: TAC, simulation, time delay",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int      ai, help=0, version=0, verbose=1;
  char     tacfile[FILENAME_MAX], parfile[FILENAME_MAX];
  int      opposite=0;
  double   delay=nan("");


  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=parfile[0]=(char)0; 
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "OPPOSITE", 2)==0) {
      opposite=1; continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* The first argument (non-option) is the file name */
  if(ai<argc) {strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(parfile, argv[ai], FILENAME_MAX); ai++;}
  else {fprintf(stderr, "Error: missing file name.\n"); return(1);}
  /* check that there are no extra arguments */
  if(ai<argc) {fprintf(stderr, "Error: extra command-line argument.\n"); return(1);}

  /* Check that TAC file exists */
  if(!fileExist(tacfile)) {
    fprintf(stderr, "Error: file '%s' does not exist.\n", tacfile);
    return(1);
  }
  /* Get delay time or parameter file */
  if(!fileExist(parfile)) {
    if(atofCheck(parfile, &delay)==0) {
      parfile[0]=(char)0;
      if(delay==0.0) {
        fprintf(stderr, "Warning: zero delay time; file not modified.\n");
        return(0);
      }
    } else {
      fprintf(stderr, "Error: invalid argument for delay time: '%s'.\n", parfile);
      return(1);
    }
  }

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    if(parfile[0]) printf("parfile := %s\n", parfile);
    if(isfinite(delay)) printf("delaytime := %g\n", delay);
    fflush(stdout);
  }


  /* 
   *  Read the file
   */
  if(verbose>1) printf("reading %s\n", tacfile);
  TAC tac; tacInit(&tac);
  if(tacRead(&tac, tacfile, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>1) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    if(tac.isframe) printf("frames := yes\n"); else printf("frames := no\n");
    printf("xunit := %s\n", unitName(tac.tunit));
    printf("yunit := %s\n", unitName(tac.cunit));
    fflush(stdout);
  }
  if(tacSortByTime(&tac, &status)!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  /* Fix any gaps and overlap in data if possible and if not then quit with error */
  {
    int ret=tacSetXContiguous(&tac);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(ret));
      tacFree(&tac); return(2);
    }
  }

  /* Get and check TAC data range */
  {
    double xmin, xmax;
    if(tacXRange(&tac, &xmin, &xmax)) {
      fprintf(stderr, "Error: invalid sample range.\n");
      tacFree(&tac); return(2);
    }
    if(verbose>1) printf("x_range: %g - %g\n", xmin, xmax);
    if(!(xmax>xmin)) {fprintf(stderr, "Warning: check the sample times.\n"); fflush(stderr);}
  }

  /*
   *  If the parameter file was given, then read it
   */
  PAR par; parInit(&par);
  int dti=0;
  if(parfile[0]) {
    if(verbose>1) printf("reading %s\n", parfile);
    if(parRead(&par, parfile, &status)!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); return(3);
    }
    if(verbose>2) {
      printf("fileformat := %s\n", parFormattxt(par.format));
      printf("parNr := %d\n", par.parNr);
      printf("tacNr := %d\n", par.tacNr);
    }
    /* Determine which parameter is the delay time */
    if(par.parNr>1) {
      int n=parSelectParameters(&par, "delayT", 1, NULL);
      if(n==0) n=parSelectParameters(&par, "dT", 1, NULL);
      if(n==0) {
        fprintf(stderr, "Error: cannot find delay time in %s\n", parfile);
        tacFree(&tac); parFree(&par); return(3);
      } else if(n>1) {
        fprintf(stderr, "Error: cannot identify delay time in %s\n", parfile);
        tacFree(&tac); parFree(&par); return(3);
      }
      for(int i=0; i<par.parNr; i++) if(par.n[i].sw) {dti=i; break;}
    }
    /* Change time unit if necessary */
    if(tac.tunit==UNIT_MIN && par.n[dti].unit==UNIT_SEC) {
      par.n[dti].unit=UNIT_MIN;
      for(int i=0; i<par.tacNr; i++) par.r[i].p[dti]/=60.0;
    } else if(tac.tunit==UNIT_SEC && par.n[dti].unit==UNIT_MIN) {
      par.n[dti].unit=UNIT_SEC;
      for(int i=0; i<par.tacNr; i++) par.r[i].p[dti]*=60.0;
    }
    /* If only one delay time in parameter, then use it for all TACs */
    if(par.tacNr==1) {
      delay=par.r[0].p[dti];
      if(opposite) delay=-delay;
      if(verbose>1) printf("delaytime := %g\n", delay);
    }
  }


  /*
   *  Change the sample times
   */
  if(isfinite(delay)) {
    /* Common delay time for all TACs */
    if(tacDelay(&tac, delay, -1, &status)!=TPCERROR_OK) {
      fprintf(stderr, "Error: cannot move TACs.\n");
      tacFree(&tac); parFree(&par); return(4);
    }
  } else {
    /* Separate delay time for TACs */
    for(int i=0; i<tac.tacNr; i++) {
      /* Try to find matching TAC in parameters data */
      int pi=-1;
      int n=parSelectTACs(&par, tac.c[i].name, 1, NULL);
      if(n==0 && tac.tacNr==par.parNr) {
        pi=i;
      } else if(n==1) {
        for(int j=0; j<par.tacNr; j++) if(par.r[j].sw) {pi=j; break;}
      } else if(n>1) {
        for(int j=0; j<par.tacNr; j++) if(par.r[j].sw && j==i) {pi=j; break;}
      }
      if(pi<0) {
        fprintf(stderr, "Error: cannot identify delay time for %s\n", tac.c[i].name);
        tacFree(&tac); parFree(&par); return(4);
      }
      delay=par.r[pi].p[dti]; if(opposite) delay=-delay;
      if(verbose>3) printf("  Moving %s for %g\n", tac.c[i].name, delay);
      if(tacDelay(&tac, delay, i, &status)!=TPCERROR_OK) {
        fprintf(stderr, "Error: cannot move TAC.\n");
        tacFree(&tac); parFree(&par); return(4);
      }
    }
  }
  parFree(&par);


  /*
   *  Save the modified data
   */
  if(verbose>1) printf("saving modified data in %s\n", tacfile);
  {
    FILE *fp; fp=fopen(tacfile, "w");
    if(fp==NULL) {
      fprintf(stderr, "Error: cannot open file for writing\n");
      tacFree(&tac); return(11);
    }
    int ret=tacWrite(&tac, fp, TAC_FORMAT_UNKNOWN, 1, &status);
    fclose(fp);
    if(ret!=TPCERROR_OK) {
      fprintf(stderr, "Error: %s\n", errorMsg(status.error));
      tacFree(&tac); return(12);
    }
  }
  tacFree(&tac);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
