#!/bin/bash
#: Title      : test_tactime
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../tactime$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi



printf "=====================================================================\n"
printf "creating test data for  %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n verifying that required data exists \n"

if [ ! -f c11.dft ] || [ ! -f c11_with_isotope.dft ] || [ ! -f c11_with_injtime.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_c11_late.dft ] || [ ! -f correct_c11_late_nogap.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f f18.dft ] || [ ! -f f18_with_isotope.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f o15.dft ] || [ ! -f o15_with_isotope.dft ] || [ ! -f o15blood.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_f18_late.dft ] || [ ! -f correct_o15_late.dft ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_o15blood_corr.kbq ] || [ ! -f correct_o15blood_forw.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi
if [ ! -f correct_o15blood_pre.kbq ] || [ ! -f correct_o15blood_forw_nogap.kbq ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f blood.bld ] || [ ! -f correct_blood.bld ]; then
  printf "time[sec]	BTAC[kBq/cc]\n" > blood.bld
  printf "2.5	0.0101\n" >> blood.bld
  printf "10.06	0.0232\n" >> blood.bld
  printf "16.96	0.0928\n" >> blood.bld
  printf "23.60	87.332\n" >> blood.bld
  printf "30.00	98.176\n" >> blood.bld
  printf "38.12	88.657\n" >> blood.bld
  printf "48.47	67.229\n" >> blood.bld
  printf "53.56	55.450\n" >> blood.bld
  printf "61.03	44.544\n" >> blood.bld
  printf "73.88	37.879\n" >> blood.bld
  printf "82.17	32.476\n" >> blood.bld

  printf "time[sec]	BTAC[kBq/cc]\n" > correct_blood.bld
  printf "0.00	98.176\n" >> correct_blood.bld
  printf "8.12	88.657\n" >> correct_blood.bld
  printf "18.47	67.229\n" >> correct_blood.bld
  printf "23.56	55.450\n" >> correct_blood.bld
  printf "31.03	44.544\n" >> correct_blood.bld
  printf "43.88	37.879\n" >> correct_blood.bld
  printf "52.17	32.476\n" >> correct_blood.bld
fi

if [ ! -f tissue.tac ] || [ ! -f correct_tissue.tac ]; then
  printf "start[minutes]	end[Bq/cc]	ROI001	ROI002	ROI003	ROI004\n" > tissue.tac
  printf "0.000	0.500	0.000e+00	0.000e+00	-1.000e-05	1.000e-04\n" >> tissue.tac
  printf "0.500	1.000	2.352e+01	1.020e+01	9.008e-01	6.123e+00\n" >> tissue.tac
  printf "1.000	2.000	4.222e+01	9.902e+01	3.232e+01	2.221e+01\n" >> tissue.tac
  printf "2.000	3.000	1.013e+03	2.108e+03	8.885e+02	4.238e+02\n" >> tissue.tac
  printf "3.000	4.000	8.671e+04	1.988e+04	3.282e+04	8.631e+03\n" >> tissue.tac
  printf "4.000	5.000	1.272e+05	8.128e+05	3.275e+04	6.631e+05\n" >> tissue.tac
  printf "5.000	10.00	1.123e+05	1.019e+06	8.987e+03	7.363e+05\n" >> tissue.tac
  printf "10.00	15.00	1.001e+06	9.856e+05	8.287e+03	8.822e+05\n" >> tissue.tac
  printf "15.00	20.00	8.302e+05	9.905e+05	6.899e+03	9.124e+05\n" >> tissue.tac
  printf "20.00	30.00	8.487e+05	9.910e+05	7.992e+03	9.846e+05\n" >> tissue.tac
  printf "30.00	45.00	7.923e+05	9.717e+05	8.162e+03	9.940e+05\n" >> tissue.tac

  printf "start[minutes]	end[Bq/cc]	ROI001	ROI002	ROI003	ROI004\n" > correct_tissue.tac
  printf "0.000	5.000	1.123e+05	1.019e+06	8.987e+03	7.363e+05\n" >> correct_tissue.tac
  printf "5.000	10.00	1.001e+06	9.856e+05	8.287e+03	8.822e+05\n" >> correct_tissue.tac
  printf "10.00	15.00	8.302e+05	9.905e+05	6.899e+03	9.124e+05\n" >> correct_tissue.tac
  printf "15.00	25.00	8.487e+05	9.910e+05	7.992e+03	9.846e+05\n" >> correct_tissue.tac
  printf "25.00	40.00	7.923e+05	9.717e+05	8.162e+03	9.940e+05\n" >> correct_tissue.tac
fi

printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case 1: Forward time given as argument. \n"
printf " Test case 2: Decay correction and isotope given with option\n"
printf " Expected result: Correct data saved. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C-11 c11.dft +20.4 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_c11_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -i=F-18 -decay f18.dft 109.8 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_f18_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay -i=O-15 o15.dft 123 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.1 \n"
printf " Test case 1: Forward time given as argument. \n"
printf " Test case 2: Decay correction and isotope inside input file\n"
printf " Expected result: Correct data saved. \n\n"

$PROGRAM -decay c11_with_isotope.dft +20.4 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_c11_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay f18_with_isotope.dft 109.8 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_f18_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay o15_with_isotope.dft 123 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case: Same isotope given inside file and with option. \n"
printf " Expected result: Correct data saved. \n\n"

$PROGRAM -decay -i=C-11 c11_with_isotope.dft +20.4 output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_c11_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.3 \n"
printf " Test case: Different isotope given inside file and with option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -decay -i=F-18 c11_with_isotope.dft +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different isotope in c11_with_isotope.dft"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.4 \n"
printf " Test case: Change times and decay forward and backward. \n"
printf " Expected result: Same as original data saved. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C c11.dft +20.4 output.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -decay -i=C-11 output.dft -20.4
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 c11.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Forward time given as argument. \n"
printf " Test case 2: Decay not changed.\n"
printf " Expected result: Correct data saved. \n\n"

rm -f output.kbq
$PROGRAM o15blood.kbq 60 output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15blood_forw.kbq output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.1.1 \n"
printf " Test case 1: Backward time given as argument. \n"
printf " Test case 2: Decay not changed.\n"
printf " Expected result: Correct data saved. \n\n"

rm -f output2.kbq
$PROGRAM output.kbq -60 output2.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 o15blood.kbq output2.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.1.2 \n"
printf " Test case: Time zero given as argument. \n"
printf " Note: This must work because in batch analysis sometimes delay can be zero.\n"
printf " Expected result: Copy of input data saved. \n\n"

rm -f output.kbq
$PROGRAM o15blood.kbq 0 output.kbq 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.1 o15blood.kbq output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "time change is zero."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.2.0 \n"
printf " Test case 1: Time changes are given in a file with key scan_start_time_difference. \n"
printf " Test case 2: Positive time without time unit.\n"
printf " Expected result: Correct data saved. \n\n"

printf "\nscan_start_time_difference := 20.4\n" > test.dit
rm -f output.dft
$PROGRAM -decay -i=C-11 c11.dft test.dit output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_c11_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.1 \n"
printf " Test case 1: Time changes are given in a file with key time_difference. \n"
printf " Test case 2: Negative time with time unit that is same as in TAC file.\n"
printf " Expected result: Correct data saved. \n\n"

printf "\ntime_difference := -20,4 [min]\n" > test.dit
rm -f output2.dft
$PROGRAM -decay -i=C-11 output.dft test.dit output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 c11.dft output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.2 \n"
printf " Test case 1: Time changes are given in a file with key time_difference. \n"
printf " Test case 2: Negative time with time unit that is different than in TAC file.\n"
printf " Expected result: Correct data saved. \n\n"

printf "\ntime_difference := -1224 s\n" > test.dit
rm -f output2.dft
$PROGRAM -decay -i=C-11 output.dft test.dit output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 c11.dft output2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.3 \n"
printf " Test case 1: Time changes are given in a file with key scan_start_time_difference. \n"
printf " Test case 2: Zero time with time unit that is same as in TAC file.\n"
printf " Expected result: Correct data saved. \n\n"

printf "\nscan_start_time_difference := 0.0 [min]\n" > test.dit
rm -f output.dft
$PROGRAM -decay c11_with_isotope.dft test.dit output.dft 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 c11.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Warning" "time change is zero."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.4 \n"
printf " Test case 1: Time changes are given in a file with key start_time. \n"
printf " Test case 2: Positive time without time unit.\n"
printf " Expected result: Correct data saved. \n\n"

printf "\nstart_time := 20.4\n" > test.dit
rm -f temp.dft temp2.dft
$PROGRAM c11.dft 20.4 temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM temp.dft test.dit temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 c11.dft temp2.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.5 \n"
printf " Test case 1: Time changes are given in a file with key Ta. \n"
printf " Test case 2: Positive time without time unit.\n"
printf " Test case 3: Output file name not given.\n"
printf " Expected result: Correct data saved overwriting input file. \n\n"

printf "\nTa := 20.4\n" > test.dit
$PROGRAM temp.dft test.dit
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 c11.dft temp.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"

printf "\n 2.0.0 \n"
printf " Test case 1: Option -nogap prevents initial gap. \n"
printf " Test case 2: File has frame start and end times.\n"
printf " Expected result: Correct data saved. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C-11 -nogap c11.dft 20,4 output.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_c11_late_nogap.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.1 \n"
printf " Test case 1: Option -nogap prevents initial gap. \n"
printf " Test case 2: File has frame middle times.\n"
printf " Expected result: Correct data saved. \n\n"

rm -f output.kbq
$PROGRAM -nogap o15blood.kbq 60 output.kbq 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15blood_forw_nogap.kbq output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.0.3 \n"
printf " Test case: Same but with PMOD formats. \n"
printf " Expected result: Correct data saved. \n\n"

tacformat -f=PMOD c11.dft c11.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacformat -f=PMOD o15blood.kbq o15blood.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM -decay -i=C-11 -nogap c11.tac 20.4 output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_c11_late_nogap.dft output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -nogap o15blood.bld 60 output.bld 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15blood_forw_nogap.kbq output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case 1: Option -keeptimes. \n"
printf " Test case 2: Cannot be used without decay correction because nothing would be changed. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -keeptimes -i=C-11 c11.dft 20.4 output.dft 1>stdout.txt 2>&1 
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "change in neither sample times or decay correction were requested."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.0.1 \n"
printf " Test case 1: Option -keeptimes with decay correction. \n"
printf " Test case 2: File contains injection_time and isotope. \n"
printf " Expected result 1: Correct data saved. \n"
printf " Expected result 2: injection_time changed accordingly. \n\n"

rm -f output.dft
iftlist c11_with_injtime.dft
$PROGRAM -keeptimes -decay c11_with_injtime.dft +20.4 output.dft 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=no -y=yes -abs=0.05 -rel=1 correct_c11_late.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=yes -y=no -abs=0.05 -rel=1 c11.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.dft injection_time "2005-12-21 11:34:31"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n Test case: Same thing to the other direction. \n\n"

$PROGRAM -keeptimes -decay output.dft -20.4 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -x=yes -y=yes -abs=0.05 -rel=1 c11.dft output.dft
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.dft injection_time "2005-12-21 11:54:55"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 3.1.0 \n"
printf " Test case 1: Option -keepnegat without decay correction. \n"
printf " Test case 2: Change times 60 s backward, then again 60 s forward. \n"
printf " Expected result: Original data is restored. \n\n"

rm -f output.kbq output2.kbq
$PROGRAM -keepnegat o15blood.kbq -60 output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM -keepnegat output.kbq   +60 output2.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 output.kbq output2.kbq
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 o15blood.kbq output2.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 3.1.1 \n"
printf " Test case 1: Without option -keepnegat. \n"
printf " Test case 2: Change times 60 s backward, then again 60 s forward. \n"
printf " Expected result: The first 60 s is lost. \n\n"

rm -f output.kbq output2.kbq
$PROGRAM o15blood.kbq -60 output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15blood_pre.kbq output.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM output.kbq   +60 output2.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.05 -rel=1 correct_o15blood_corr.kbq output2.kbq
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 4.0.0 \n"
printf " Test case 1: Move to start at the peak. \n"
printf " Test case 2: PMOD BLD file format. \n"
printf " Expected result: Correct data saved. \n\n"

rm -f output.bld
$PROGRAM blood.bld peak output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 correct_blood.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.0.1 \n"
printf " Test case 1: Move to start at the peak. \n"
printf " Test case 2: PMOD TAC file format. \n"
printf " Expected result: Correct data saved. \n\n"

rm -f output.tac
$PROGRAM tissue.tac peak output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 correct_tissue.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 4.1.0 \n"
printf " Test case 1: Move to remove gap. \n"
printf " Test case 2: PMOD BLD file format. \n"
printf " Expected result: Correct data saved. \n\n"

$PROGRAM blood.bld -2.5 temp.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.bld
$PROGRAM blood.bld gap output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 temp.bld output.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 4.1.1 \n"
printf " Test case 1: Move to start at the peak. \n"
printf " Test case 2: PMOD TAC file format. \n"
printf " Expected result: Correct data saved. \n\n"

$PROGRAM tissue.tac +3 temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.tac
$PROGRAM temp.tac GAP output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacmatch -abs=0.01 -rel=0.01 tissue.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"

printf "\n 5.0.0 \n"
printf " Test case: Unknown option. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -stupidoption -decay -i=C-11 c11.dft +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.0.1 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C-11 c11.dft +20.4 output.dft extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "extra command-line argument."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.1.0 \n"
printf " Test case: Missing input filename. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C-11 +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "file '+20.4' does not exist."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.1.1 \n"
printf " Test case: Non-existing input filename. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C-11 nonexisting.dft +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "file 'nonexisting.dft' does not exist."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.1.2 \n"
printf " Test case: Wrong format input file. \n"
printf " Expected result: Error. \n\n"

printf "XXXX YYYY zyc\n" > stupid.dat
printf "# comment\n" >> stupid.dat
printf "argh banzai.\n" >> stupid.dat

rm -f output.dft
$PROGRAM -decay -i=C-11 stupid.dat +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt "Error" "invalid file format"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 5.2.0 \n"
printf " Test case: Missing time. \n"
printf " Expected result: Error. \n\n"

cp -f c11.dft test.dft
rm -f output.dft

$PROGRAM -decay -i=C-11 test.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "missing time."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM -decay -i=C-11 test.dft output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftlist stdout.txt
iftisval stdout.txt Error "invalid argument for time: 'output.dft'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.2.1 \n"
printf " Test case: Missing time file. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -decay -i=C-11 c11.dft nonexistingfile.ift output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument for time: 'nonexistingfile.ift'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 5.3.0 \n"
printf " Test case: Invalid isotope code as option. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -decay -i=Hiili c11.dft +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid isotope 'HIILI'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.3.1 \n"
printf " Test case: Invalid isotope code in the file. \n"
printf " Expected result: Error. \n\n"

cp -f c11.dft c11_with_invalid_isotope.dft
printf "\n# isotope := Hiili\n" >> c11_with_invalid_isotope.dft
rm -f output.dft
$PROGRAM -decay c11_with_invalid_isotope.dft +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "isotope not specified."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 5.3.2 \n"
printf " Test case: Isotope not given in file nor as option. \n"
printf " Expected result: Error. \n\n"

rm -f output.dft
$PROGRAM -decay c11.dft +20.4 output.dft 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "isotope not specified."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0
