/** @file abssio.c
    @brief Functions for writing raw ABSS data files.
    @details ABSS raw data read functions are not here but in libtpctac,
     in order to prevent user from accidentally reading raw ABSS data as
     input function. 
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
//#include "tpcextensions.h"
//#include "tpctac.h"
/*****************************************************************************/
#include "tpcabss.h"
/*****************************************************************************/

/*****************************************************************************/
/** Identify ABSS device based on filename.
    @returns the enum ABSS_DEVICE, which is ABSS_UNKNOWN if format
     cannot be identified.
    @sa tacRead, abssWrite
 */
ABSS_DEVICE abssIdFromFName(
  /** ABSS data filename */
  const char *fname
) {
  if(fname==NULL || strnlen(fname, 10)<5) return(ABSS_UNKNOWN);
  /* Get pointer to the extension, starting with the dot */
  char *cptr=filenameGetExtension(fname);
  if(cptr==NULL) return(ABSS_UNKNOWN);
  if(strcasecmp(cptr, ".lis")==0) return(ABSS_SCANDITRONICS);
  if(strcasecmp(cptr, ".bld")==0) return(ABSS_GEMS);
  if(strcasecmp(cptr, ".alg")==0) return(ABSS_ALLOGG_OLD);
  if(strcasecmp(cptr, ".txt")==0) return(ABSS_ALLOGG);
  return(ABSS_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Write ABSS data into file opened for writing. 

    @return enum tpcerror (TPCERROR_OK when successful).
    @sa tacRead, tacInit, tacFree, abssIdFromFName
    @author Vesa Oikonen
 */
int abssWrite(
  /** Pointer to ABSS data, stored in TAC struct, contents of which 
      are to be written */
  TAC *d,
  /** Output file pointer */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(d==NULL || d->sampleNr<1 || d->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  char buf[1024];
  int ret=TPCERROR_OK;

  if(d->format==TAC_FORMAT_ABSS_GEMS) {

    ret=TPCERROR_OK;

    /* Write the header */
    int ii;
    d->h.space_before_eq=0; d->h.space_after_eq=1; /* Set no space before ':' */
    ii=iftFindKey(&d->h, "Protocol", 0);
    if(ii>=0) ret=iftWriteItem(&d->h, ii, fp, status);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    if(tacGetHeaderStudynr(&d->h, buf, NULL)!=TPCERROR_OK) {
      ii=iftFindKey(&d->h, "Patient", 0);
      if(ii>=0) strlcpy(buf, d->h.item[ii].value, 256); else strcpy(buf, "");
    }
    fprintf(fp, "# Patient: %s\n", buf);

    if(tacGetHeaderIsotope(&d->h, buf, NULL)==TPCERROR_OK) {
      double hl=isotopeHalflife(isotopeIdentify(buf));
      if(!isnan(hl)) sprintf(buf, "%f", hl); else strcpy(buf, "");
    } else {
      ii=iftFindKey(&d->h, "Isotope half-life", 0);
      if(ii>=0) strlcpy(buf, d->h.item[ii].value, 256); else strcpy(buf, "");
    }
    fprintf(fp, "# Isotope half-life: %s\n", buf);

    fprintf(fp, "#                 Start   Interv       1st detector pair");
    fprintf(fp, "       2nd detector pair     AUX\n");
    fprintf(fp, "#                  time     time   coinc  singl1  singl2");
    fprintf(fp, "   coinc  singl1  singl2  counts\n");

    ii=iftFindKey(&d->h, "sampler_start_time", 0);
    if(ii>=0) {
      fprintf(fp, "# %s\n", d->h.item[ii].value);
    } else if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      fprintf(fp, "# %s\n", buf);
    }


    /* Write the sample data */
    double sod=0.0; /* Time from start of the day */
    if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      struct tm tmsod;
      if(!strDateTimeRead(buf, &tmsod))
        sod=3600.*tmsod.tm_hour+60.*tmsod.tm_min+tmsod.tm_sec;
    }
    int i, j;
    for(i=0; i<d->sampleNr; i++) {
      fprintf(fp, "%9.1f %9.1f %9.1f", sod+d->x1[i], d->x1[i], d->x2[i]-d->x1[i]);
      for(j=0; j<7; j++) {
        if(j<d->tacNr) fprintf(fp, " %7.0f", d->c[j].y[i]);
        else fprintf(fp, " %7.0f", 0.0);
      }
      if(fprintf(fp, "\n")<1) {ret=TPCERROR_CANNOT_WRITE; break;}
    }
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK);
  }


  if(d->format==TAC_FORMAT_ABSS_SCANDITRONICS) {

    ret=TPCERROR_OK;

    /* Write the header */
    fprintf(fp, "# Scanditronics Automated Blood Measurement System\n");
    int ii;
    d->h.space_before_eq=0; d->h.space_after_eq=1; /* Set no space before ':' */
    ii=iftFindKey(&d->h, "Protocol", 0);
    if(ii>=0) ret=iftWriteItem(&d->h, ii, fp, status);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    fprintf(fp, "#                 Start   Interv       1st detector pair");
    fprintf(fp, "       2nd detector pair     AUX\n");
    fprintf(fp, "#                  time     time   coinc  singl1  singl2");
    fprintf(fp, "   coinc  singl1  singl2  counts\n");

    ii=iftFindKey(&d->h, "sampler_start_time", 0);
    if(ii>=0) {
      fprintf(fp, "# %s\n", d->h.item[ii].value);
    } else if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      fprintf(fp, "# %s\n", buf);
    }


    /* Write the sample data */
    time_t t=0.0;
    if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      //printf("scanstarttime=%s\n", buf);
      struct tm tmsod;
      if(!strDateTimeRead(buf, &tmsod)) t=mktime(&tmsod);
    }
    int i, j;
    for(i=0; i<d->sampleNr; i++) {
      fprintf(fp, "%13.1f %9.1f %9.1f", 
              d->x1[i]+(unsigned long int)t, d->x1[i], d->x2[i]-d->x1[i]);
      for(j=0; j<7; j++) {
        if(j<d->tacNr) fprintf(fp, " %7.0f", d->c[j].y[i]);
        else fprintf(fp, " %7.0f", 0.0);
      }
      if(fprintf(fp, "\n")<1) {ret=TPCERROR_CANNOT_WRITE; break;}
    }
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK);
  }


  if(d->format==TAC_FORMAT_ABSS_ALLOGG_OLD) {

    ret=TPCERROR_OK;

    if(d->tacNr<2) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
      return(TPCERROR_NO_DATA);
    }

    /* Write the header */
    fprintf(fp, "# Allogg1\n");
    int ii;
    d->h.space_before_eq=0; d->h.space_after_eq=1; /* Set no space before ':' */

    ii=iftFindKey(&d->h, "Protocol", 0);
    if(ii>=0) ret=iftWriteItem(&d->h, ii, fp, status);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    ii=iftFindKey(&d->h, "Discriminators", 0);
    if(ii>=0) ret=iftWriteItem(&d->h, ii, fp, status);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      fprintf(fp, "# %s\n", buf);
    }

    ii=iftSearchKey(&d->h, "Background", 0);
    if(ii>=0) ret=iftWriteItem(&d->h, ii, fp, status);
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    /* Start time in format HHMMSS */
    if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      fprintf(fp, "\n%2.2s%2.2s%2.2s\n", buf+11, buf+14, buf+17);
    } else {
      fprintf(fp, "\n000000\n");
    }

    /* Write the sample data */
    for(int i=0; i<d->sampleNr; i++) {
      if(fprintf(fp, "%6.1f %5.0f %5.0f\n", 
                 d->x2[i], d->c[0].y[i], d->c[1].y[i]) < 10)
      {
        ret=TPCERROR_CANNOT_WRITE; break;
      }
    }
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK);
  }


  if(d->format==TAC_FORMAT_ABSS_ALLOGG) {

    if(d->tacNr<2) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
      return(TPCERROR_NO_DATA);
    }

    int ii;
    //d->h.space_before_eq=0; d->h.space_after_eq=1; /* Set no space before ':' */
    ret=TPCERROR_OK;

    /* Write the background */
    ii=iftSearchKey(&d->h, "background counts", 0);
    if(ii>=0 && 
       fprintf(fp, "%s :\t%s\n", d->h.item[ii].key, d->h.item[ii].value)<10)
    {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }

    /* Write the header */
    fprintf(fp, "//Heading\n");

    ii=iftFindKey(&d->h, "System ID", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "User name", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "Run number", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "Patient ID", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "HalfTime", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "Tube length", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "Pump speed", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "DeadTimeSingles", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "DeadTimeCoincidents", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "DiscriminatorSingles", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "DiscriminatorCoincidents", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "EffectivitySingles", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);
    ii=iftFindKey(&d->h, "EffectivityCoincidents", 0);
    if(ii>=0) fprintf(fp, "%s:\t%s\n", d->h.item[ii].key, d->h.item[ii].value);

    /* Write the data */
    if(fprintf(fp, "//Data\n")<6) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
    fprintf(fp, "Absolute time\t");
    fprintf(fp, "Time after start [s]\t");
    fprintf(fp, "Singles [cnt]\t");
    fprintf(fp, "Coincidents [cnt]\t");
    fprintf(fp, "Singles count rate [kBq/ml]\t");
    fprintf(fp, "Coincidents count rate [kBq/ml]\t");
    fprintf(fp, "Singles count rate DTC [kBq/ml]\t");
    fprintf(fp, "Coincidents count rate DTC [kBq/ml]\t");
    fprintf(fp, "Singles DTC&decay [kBq/ml]\t");
    fprintf(fp, "Coincidents DTC&decay [kBq/ml]\n");
    struct tm tmtod;
    if(tacGetHeaderScanstarttime(&d->h, buf, NULL)==TPCERROR_OK) {
      //printf("scanstarttime=%s\n", buf);
      strDateTimeRead(buf, &tmtod);
    }
    int i, j;
    for(i=0; i<d->sampleNr; i++) {
      if(i>0) tmAdd(d->x1[i]-d->x1[i-1], &tmtod);
      if(strftime(buf, 32, "%Y-%m-%d %H:%M:%S", &tmtod)==0)
        strcpy(buf, "1970-01-01 00:00:00");
      fprintf(fp, "%s\t%.1f\t%.0f\t%.0f", 
              buf, d->x1[i], d->c[0].y[i], d->c[1].y[i]);
      for(j=2; j<8; j++) {
        if(j<d->tacNr) fprintf(fp, "\t%.12f", d->c[j].y[i]);
        else fprintf(fp, "\t%f", 0.0);
      }
      if(fprintf(fp, "\n")<1) {ret=TPCERROR_CANNOT_WRITE; break;}
    }
    if(ret!=TPCERROR_OK) {
      statusSet(status, __func__, __FILE__, __LINE__, ret); return(ret);
    }

    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return(TPCERROR_OK);
  }


  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
  return(TPCERROR_UNSUPPORTED);
}
/*****************************************************************************/

/*****************************************************************************/
