/** @file dcmvr.c
    @brief DICOM Value Representations (VRs).
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcdcm.h"
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of DICOM value representations (VRs). */
typedef struct DCM_VR {
  /** VR id */
  dcmvr vr;
  /** Pointer to the NULL terminated VR string. */
  char name[8];
  /** Nr of reserved bytes following VR; 0 or 2. */
  unsigned char res;
  /** Endian sensitive or not. */
  short int es;
  /** Corresponding max value length in bytes; 0 if not defined. */
  size_t s;
  /** Description of the VR; NULL terminated string. */
  char descr[64];
} DCM_VR;

/** Table of DICOM value representations (VRs). 

    Items must be the same and in the same order as the dcmdr enums in tpcdcm.h.
    Table can not be accessed directly outside the c file.
*/
static DCM_VR dcm_vr[]={
  {DCM_VR_AE, "AE", 0, 0,    16, "application entity"},
  {DCM_VR_AS, "AS", 0, 0,     4, "age string"},
  {DCM_VR_AT, "AT", 0, 0,     4, "attribute tag"},
  {DCM_VR_CS, "CS", 0, 0,    16, "code string"},
  {DCM_VR_DA, "DA", 0, 0,     8, "date"},
  {DCM_VR_DS, "DS", 0, 0,    16, "decimal string"}, // actual files may contain longer strings too
  {DCM_VR_DT, "DT", 0, 0,    26, "date and time"},
  {DCM_VR_FL, "FL", 0, 1,     4, "floating point single precision"},
  {DCM_VR_FD, "FD", 0, 1,     8, "floating point double precision"},
  {DCM_VR_IS, "IS", 0, 0,    12, "integer string"},
  {DCM_VR_LO, "LO", 0, 0,    64, "long string"},
  {DCM_VR_LT, "LT", 0, 0, 10240, "long text"},
  {DCM_VR_OB, "OB", 2, 0,     0, "other byte (8-bit) stream"}, // even bytes
  {DCM_VR_OD, "OD", 2, 1,     0, "other double (64-bit) stream"},
  {DCM_VR_OF, "OF", 2, 1,     0, "other float (32-bit) stream"},
  {DCM_VR_OL, "OL", 2, 1,     0, "other long (32-bit) stream"},
  {DCM_VR_OW, "OW", 2, 1,     0, "other word (16-bit) stream"},
  {DCM_VR_PN, "PN", 0, 0,    64, "person name"},
  {DCM_VR_SH, "SH", 0, 0,    16, "short string"},
  {DCM_VR_SL, "SL", 0, 1,     4, "signed long (32-bit integer)"},
  {DCM_VR_SQ, "SQ", 2, 0,     0, "sequence of elements (used for nested data)"},
  {DCM_VR_SS, "SS", 0, 1,     2, "signed short (16-bit integer)"},
  {DCM_VR_ST, "ST", 0, 0,  1024, "short text"},
  {DCM_VR_TM, "TM", 0, 0,    14, "time"},
  {DCM_VR_UC, "UC", 2, 0,     0, "unlimited characters"},
  {DCM_VR_UI, "UI", 0, 0,    64, "UID"},
  {DCM_VR_UL, "UL", 0, 1,     4, "unsigned long (32-bit integer)"},
  {DCM_VR_UN, "UN", 2, 0,     0, "unknown, any valid length of another VR"},
  {DCM_VR_UR, "UR", 2, 0,    64, "URI or URL string"},
  {DCM_VR_US, "US", 0, 1,     2, "unsigned short (16-bit integer)"},
  {DCM_VR_UT, "UT", 2, 0,     0, "unlimited text"},
  // This MUST be kept as the last element
  {DCM_VR_INVALID, "INVALID", 0, 0, 0, "invalid value representation"}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Is the explicit VR (2 bytes) followed by reserved 2 bytes?
    If yes, then the following Value Length is also given as 32-byte integer,
    if no, then as 16-bit integer.
    @return Returns 0, if not, and 2, if it is.
 */ 
unsigned char dcmVRReserved(
  /** VR id (DCM_VR_AE, ...). */ 
  dcmvr id
) {
  unsigned short int i=0;
  while(dcm_vr[i].vr!=DCM_VR_INVALID) {
    if(id==dcm_vr[i].vr) return(dcm_vr[i].res);
    i++;
  }
  return(2);
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the DICOM VR based on the two-character long string.
 *  @return Returns the VR id.
 *  @sa dcmVRName
 */
dcmvr dcmVRId(
  /** VR string. Two first characters are used. 
      String does not need to be null-terminated. */
  const char *s
) {
  if(s==NULL) return(DCM_VR_INVALID);
  char buf[3]; buf[0]=s[0]; buf[1]=s[1]; buf[2]=(char)0;

  /* Identify the VR */
  unsigned short int i=0;
  while(dcm_vr[i].vr!=DCM_VR_INVALID) {
    if(strncmp(dcm_vr[i].name, buf, 2)==0) return(dcm_vr[i].vr);
    i++;
  }
  return(DCM_VR_INVALID);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM VR name.
 *  @return Returns pointer to the name string.
 *  @sa dcmIdentifyVR
 */
char *dcmVRName(
  /** VR id (DCM_VR_AE, ...). */ 
  dcmvr id
) {
  unsigned short int i=0;
  while(dcm_vr[i].vr!=DCM_VR_INVALID) {
    if(id==dcm_vr[i].vr) return(dcm_vr[i].name);
    i++;
  }
  return(dcm_vr[DCM_VR_INVALID].name);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM VR max value length in bytes; 0 if not defined.
 *  @return Returns the length in bytes.
 *  @sa dcmIdentifyVR
 */
size_t dcmVRVLength(
  /** VR id (DCM_VR_AE, ...). */ 
  dcmvr id
) {
  unsigned short int i=0;
  while(dcm_vr[i].vr!=DCM_VR_INVALID) {
    if(id==dcm_vr[i].vr) return(dcm_vr[i].s);
    i++;
  }
  return(dcm_vr[DCM_VR_INVALID].s);
}
/*****************************************************************************/

/*****************************************************************************/
/** Get the DICOM VR description.
 *  @return Returns pointer to the description string.
 *  @sa dcmIdentifyVR
 */
char *dcmVRDescr(
  /** VR id (DCM_VR_AE, ...). */ 
  dcmvr id
) {
  unsigned short int i=0;
  while(dcm_vr[i].vr!=DCM_VR_INVALID) {
    if(id==dcm_vr[i].vr) return(dcm_vr[i].descr);
    i++;
  }
  return(dcm_vr[DCM_VR_INVALID].descr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert DICOM date 'DA' to international format YYYY-MM-DD.
 *  @return Returns pointer to the date string, or NULL in case of an error.
 */
char *dcmDA2intl(
  /** Pointer to original DICOM string. */ 
  const char *orig,
  /** Pointer to string where date in international format will be written;
      must be allocated for at least 11 characters. */
  char *intl 
) {
  if(orig==NULL || intl==NULL) return(NULL);
  if(strnlen(orig, 10)<8) return(NULL);
  if(isdigit(orig[4])) { // modern format YYYYMMDD
    sprintf(intl, "%4.4s-%2.2s-%2.2s", orig, orig+4, orig+6);
  } else { // old format YYYY.MM.DD
    sprintf(intl, "%4.4s-%2.2s-%2.2s", orig, orig+5, orig+8);
  }
  if(strDateValid(intl)) {intl[0]=(char)0; return(NULL);}
  return(intl);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert DICOM time 'TM' to international format hh:mm:ss.
 *  @return Returns pointer to the time string, or NULL in case of an error.
 */
char *dcmTM2intl(
  /** Pointer to original DICOM string. */ 
  const char *orig,
  /** Pointer to string where time in international format will be written;
      must be allocated for at least 9 characters. */
  char *intl 
) {
  if(orig==NULL || intl==NULL) return(NULL);
  if(strnlen(orig, 14)<6) return(NULL);
  if(isdigit(orig[2])) { // modern format hhmmss.fract
    sprintf(intl, "%2.2s:%2.2s:%2.2s", orig, orig+2, orig+4);
  } else { // old format hh.mm.ss
    sprintf(intl, "%2.2s:%2.2s:%2.2s", orig, orig+3, orig+6);
  }
  if(strTimeValid(intl)) {intl[0]=(char)0; return(NULL);}
  return(intl);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert DICOM datetime 'DT' to international format YYYY-MM-DD hh:mm:ss.
 *  @return Returns pointer to the time string, or NULL in case of an error.
 */
char *dcmDT2intl(
  /** Pointer to original DICOM string. 
      Should be in format YYYYMMDDhhmmss.FFFFFF+hhmm */ 
  const char *orig,
  /** Pointer to string where date and time in international format will be
      written; must be allocated for at least 20 characters. */
  char *intl 
) {
  if(orig==NULL || intl==NULL) return(NULL);
  if(strnlen(orig, 26)<14) return(NULL);
  sprintf(intl, "%4.4s-%2.2s-%2.2s %2.2s:%2.2s:%2.2s",
          orig, orig+4, orig+6, orig+8, orig+10, orig+12);
  if(strDateTimeValid(intl, NULL)) {intl[0]=(char)0; return(NULL);}
  return(intl);
}
/*****************************************************************************/

/*****************************************************************************/
