/** @file ecatmatrixlist.c
    @brief Functions for processing ECAT matrix list.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
/*****************************************************************************/
#include "tpcecat.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the ECAT matrix list struct before any use.
    @sa ecatMListFree, ecatMListRead, ecatMListPrint
 */
void ecatMListInit(
  /** Pointer to matrix list. */
  ECAT_MATRIXLIST *ml
) {
  if(ml==NULL) return;
  ml->matdir=NULL;
  ml->matrixSpace=ml->matrixNr=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for ECAT matrix list. All contents are destroyed.
    @pre Before first use initialize the struct with ecatMListInit().
    @sa ecatMListInit
 */
void ecatMListFree(
  /** Pointer to matrix list. */
  ECAT_MATRIXLIST *ml
) {
  if(ml==NULL) return;
  if(ml->matrixSpace>0) free((char*)(ml->matdir));
  ml->matrixSpace=ml->matrixNr=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Read matrix identifier code (matnum) into matrix value structure.
    @sa ecatMListMakeId, ecatMValToId, ecatMListPrint
 */
void ecatMListReadId(
  /** ECAT format: 6 or 7. */
  int format,
  /** Matrix identifier code (matnum) to decode. */
  unsigned int id,
  /** Pointer to data structure in where frame, plane etc numbers are written. */
  ECAT_MATVAL *mv
) {
  if(mv==NULL) return;
  mv->frame=mv->plane=mv->gate=mv->data=mv->bed=0;
  if(format==7) {
    mv->frame = id & 0x1FF;
    mv->plane = ((id >> 16) & 0xFF) + ((id >> 1) & 0x300);
    mv->gate  = (id >> 24) & 0x3F;
    mv->data  = ((id >> 30) & 0x3) + ((id >> 9) & 0x4);
    mv->bed   = (id >> 12) & 0xF;
  } else if(format==6) {
    mv->frame = id&0xFFF;
    mv->plane = (id>>16)&0xFF;
    mv->gate  = (id>>24)&0x3F;
    mv->data  = (id>>30)&0x3;
    mv->bed   = (id>>12)&0xF;
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Make matrix identifier (matnum) code for specific matrix.
    @sa ecatMListReadId, ecatMValToId
    @return The matrix identifier.
 */
unsigned int ecatMListMakeId(
  /** ECAT format: 6 or 7. */
  int format,
  /** Frame [0..511] for ECAT7, [0..4095] for ECAT6. */
  unsigned int frame,
  /** Plane [0..1023] for ECAT7, [0..255] for ECAT6. */
  unsigned int plane,
  /** Gate [0..63]. */
  unsigned int gate,
  /** Data [0..7] for ECAT7, [0..3] for ECAT6. */
  unsigned int data,
  /** Bed position [0..15]. */
  unsigned int bed
) {
  if(format==7) {
    return(
      (frame & 0x1FF) |        /* frame */
      ((plane & 0x300) << 1) | /* plane high */
      ((data & 0x4) << 9) |    /* data high */
      ((bed & 0xF) << 12) |    /* bed */
      ((plane & 0xFF) << 16) | /* plane low */
      ((gate & 0x3F) << 24) |  /* gate */
      ((data & 0x3) << 30)     /* data low */
    );
  } else if(format==6) {
    return(
      (frame&0xFFF) |
      ((bed&0xF)<<12) |
      ((plane&0xFF)<<16) |
      ((gate&0x3F)<<24) |
      ((data&0x3)<<30)
    );
  }
  return(0);
}
/*****************************************************************************/
/** Make matrix identifier (matnum) code for specific matrix.
    @sa ecatMListReadId, ecatMListMakeId
    @return The matrix identifier.
 */
unsigned int ecatMValToId(
  /** ECAT format: 6 or 7. */
  int format,
  /** Pointer to data structure from where frame, plane etc numbers are read. */
  ECAT_MATVAL *mv
) {
  if(mv==NULL) return(0);
  return(ecatMListMakeId(format, mv->frame, mv->plane, mv->gate, mv->data, mv->bed));
}
/*****************************************************************************/

/*****************************************************************************/
/** Read ECAT matrix list from ECAT file.
    @sa ecatMListInit, ecatMListPrint, ecatMListFree
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int ecatMListRead(
  /** ECAT format: 6 or 7. */
  int format,
  /** File pointer of file, opened with fp=fopen(filename, "rb"), to read the matrix list from. */
  FILE *fp,
  /** Pointer to initiated matrix list; any previous content is deleted. 
      @pre Matrix list must be initiated (once) before calling this. */
  ECAT_MATRIXLIST *ml,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s(%d, ...)\n", __func__, format); fflush(stdout);}

  if(fp==NULL || ml==NULL)  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return(TPCERROR_FAIL);
  }

  /* Is current platform little endian (1) or not (0) ? */
  int little=endianLittle();
  if(verbose>2) {
    if(little) printf("little endian platform\n"); else printf("big endian platform\n");
  }

  /* Make sure that matrix list is empty */
  ecatMListFree(ml);

  /* Read the blocks belonging to the matrix list */
  unsigned int blk=2; // Matrix list starts at block 2
  unsigned int blkNext=0, blkPrev=0, nrFree=0, nrUsed=0;
  unsigned int dirbuf[ECATBLKSIZE/4];
  do {
    /* Read the data block */
    if(verbose>1) printf("  reading block %u\n", blk);
    int ret=ecatReadBlock(NULL, fp, blk, (unsigned char*)dirbuf);
    if(ret!=TPCERROR_OK) {
      if(verbose>0) fprintf(stderr, "Error: cannot read block %u\n", blk);
      ecatMListFree(ml);
      statusSet(status, __func__, __FILE__, __LINE__, ret);
      return(ret);
    }
    /* Byte order conversion for integers in little/big endian platforms */
    if(little && format==7) swawbip(dirbuf, ECATBLKSIZE);
    if(!little && format==6) swawbip(dirbuf, ECATBLKSIZE);
    /* Read the "header" integers from the block */
    nrFree  = dirbuf[0];
    blkNext = dirbuf[1];
    blkPrev = dirbuf[2];
    nrUsed  = dirbuf[3];
    if(verbose>2)
      printf("nrFree=%u blkNext=%u blkPrev=%u nrUsed=%u\n", nrFree, blkNext, blkPrev, nrUsed);
    /* Allocate (more) memory for the matrix list */
    if(ml->matrixSpace==0) {
      ml->matrixSpace=ECATBLKSIZE/4;
      ml->matdir=(ECAT_MATDIR*)malloc(ml->matrixSpace*sizeof(ECAT_MATDIR));
    } else if(ml->matrixSpace<(ml->matrixNr+ECATBLKSIZE/4)) {
      ml->matrixSpace+=ECATBLKSIZE/4;
      ml->matdir=(ECAT_MATDIR*)realloc(ml->matdir, sizeof(ECAT_MATDIR)*ml->matrixSpace);
    }
    if(ml->matdir==NULL) {
      ecatMListFree(ml);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
      return(TPCERROR_OUT_OF_MEMORY);
    }
    /* Add the content of this block to the matrix list */
    for(unsigned int i=4; i<ECATBLKSIZE/4; i+=4) if(dirbuf[i]>0) {
      ml->matdir[ml->matrixNr].id=dirbuf[i];
      ml->matdir[ml->matrixNr].strtblk=dirbuf[i+1];
      ml->matdir[ml->matrixNr].endblk=dirbuf[i+2];
      ml->matdir[ml->matrixNr].status=dirbuf[i+3];
      if(verbose>3) {
        printf("matnum=%u strtblk=%u endblk=%u matstat=%u matrixNr=%u\n",
          ml->matdir[ml->matrixNr].id, ml->matdir[ml->matrixNr].strtblk,
          ml->matdir[ml->matrixNr].endblk, ml->matdir[ml->matrixNr].status,
          ml->matrixNr);
      }
      /* verify that referred data block can be found in the file */
      fpos_t current_fp; fgetpos(fp, &current_fp); // save current file position
      int ret=fseeko(fp, (off_t)(ml->matdir[ml->matrixNr].endblk-1)*ECATBLKSIZE, SEEK_SET);
      fsetpos(fp, &current_fp);  // back to saved file position
      if(ret==0) { // end block can be found
        ml->matrixNr++; // save this matrix into the list
      } else { // not found, file probably broken
        if(verbose>0) {
          printf("matnum %d points to data outside of file.\n", ml->matdir[ml->matrixNr].id);
          fflush(stdout);
        }
        // matrixNr not incremented, thus this matrix is left out of the list
      }
    }
    blk=blkNext;
  } while(feof(fp)==0 && blk>2);

  if(ml->matrixNr==0) {
    ecatMListFree(ml);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }
  if(verbose>1) printf("  matrixNr := %u\n", ml->matrixNr);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Print ECAT matrix list.
    @sa ecatMListInit, ecatMListRead, ecatMListFree
 */
void ecatMListPrint(
  /** ECAT format: 6 or 7. */
  int format,
  /** Pointer to matrix list to print. */
  ECAT_MATRIXLIST *ml,
  /** File pointer to print to; usually stdout. */
  FILE *fp
) {
  if(ml==NULL || fp==NULL) return;
  ECAT_MATVAL mv;

  fprintf(fp, "nr\tmatrix\tpl\tfr\tgate\tbed\tstartblk\tblknr\n");
  for(unsigned int i=0; i<ml->matrixNr; i++) {
    ecatMListReadId(format, ml->matdir[i].id, &mv);
    fprintf(fp, "%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\n", i+1, ml->matdir[i].id,
      mv.plane, mv.frame, mv.gate, mv.bed,
      ml->matdir[i].strtblk, 1+ml->matdir[i].endblk-ml->matdir[i].strtblk);
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
