/** @file readasciifile.c
 *  @brief Read ASCII text file raw contents for further processing.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** Get the size of ASCII text file, or size of initial ASCII part of file
    containing binary data after the text part (in certain image formats).
  
    One NUL character is not counted as binary data, but two already is;
    this, because equipment failures sometimes lead to data ending with NUL.

    @author Vesa Oikonen
    @return The number of characters in the initial ASCII part of file.
    This does not include the terminal zero that would be added to a string. 
    @sa asciiFileRead, asciiCommentLine, fileExist
 */
size_t asciiFileSize(
  /** File pointer; rewind() is called before returning. */
  FILE *fp,
  /** Value will be set to 1 if reading was stopped at a non-printable character
      meaning that file may contain a binary part, or to 0 if file contains only
      printable characters and size of all contents was counted. 
      Enter NULL, if not needed. */
  int *nonprintable
) {
  if(nonprintable!=NULL) *nonprintable=0;
  if(fp==NULL) return(0);
  size_t n, bn;
  int c;
  n=bn=0;
  while((c=fgetc(fp))!=EOF && n<SIZE_MAX-1 && bn<2) {
    if(iscntrl(c) && c!=13 && c!=10 && c!=9 && c!=169) {
      bn++;
      if(c!=0) {
        if(nonprintable!=NULL) *nonprintable=1; 
        break;
      }
    }
    n++;
  }
  rewind(fp);
  if(nonprintable!=NULL && bn>1) *nonprintable=1; 
  return n;
}
/*****************************************************************************/

/*****************************************************************************/
#if(0) // NOT finalized, better to do with strings read line-by-line

/** Count how many times characters specified in string s are found in ASCII
 *  text file. Search is case-sensitive. Search stops at the first non-printable
 *  character because certain image formats contain binary data after ASCII
 *  header part. 
 *  @return The number of specified characters found in the initial ASCII part
 *  of file.
 *  @author Vesa Oikonen
 * */
size_t asciiFileChrCount(
  /** File pointer; rewind() is called before returning. */
  FILE *fp,
  /** String containg characters which are searched for; not modified. */
  const char *s,
  /** Omit comment lines with '#' as the first character (1=omit, 0=include). */
  int omit_comments,
  /** Omit characters inside quotes (1=omit, 0=include). */
  int omit_quoted
) {
  size_t n=0;
  if(fp==NULL) return(n);
  if(s==NULL || strnlen(s, 1)==0) return(n);
  int c, j;
  int comment_line=0;
  int single_quotation=0;
  int double_quotation=0;
  while((c=fgetc(fp))!=EOF && n<SIZE_MAX-1) {
    /* Stop if non-printable character */
    if(iscntrl(c) && c!=13 && c!=10 && c!=9) break;
    /* Check for quotations */
    if(c=='\'') {
      if(single_quotation==0 && strchr(cptr+1, '\'')!=NULL) single_quotation=1;
      else single_quotation=0;
      continue;
    }
    if(*cptr=='\"') {
      if(double_quotation==0 && strchr(cptr+1, '\"')!=NULL) double_quotation=1;
      else double_quotation=0;
      continue;
    }
    if(single_quotation==1 || double_quotation==1) continue;


    /* Is character included in s ? */
    for(j=0; j<strlen(s); j++) if(c==(int)s[j]) n++;
    n++;
  }
  rewind(fp);
  return(n);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/** Read at most maxlen-1 characters from given file pointer to ASCII text file.
    @author Vesa Oikonen
    @return Pointer to the string, or NULL in case of an error.
    @sa asciiFileSize, asciiCommentLine, fileExist, filenameGetExtension
 */
char *asciiFileRead(
  /** File pointer; rewind() is NOT called here to allow reading data in chunks. */
  FILE *fp,
  /** Pointer to character array in which the data is written; NULL, if memory is to allocated here
      (in that case remember to free it after use). Terminal zero is added. */
  char *data,
  /** Nr of characters to read, and the size of data array (including terminal zero). */
  size_t maxlen
) {
  /* Check the input */
  if(fp==NULL || maxlen<2) return NULL;
  /* Allocate memory if needed */
  char *cptr;
  if(data!=NULL) cptr=data; else {
    cptr=malloc(maxlen*sizeof(char)); if(cptr==NULL) return NULL;
  }
  size_t i=0; int c;
  while((c=fgetc(fp))!=EOF && i<maxlen-1) cptr[i++]=(char)c;
  cptr[i]=(char)0; if(i==0) {if(data==NULL) free(cptr); cptr=NULL;}
  return cptr;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check if ASCII text line starts with comment character '#'.

    Comment character is searched from the first non-space character (space characters here include
    spaces and tabs).
    @return 1 if this is comment line and 0 if not.
    @author Vesa Oikonen
    @sa asciiFileRead, asciiFileSize, strTrimLeft, strstrNoQuotation
 */
int asciiCommentLine(
  /** Pointer to string containing one line of ASCII text file. */ 
  const char *line,
  /** Optional pointer which is set to the index of line where the first non-space character after
      the comment character starts. If line does not start with comment character, then this will
      point to the first non-space character of the line. Enter NULL if not needed. */
  int *cont
) {
  if(cont!=NULL) *cont=0; 
  if(line==NULL) return 0;
  char *cptr=(char*)line;
  int i=strspn(cptr, " \t"); cptr+=i; if(cont!=NULL) *cont=i;
  if(*cptr!='#') return 0;
  if(cont==NULL) return 1;
  cptr++; i=strspn(cptr, " \t"); *cont+=(i+1);
  return 1;
}
/*****************************************************************************/

/*****************************************************************************/
