/** @file statusmsg.c
 *  @brief Status information for library functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/

/*****************************************************************************/
/** List of error messages.
    @sa tpcerror, errorMsg, statusPrint
 */
static char *tpcerrormsg[] = {
// typedef enum tpcerror in tpcextensions.h must be edited accordingly
  "OK",
  "unspecified error",
  "unable to allocate memory",
  "file does not exist",
  "file exists",
  "invalid filename",
  "cannot open file",
  "cannot read file",
  "cannot write file",
  "cannot delete",
  "invalid file format",
  "unsupported file type",
  "file contains no data",
  "file contains duplicate data",
  "file contains no weights",
  "file contains no date or time",
  "file contains missing values",
  "file contains too few samples",
  "file is too big",
  "overlapping data",
  "large gap in data",
  "no sample times",
  "invalid sample time",
  "invalid sample time range",
  "incompatible units",
  "incompatible data",
  "reference not found",
  "key not found",
  "value not found",
  "invalid value",
  "invalid field delimiter",
  "invalid nr of parameters",
  "invalid header",
  "missing header",
  "unknown isotope",
  "unknown data unit",
  "no solution",
  "fitting not successful",
0};
/*****************************************************************************/

/*****************************************************************************/
/** Return pointer to error message specified with the error code.
    @return pointer to the error message string.
    @sa tpcerror, statusPrint
    @author Vesa Oikonen
 */
char *errorMsg(
  /** Error code @sa tpcerror. */
  tpcerror e
) {
  if(e<TPCERROR_OK || e>=TPCERROR_LAST) return NULL;
  return tpcerrormsg[e];
}
/*****************************************************************************/

/*****************************************************************************/
/** Print the contents of status structure.
    @author Vesa Oikonen
    @sa statusSet, errorMsg, statusInit, statusFree
 */
void statusPrint(
  /** File pointer for the output. */
  FILE *fp,
  /** Pointer to status. */
  TPCSTATUS *s
) {
  if(fp==NULL || s==NULL) return;
  if(s->current_file!=NULL && s->current_func!=0 && s->current_line>0)
    fprintf(fp, "In %s() on line %d of file %s\n", s->current_func, s->current_line, s->current_file);
  if(s->verbose<=0) return;
  if(s->last_file!=NULL && s->last_func!=0 && s->last_line>0)
    fprintf(fp, "called in %s() on line %d of file %s\n", s->last_func, s->last_line, s->last_file);
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Before first use, initialize the structure contents using this function.
    After last use of status structure, call statusFree(), which also calls
    this function to re-initiate the structure contents.
    @sa errorMsg, statusSet, statusFree
 */
void statusInit(
  /** Pointer to status. */
  TPCSTATUS *s
) {
  if(s==NULL) return;
  s->last_line=0;
  s->last_file=NULL;
  s->last_func=NULL;
  s->current_line=0;
  s->current_file=NULL;
  s->current_func=NULL;
  s->verbose=0;
  s->fp=stdout;
  s->forgiving=0;
  s->error=0;
}
/*****************************************************************************/

/*****************************************************************************/
/** After last use, free any memory allocated in the structure.
    @sa errorMsg, statusSet, statusInit
 */
void statusFree(
  /** Pointer to status. */
  TPCSTATUS *s
) {
  if(s==NULL) return;
  free(s->last_file); free(s->last_func);
  free(s->current_file); free(s->current_func);
  if(s->fp!=NULL && s->fp!=stdout && s->fp!=stderr) fclose(s->fp);
  statusInit(s);
}
/*****************************************************************************/

/*****************************************************************************/
/** Set the status contents.
    @sa statusInit, statusFree, errorMsg, tpcProgramName, tpcProcessStdOptions
 */
void statusSet(
  /** Pointer to status. */
  TPCSTATUS *s,
  /** Pointer to function name, usually __func__. */
  const char *func,
  /** Pointer to source file name, usually __FILE__. */
  const char *srcfile,
  /** Pointer to source file line number, usually __LINE__. */
  int srcline,
  /** Error code, 0 if status is ok. */
  tpcerror error
) {
  if(s==NULL) return;
  s->error=error;
  /* Delete any previous content */
  s->last_line=0;
  free(s->last_file); free(s->last_func);
  s->last_file=NULL; s->last_func=NULL;
  /* Move current content to previous */
  s->last_line=s->current_line;
  s->last_file=s->current_file;
  s->last_func=s->current_func;
  /* Set current contents */
  s->current_line=srcline;
  if(func!=NULL) s->current_func=strdup(func);
  if(srcfile!=NULL) s->current_file=strdup(srcfile);
  s->error=error;
}
/*****************************************************************************/

/*****************************************************************************/
