/** @file stringext.c
 *  @brief String processing functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
/*****************************************************************************/

/*****************************************************************************/
/** Calculate the number of tokens in the string s1. The characters making up the string s2 are 
    the delimiters that determine the token.
    @sa strTokenNCpy, strTokenDup
    @return Returns the nr of tokens.
    @author Vesa Oikonen
 */
int strTokenNr(
  /** String from where tokens are calculated; not modified in any way. */
  const char *s1,
  /** String containing character delimiters. */
  const char *s2
) {
  int i=0, n=0;
  char *cptr;
  if(s1==NULL || s2==NULL || strnlen(s1, 1)==0 || strnlen(s2, 1)==0) return(0);

  cptr=(char*)s1;
  do {
    // pass delimiter characters
    i=strspn(cptr, s2); cptr+=i;
    // pass characters between delimiters
    i=strcspn(cptr, s2); cptr+=i; if(i>0) n++; 
  } while(i>0);
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy the i'th token in the string s1 into string s3. The characters making up the string s2 
    are the delimiters that determine the tokens.
    @sa strTokenNr, strTokenDup
    @return Returns the length of token, 0 if no token(s) found.
    @author Vesa Oikonen
 */  
int strTokenNCpy(
  /** String from where tokens are searched; not modified in any way. */
  const char *s1,
  /** String containing character delimiters. */
  const char *s2,
  /** Token number to copy (1..nr of tokens). */
  int i,
  /** String array into where the token is copied; string will be null terminated. */
  char *s3,
  /** Length of s3, including terminal null. */
  int count
) {
  int j=0, n=0;
  char *cptr;
  if(s3!=NULL && count>0) s3[0]=(char)0;
  if(s1==NULL || s2==NULL || strlen(s1)==0 || strlen(s2)==0) return(0);
  if(i<1 || s3==NULL || count<2) return(0);

  cptr=(char*)s1;
  do {
    // pass delimiter characters
    j=strspn(cptr, s2); cptr+=j;
    // pass characters between delimiters
    j=strcspn(cptr, s2); if(j>0) n++;
    // if this is the required token nr, then stop here
    if(n==i) {
      if(j>count-1) j=count-1;
      strlcpy(s3, cptr, j+1); //strncpy(s3, cptr, j); s3[j]=(char)0;
      break;
    }
    cptr+=j;
  } while(j>0);
  if(n>i) {s3[0]=(char)0; return(0);}
  return(j);
}
/*****************************************************************************/

/*****************************************************************************/
/** Count how many times characters specified in string s2 are found in string s1. 

    Search is case-sensitive.
    @author Vesa Oikonen
    @return Returns the nr of characters of s2 found in s1. 
    @sa strReplaceChar, strstrNoQuotation, strUppercaseCount
 */
int strChrCount(
  /** String to search for characters; not modified. */
  const char *s1,
  /** String containing characters which are searched for; not modified. */
  const char *s2
) {
  unsigned int n=0, i, j;
  if(s1==NULL || s2==NULL || strlen(s1)==0 || strlen(s2)==0) return (int)n;
  for(i=0; i<strlen(s1); i++)
    for(j=0; j<strlen(s2); j++)
      if(s1[i]==s2[j]) n++;
  return (int)n;
}
/*****************************************************************************/

/*****************************************************************************/
/** Count how many upper case characters are found in string. 
    @return Returns the nr of upper case characters found in s. 
    @sa strChrCount
 */
int strUppercaseCount(
  /** String to search for upper case characters; not modified. */
  const char *s
) {
  unsigned int n=0, i;
  if(s==NULL || strlen(s)==0) return (int)n;
  for(i=0; i<strlen(s); i++) if(isupper(s[i])) n++;
  return (int)n;
}
/*****************************************************************************/

/*****************************************************************************/
/** Replace all characters c1 in string s with character c2.
    @sa strChrCount, strdelstr, strTrimleft, strstrNoQuotation
    @author Vesa Oikonen
 */
void strReplaceChar(
  /** Pointer to string in which the character is replaced. */
  char *s,
  /** Character to be replaced. */
  char c1,
  /** Character to use instead. If NULL, then only the first character is replaced. */
  char c2
) {
  char *cptr;
  if(s==NULL || strlen(s)==0) return;
  while((cptr=strchr(s, c1))!=NULL) *cptr=c2;
  return;
}
/*****************************************************************************/

/*****************************************************************************/
#ifndef HAVE_STRCASESTR
/** Case-insensitive version of strstr().
    @return a pointer to the beginning of the first occurrence, or NULL if not found.
    @sa strdelstr, strstrNoQuotation
 */
char *strcasestr(
  /** Pointer to string in which sub-string needle is searched. */
  const char *haystack,
  /** Pointer to sub-string which is searched for in source string haystack. */
  const char *needle
) {
  if(!haystack || !*haystack || !needle || !*needle) return 0;

  const char *s=haystack, *p=needle;
  do {
    if(!*p) return(char*)haystack;
    if((*p==*s) || (tolower(*p)==tolower(*s))) {
      p++; s++;
    } else {
      p=needle; if(!*s) return(NULL);
      s=++haystack;
    }
  } while(1);
  return *p ? NULL : (char*)haystack;
}
#endif // HAVE_STRCASESTR
/*****************************************************************************/

/*****************************************************************************/
#ifndef HAVE_STRDUP
/** Allocates memory and copies into it the string addressed by s, including
    the terminating character.
    @post Remember to free the memory from the returned pointer after last use.
    @return pointer to allocated string, or NULL in case of error.
 */
char *strdup(
  /** String to be duplicated. */
  const char *s
) {
  if(s==NULL) return NULL;
  size_t length=strlen(s)+1;
  void *r;
  r=calloc(length, sizeof(char)); if(r==NULL) return NULL;
  return (char*)memcpy(r, s, length);
}
#endif // HAVE_STRDUP
/*****************************************************************************/

/*****************************************************************************/
#ifndef HAVE_STRNDUP
/** Allocates memory and copies into it first n characters from string addressed by s, 
    and adds the terminating character.
    @post Remember to free the memory from the returned pointer after last use.
    @return pointer to allocated string, or NULL in case of error.
 */
char *strndup(
  /** String to be duplicated. */
  const char *s,
  /** Max nr of characters to copy, not including terminal zero. */
  size_t n
) {
  size_t length=strnlen(s, n);
  void *r; r=calloc(length+1, sizeof(char)); if(r==NULL) return NULL;
  return (char*)memcpy(r, s, length);
}
#endif // HAVE_STRNDUP
/*****************************************************************************/

/*****************************************************************************/
/** Version of strstr() which ignores any matches inside quotation marks "" or ''.
    @return a pointer to the beginning of the first occurrence, or NULL if not found.
    @author Vesa Oikonen
    @remark This function is an enhanced version of the one from libtpcmisc. 
    @sa strClean, strCleanSpaces
 */
char *strstrNoQuotation(
  /** Pointer to string to be searched. */
  const char *haystack,
  /** Pointer to string with quotation marks. */
  const char *needle
) {
  if(haystack==NULL) return((char*)NULL);
  if(needle==NULL) return((char*)haystack);

  size_t i, test_len;
  int single_quotation=0;
  int double_quotation=0;
  char *cptr;
  
  test_len=strlen(needle); if(test_len<1) return((char*)haystack);
  for(i=0, cptr=(char*)haystack; i<strlen(haystack); i++, cptr++) {
    if(*cptr=='\'') {
      if(single_quotation==0 && strchr(cptr+1, '\'')!=NULL) single_quotation=1;
      else single_quotation=0;
      continue;
    }
    if(*cptr=='\"') {
      if(double_quotation==0 && strchr(cptr+1, '\"')!=NULL) double_quotation=1;
      else double_quotation=0;
      continue;
    }
    if(single_quotation==1 || double_quotation==1) continue;
    if(strncmp(cptr, needle, test_len)==0) return(cptr);
  }
  return((char*)NULL);
}
/*****************************************************************************/

/*****************************************************************************/
/** Version of strncpy() which as usual copies s2 to s1, but without any
    space characters or line end characters that may be around the string s2.
    @return the length of the new string s1.
    @author Vesa Oikonen
    @sa strCleanSpaces, strncpyClean, strIsSpaceOnly
 */
int strncpyCleanSpaces(
  /** Pointer to pre-allocated result string with length of at least maxlen characters, 
      including NULL character. */
  char *s1,
  /** Pointer to the original string. */
  const char *s2,
  /** Max length of s1, including the trailing zero. */
  int maxlen
) {
  if(s1==NULL) return(0);
  s1[0]=(char)0; if(maxlen<1) return(0);
  if(maxlen<2) {strcpy(s1, ""); return(0);}
  if(s2==NULL || strlen(s2)<1) return(0);

  char *cptr;
  int i;

  cptr=(char*)s2; cptr+=strspn(s2, "\t\n\r ");
  strlcpy(s1, cptr, maxlen); i=strlen(s1); if(i<1) return(0);
  cptr=s1+(i-1);
  while(i>0) {
    if(*cptr!='\t' && *cptr!='\n' && *cptr!='\r' && *cptr!=' ') break;
    i--; s1[i]=(char)0; cptr--;
  }
  return(i);
}
/*****************************************************************************/

/*****************************************************************************/
/** Removes any initial and trailing space characters from specified string s.
    Space characters in the middle of the string are not removed.
    @return 0 when successful, otherwise >0.
    @author Vesa Oikonen
    @sa strClean, strncpyClean, strncpyCleanSpaces, strIsSpaceOnly
 */
int strCleanSpaces(
  /** Pointer to the string. */
  char *s
) {
  if(s==NULL) return 0;
  int len=strlen(s); if(len<0) return 0;
  char *s2; s2=strdup(s); if(s2==NULL) return(1);
  int n=strncpyCleanSpaces(s2, s, len+1);
  if(n<1) strcpy(s, ""); else strcpy(s, s2);
  free(s2);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Version of strncpy() which as usual copies s2 to s1, but without any quotation marks 
    ("" or ''), space characters, or line end characters that may be around the string s2.
    @return the length of the new string s1.
    @author Vesa Oikonen
    @sa strncpyCleanSpaces, strCleanSpaces, strClean, strIsSpaceOnly, strstrNoQuotation
 */
int strncpyClean(
  /** Pointer to pre-allocated result string with length of at least maxlen characters, 
      including NULL character. */
  char *s1,
  /** Pointer to the original string. */
  const char *s2,
  /** Max length of s1, including the trailing zero. */
  int maxlen
) {
  if(s1==NULL) return(0);
  s1[0]=(char)0; if(maxlen<1) return(0);
  if(maxlen<2) {strcpy(s1, ""); return(0);}
  if(s2==NULL || strlen(s2)<1) return(0);

  char *tmp;
  int i, sqn, dqn, m;

  /* Allocate temp string */
  tmp=calloc(strlen(s2)+1, sizeof(char));

  /* Trim space characters */
  i=strncpyCleanSpaces(tmp, s2, strlen(s2)+1);
  if(i<1) {strcpy(s1, ""); free(tmp); return(0);}

  /* Loop as long as there is something to trim */
  i=1;
  while(i) {
    m=0; // nothing modified so far  
    /* Count quotes */
    sqn=strChrCount(tmp, "\'");
    dqn=strChrCount(tmp, "\"");
    if(sqn==0 && dqn==0) {i=0; break;}
    /* Trim matching quotation marks */
    if(dqn>=2 && tmp[0]=='\"' && tmp[strlen(tmp)-1]=='\"') {
      tmp[0]=' '; tmp[strlen(tmp)-1]=(char)0; dqn-=2; m+=2;
    }
    if(sqn>=2 && tmp[0]=='\'' && tmp[strlen(tmp)-1]=='\'') {
      tmp[0]=' '; tmp[strlen(tmp)-1]=(char)0; sqn-=2; m+=2;
    }
    /* Trim orphan quotation marks */
    if(sqn%2) {
      if(tmp[0]=='\'') {tmp[0]=' '; sqn--; m++;}
      else if(tmp[strlen(tmp)-1]=='\'') {tmp[strlen(tmp)-1]=(char)0; sqn--; m++;}
    }
    if(dqn%2) {
      if(tmp[0]=='\"') {tmp[0]=' '; dqn--; m++;}
      else if(tmp[strlen(tmp)-1]=='\"') {tmp[strlen(tmp)-1]=(char)0; dqn--; m++;}
    }
    /* If nothing was modified, then do not continue */
    if(m==0) {i=0; break;}
    /* Trim spaces, quit if error encountered */
    if(strCleanSpaces(tmp)) i=0;
  }

  strlcpy(s1, tmp, maxlen);
  free(tmp);
  return(strlen(s1));
}
/*****************************************************************************/

/*****************************************************************************/
/** Removes any initial and trailing space characters from specified string s.
    Removes also quotation marks ("" or '') that may be around the string s.
    Quotes or space characters in the middle of the string are not removed.
    @return 0 when successful, otherwise >0.
    @author Vesa Oikonen
    @sa strncpyCleanSpaces, strCleanSpaces, strIsSpaceOnly, strCleanPars
 */
int strClean(
  /** Pointer to the string. */
  char *s
) {
  if(s==NULL) return 0;
  int len=strlen(s); if(len<0) return 0;
  char *s2; s2=calloc(len+1, sizeof(char)); if(s2==NULL) return(1);
  int n=strncpyClean(s2, s, len+1);
  if(n<1) strcpy(s, ""); else strcpy(s, s2);
  free(s2);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Search the string s1 for the first token. The characters making
    up the string s2 are the delimiters that determine the tokens.

    @return Returns pointer to a copy of the token string, or NULL in case of 
    an error or if no token found.
    @post Remember to free the memory from the returned pointer after last use.
    @author Vesa Oikonen
    @sa strTokenNCpy, strstrNoQuotation
 */
char *strTokenDup(
  /** String from where tokens are searched; not modified in any way. */
  const char *s1,
  /** String containing character delimiters. */
  const char *s2,
  /** Index of s1 where the token ended; set to NULL, if not needed. */
  int *next
) {
  if(next!=NULL) *next=0;
  if(s1==NULL) return NULL;

  char *s3=NULL, *cptr;
  size_t j;
  
  /* If no delimiters, then return copy of s1 */
  if(s2==NULL || strlen(s2)<1) {
    s3=strdup(s1); if(next!=NULL) *next=strlen(s1);
    return s3;
  } 
  /* Pass initial delimiter characters */
  cptr=(char*)s1; j=strspn(cptr, s2); cptr+=j; if(next!=NULL) *next=j;
  /* calculate characters between delimiters */
  j=strcspn(cptr, s2); if(j==0) {return NULL;}
  if(next!=NULL) *next+=j;
  /* Allocate space for token */
  s3=calloc(j+1, sizeof(char)); if(s3==NULL) return NULL;
  strlcpy(s3, cptr, j+1);
  return s3;
}
/*****************************************************************************/

/*****************************************************************************/
/** Checks whether string is inside parenthesis (s), [s], or {s}, that is, 
    the same pars as the first and last character of the string.

    @pre If necessary, remove possible spaces or quotation marks before this. 
    @return 0 if not in pars, 1 if inside pars.
    @author Vesa Oikonen
    @sa strCleanPars, strCleanSpaces, strIsValidNumber
 */
int strInPars(
  /** Pointer to the string. */
  char *s
) {
  if(s==NULL) return 0;
  int len=strlen(s); if(len<2) return 0;
  if(s[0]=='(' && s[len-1]==')') return 1;
  if(s[0]=='{' && s[len-1]=='}') return 1;
  if(s[0]=='[' && s[len-1]==']') return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove parenthesis (s), [s], or {s} from around the string s.
 
    Only one set of pars are removed, and only if the same pars are 
    as the first and last character of the string.

    @pre If necessary, remove possible spaces or quotation marks before this. 
    @sa strInPars, strCleanSpaces
    @author Vesa Oikonen
 */
void strCleanPars(
  /** Pointer to the string. */
  char *s
) {
  if(s==NULL || strInPars(s)==0) return;
  int len=strlen(s);
  s[--len]=(char)0;
  for(int i=1; i<=len; i++) s[i-1]=s[i];
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Concatenate not more than count characters of the string representation
    of integer n to the string pointed to by str1. 
    String is terminated with a null.
    @sa strncatDouble, strncatIntZ, strIsValidNumber
    @return Pointer to str1.
 */
char *strncatInt(
  /** String to be catenated. */
  char *str1,
  /** Integer value to be catenated as string into str1. */
  const int n,
  /** No more than count characters are concatenated to str1.
      If str1 was allocated like char str1[MAX_LEN], then set count=MAX_LEN-strlen(str1)-1.
  */
  size_t count
) {
  char tmp[128]; sprintf(tmp, "%d", n);
  return strncat(str1, tmp, count);
}
/*****************************************************************************/

/*****************************************************************************/
/** Concatenate not more than count characters of the string representation
    of double d to the string pointed to by str1. 
    String is terminated with a null.
    @sa strncatInt 
    @return Pointer to str1.
 */
char *strncatDouble(
  /** String to be catenated. */
  char *str1,
  /** Double value to be catenated as string into str1. */
  const double d,
  /** No more than count characters are concatenated to str1.
      If str1 was allocated like char str1[MAX_LEN], then set count=MAX_LEN-strlen(str1)-1.
   */
  size_t count
) {
  char tmp[128]; sprintf(tmp, "%g", d);
  return strncat(str1, tmp, count);
}
/*****************************************************************************/

/*****************************************************************************/
/** Concatenate not more than count characters of the zero-padded string 
    representation of integer n to the string pointed to by str1. 
    String is terminated with a null.
    @sa strncatInt, strncatDouble
    @return Pointer to str1.
 */  
char *strncatIntZ(
  /** String to be catenated. */
  char *str1,
  /** Integer value to be catenated as string into str1. */
  const int n,
  /** Integer value which is large enough that it would not need to be zero-padded. */
  const int maxn,
  /** No more than count characters are concatenated to str1.
      If str1 was allocated like char str1[MAX_LEN], then set count=MAX_LEN-strlen(str1)-1.
   */
  size_t count
) {
  if(n<0) strncat(str1, "-", count);
  char tmp[128]; sprintf(tmp, "%d", abs(maxn));
  int len=strlen(tmp);
  sprintf(tmp, "%0*d", len, abs(n));
  return strncat(str1, tmp, count);
}
/*****************************************************************************/

/*****************************************************************************/
#ifndef HAVE_STRNLEN
/** Safer version of strlen, in case the argument s is not NUL terminated string. 
    Computes the length of string s, but never scans beyond the n first bytes of the string. 
    @remark Included in POSIX and GCC, so this implementation may not be needed.
    @return same as strlen() or n, whicever is smaller. 
 */
size_t strnlen(
  /** Pointer to string, or character array, that may not be NULL terminated. */
  const char *s,
  /** The actual length of buffer allocated for the string;
      for example, string could have been allocated as char s[n]; */ 
  size_t n
) {
  if(s==NULL) return(0);
  char *ps=(char*)s;
  size_t i=0;
  while(i<n && *ps!='\0') {i++; ps++;}
  return(i);
}
#endif // HAVE_STRNLEN
/*****************************************************************************/

/*****************************************************************************/
#ifndef HAVE_STRLCAT
/** Safer version of strncat. 

    At most dstsize-1 characters are appended from the source string to destination string. 
    Destination string will be NUL terminated, unless dstsize <= strlen(dst). 
    @remark Included in POSIX but not in GCC.
    @return the size of the buffer that would have been needed for the
    destination string; if >=dstsize, then truncation occurred. 
 */
size_t strlcat(
  /** Destination string. */
  char *dst,
  /** Source string. */
  const char *src,
  /** The actual length of buffer allocated for the destination string;
      for example, destination string has been allocated as char dst[dstsize];
   */ 
  size_t dstsize
) {
  char *d;
  const char *s=src;
  size_t dlen, n;

  /* Find the current length of dst */
  dlen=strnlen(dst, dstsize);
  if(s==NULL) return(dlen);
  n=dstsize-dlen;
  if(n==0) return(dlen+strlen(s));
  d=dst+dlen;
  while(*s!='\0') {
    if(n!=1) {*d=*s; d++; n--;}
    s++;
  }
  *d='\0';
  return(dlen+(s-src));
}
#endif // HAVE_STRLCAT
/*****************************************************************************/

/*****************************************************************************/
#ifndef HAVE_STRLCPY
/** Safer version of strncpy or strcpy.

    At most dstsize-1 characters are copied from the source string to destination string. 
    Destination string will be NUL terminated. 
    @remark Included in POSIX but not in GCC.
    @return the size of the buffer that would have been needed for the destination string; 
    if >=dstsize, then truncation occurred. 
 */
size_t strlcpy(
  /** Destination string. */
  char *dst,
  /** Source string. */
  const char *src,
  /** The actual length of buffer allocated for the destination string;
      for example, destination string has been allocated as char dst[dstsize];
   */ 
  size_t dstsize
) {
  if(dstsize>0) dst[0]='\0';
  if(strlen(src)==0) return(0);

  char *d=dst;
  const char *s=src;
  size_t n;

  /* Copy as many chars as allowed */
  n=dstsize;
  if(n!=0) while(--n!=0) {*d=*s; if(*d=='\0') {d++; s++; break;} d++; s++;}
  if(n==0) { // not enough space, add NUL, and check how much space were needed
    if(dstsize!=0) *d='\0';
    while(*s++) {}
  }
  return(s-src-1);
}
#endif // HAVE_STRLCPY
/*****************************************************************************/

/*****************************************************************************/
/** Checks whether string contains only space characters.
    @details Text files saved in text editor or Excel may accidentally contain empty looking lines 
    which still may contain tabs and spaces, leading to problems when reading them for example 
    as CSV files.
    @return 0 if lines contains something more than spaces, 1 if only space characters or 
    string is empty.
    @author Vesa Oikonen
    @sa strClean, strncpyCleanSpaces
 */
int strIsSpaceOnly(
  /** Pointer to the string. */
  char *s
) {
  if(s==NULL) return(1);
  int len=strlen(s); if(len<1) return(1);
  for(int i=0; i<len; i++) if(!isspace(s[i])) return(0);
  return 1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find (case-insensitive) s2 in s1, and delete it from s1.
   @return Returns pointer to s1 at where the deletion stopped, or to the
    start of s1 in case s2 was not found.
   @sa strReplaceChar, strClean, strTrimleft
 */
char *strdelstr(
  /** Pointer to string in which sub-string s2 is searched; modified. */
  char *s1,
  /** Pointer to sub-string which is searched for in source string s1. */
  const char *s2
) {
  if(s1==NULL || s2==NULL) return(s1); //printf("strdelstr('%s', '%s')\n", s1, s2);
  int n=strlen(s2); if(n<1) return(s1);
  char *cptr, *rptr;
  cptr=strcasestr(s1, s2);
  if(cptr==NULL || *cptr==(char)0) return(s1);
  rptr=cptr;
  while(*cptr) {*cptr=cptr[n]; cptr++;}
  return(rptr);
}
/*****************************************************************************/

/*****************************************************************************/
/** Strip string of t characters from the left.
    @return Returns pointer to the string.
    @sa filenameGetExtension
 */
char *strTrimLeft(
  /** Pointer to the string to be trimmed. Overwritten. */
  char *s,
  /** Number of characters to remove. */
  const size_t t
) {
  if(s==NULL || t==0) return(s);
  size_t len=strlen(s); if(len<1) return(s);
  if(t>=len) {s[0]=(char)0; return(s);}
  memmove(s, s+t, 1+len-t);
  return(s);
}
/*****************************************************************************/

/*****************************************************************************/
/** Encode special characters for XML, including SVG.
    @post Free the memory of returned string pointer.
    @return Returns pointer to encoded string.
            NULL is returned in case of an error, or if encoding is not necessary.
    @sa strCleanForXML
 */
char *strEncodeForXML(
  /** Pointer to the string to be encoded. */
  const char *s
) {
  if(s==NULL) return(NULL);
  /* Count the characters needing encoding */
  int n=0;
  for(size_t i=0; i<strlen(s); i++) {
    if(s[i]=='&') {n++; continue;}
    if(s[i]=='\'') {n++; continue;}
    if(s[i]=='\"') {n++; continue;}
    if(s[i]=='<') {n++; continue;}
    if(s[i]=='>') {n++; continue;}
  }
  if(n==0) return(NULL);
  /* Allocate memory for new string (one char to max 6 chars) */
  n*=5; n+=strlen(s)+1;
  char *ns=(char*)malloc(n*sizeof(char));
  if(ns==NULL) return(NULL);
  /* Process the string */
  for(int i=0; i<n; i++) ns[i]=(char)0;
  for(size_t i=0; i<strlen(s); i++) {
    if(s[i]=='&') {strcat(ns, "&amp;"); continue;}
    if(s[i]=='\'') {strcat(ns, "&apos;"); continue;}
    if(s[i]=='\"') {strcat(ns, "&quot;"); continue;}
    if(s[i]=='<') {strcat(ns, "&lt;"); continue;}
    if(s[i]=='>') {strcat(ns, "&gt;"); continue;}
    ns[strlen(ns)]=s[i];
  }
  return(ns);
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove from string those characters that would require encoding in XML.
    Replaced by character '-'. 
    @sa strEncodeForXML
 */
void strCleanForXML(
  /** Pointer to the string to be cleaned. */
  char *s
) {
  if(s==NULL || *s=='\0') return;
  strReplaceChar(s, '&', '-');
  strReplaceChar(s, '\'', '-');
  strReplaceChar(s, '\"', '-');
  strReplaceChar(s, '<', '-');
  strReplaceChar(s, '>', '-');
  return;
}
/*****************************************************************************/

/*****************************************************************************/
