/** @file units.c
 *  @brief Concentration and time unit functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/

/*****************************************************************************/
/*! @cond PRIVATE */
/** One item for table of units.
 */
typedef struct TPC_UNIT {
  /** Pointer to the NULL terminated key string; NULL if not allocated */
  char name[MAX_UNITS_LEN];
  /** Unit #1 of dividend */
  int u1;
  /** Unit #2 of dividend */
  int u2;
  /** Unit #1 of divider */
  int v1;
  /** Unit #2 of divider */
  int v2;
} TPC_UNIT;

/** Table of units. */
static TPC_UNIT tpc_unit[]={
  {"unknown",         UNIT_UNKNOWN, UNIT_UNKNOWN, UNIT_UNKNOWN, UNIT_UNKNOWN},
  {"unitless",        UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"HU",              UNIT_HU, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"msec",            UNIT_MSEC, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"sec",             UNIT_SEC,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"min",             UNIT_MIN,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"h",               UNIT_HOUR, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"days",            UNIT_DAY,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"months",          UNIT_MONTH, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"y",               UNIT_YEAR,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"um",              UNIT_UM,   UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mm",              UNIT_MM,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"cm",              UNIT_CM,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"m",               UNIT_M,  UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"uL",              UNIT_UL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mL",              UNIT_ML, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"dL",              UNIT_DL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"L",               UNIT_L, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"ug",              UNIT_UG, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mg",              UNIT_MG, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"g",               UNIT_G, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"100g",            UNIT_100G, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"kg",              UNIT_KG, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"pmol",            UNIT_PMOL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"nmol",            UNIT_NMOL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"umol",            UNIT_UMOL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mmol",            UNIT_MMOL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mol",             UNIT_MOL, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"counts",          UNIT_COUNTS, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"kcounts",         UNIT_KCOUNTS, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"Bq",              UNIT_BQ, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"kBq",             UNIT_KBQ, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"MBq",             UNIT_MBQ, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"GBq",             UNIT_GBQ, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"nCi",             UNIT_NCI, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"uCi",             UNIT_UCI, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mCi",             UNIT_MCI, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"Ci",              UNIT_CI, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"cps",             UNIT_COUNTS, UNIT_UNITLESS, UNIT_SEC, UNIT_UNITLESS},
  {"kcps",            UNIT_KCOUNTS, UNIT_UNITLESS, UNIT_SEC, UNIT_UNITLESS},
  {"cpm",             UNIT_COUNTS, UNIT_UNITLESS, UNIT_MIN, UNIT_UNITLESS},
  {"kcpm",            UNIT_KCOUNTS, UNIT_UNITLESS, UNIT_MIN, UNIT_UNITLESS},
  {"Bq/mL",           UNIT_BQ, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"kBq/mL",          UNIT_KBQ, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"MBq/mL",          UNIT_MBQ, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"nCi/mL",          UNIT_NCI, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"uCi/mL",          UNIT_UCI, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"Bq/g",            UNIT_BQ, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"kBq/g",           UNIT_KBQ, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"MBq/g",           UNIT_MBQ, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"nCi/g",           UNIT_NCI, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"uCi/g",           UNIT_UCI, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"pmol/mL",         UNIT_PMOL, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"nmol/mL",         UNIT_NMOL, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"umol/mL",         UNIT_NMOL, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"mmol/mL",         UNIT_MMOL, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"pmol/L",          UNIT_PMOL, UNIT_UNITLESS, UNIT_L, UNIT_UNITLESS},
  {"nmol/L",          UNIT_NMOL, UNIT_UNITLESS, UNIT_L, UNIT_UNITLESS},
  {"umol/L",          UNIT_NMOL, UNIT_UNITLESS, UNIT_L, UNIT_UNITLESS},
  {"mmol/L",          UNIT_MMOL, UNIT_UNITLESS, UNIT_L, UNIT_UNITLESS},
  {"mol/L",           UNIT_MOL, UNIT_UNITLESS, UNIT_L, UNIT_UNITLESS},
  {"sec*kBq/mL",      UNIT_KBQ, UNIT_SEC, UNIT_ML, UNIT_UNITLESS},
  {"min*kBq/mL",      UNIT_KBQ, UNIT_MIN, UNIT_ML, UNIT_UNITLESS},
  {"sec*Bq/mL",       UNIT_BQ, UNIT_SEC, UNIT_ML, UNIT_UNITLESS},
  {"min*Bq/mL",       UNIT_BQ, UNIT_MIN, UNIT_ML, UNIT_UNITLESS},
  {"%",               UNIT_PERCENTAGE, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"mL/mL",           UNIT_ML, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"mL/dL",           UNIT_ML, UNIT_UNITLESS, UNIT_DL, UNIT_UNITLESS},
  {"g/mL",            UNIT_G, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"mL/g",            UNIT_ML, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"1/sec",           UNIT_UNITLESS, UNIT_UNITLESS, UNIT_SEC, UNIT_UNITLESS},
  {"1/min",           UNIT_UNITLESS, UNIT_UNITLESS, UNIT_MIN, UNIT_UNITLESS},
  {"1/h",             UNIT_UNITLESS, UNIT_UNITLESS, UNIT_HOUR, UNIT_UNITLESS},
  {"mL/(mL*sec)",     UNIT_ML, UNIT_UNITLESS, UNIT_ML, UNIT_SEC},
  {"mL/(mL*min)",     UNIT_ML, UNIT_UNITLESS, UNIT_ML, UNIT_MIN},
  {"mL/(dL*min)",     UNIT_ML, UNIT_UNITLESS, UNIT_DL, UNIT_MIN},
  {"mL/(g*min)",      UNIT_ML, UNIT_UNITLESS, UNIT_G, UNIT_MIN},
  {"mL/(100g*min)",   UNIT_ML, UNIT_UNITLESS, UNIT_100G, UNIT_MIN},
  {"umol/(mL*min)",   UNIT_UMOL, UNIT_UNITLESS, UNIT_ML, UNIT_MIN},
  {"umol/(dL*min)",   UNIT_UMOL, UNIT_UNITLESS, UNIT_DL, UNIT_MIN},
  {"umol/(g*min)",    UNIT_UMOL, UNIT_UNITLESS, UNIT_G, UNIT_MIN},
  {"umol/(100g*min)", UNIT_UMOL, UNIT_UNITLESS, UNIT_100G, UNIT_MIN},
  {"mmol/(mL*min)",   UNIT_MMOL, UNIT_UNITLESS, UNIT_ML, UNIT_MIN},
  {"mmol/(dL*min)",   UNIT_MMOL, UNIT_UNITLESS, UNIT_DL, UNIT_MIN},
  {"mmol/(g*min)",    UNIT_MMOL, UNIT_UNITLESS, UNIT_G, UNIT_MIN},
  {"mmol/(100g*min)", UNIT_MMOL, UNIT_UNITLESS, UNIT_100G, UNIT_MIN},
  {"mg/(100g*min)",   UNIT_MG, UNIT_UNITLESS, UNIT_100G, UNIT_MIN},
  {"mg/(dL*min)",     UNIT_MG, UNIT_UNITLESS, UNIT_DL, UNIT_MIN},
  {"%ID",             UNIT_PID, UNIT_UNITLESS, UNIT_UNITLESS, UNIT_UNITLESS},
  {"%ID/g",           UNIT_PID, UNIT_UNITLESS, UNIT_G, UNIT_UNITLESS},
  {"%ID/mL",          UNIT_PID, UNIT_UNITLESS, UNIT_ML, UNIT_UNITLESS},
  {"%ID/kg",          UNIT_PID, UNIT_UNITLESS, UNIT_KG, UNIT_UNITLESS},
  {"%ID/L",           UNIT_PID, UNIT_UNITLESS, UNIT_L, UNIT_UNITLESS},
  {"MBq/nmol",        UNIT_MBQ, UNIT_UNITLESS, UNIT_NMOL, UNIT_UNITLESS},
  {"GBq/nmol",        UNIT_GBQ, UNIT_UNITLESS, UNIT_NMOL, UNIT_UNITLESS},
  {"",                UNIT_UNKNOWN, UNIT_UNKNOWN, UNIT_UNKNOWN, UNIT_UNKNOWN}
};
/*! @endcond */
/*****************************************************************************/

/*****************************************************************************/
/** Get the string representation of the unit specified with its unit_code.

    @return pointer to string representation of the unit, "unknown" if not identified.
    @sa unitIdentify, unitIsTime, unitCombination
 */
char *unitName(
  /** unit_code as enum or the index of unit in the table. */
  int unit_code
) {
  int n=0;
  
  while(strlen(tpc_unit[n].name)>0) n++;
  if(unit_code<1 || unit_code>n-1)
    return tpc_unit[UNIT_UNKNOWN].name;
  else
    return(tpc_unit[unit_code].name);
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify the unit based on string representation s of the unit.
    @return enum unit, or 0 (enum UNIT_UNKNOWN) if not identified.
    @sa unitName, unitIdentifyFilename, unitIsTime, unitDividerMassVolumeConversion
 */
int unitIdentify(
  /** Unit as a string. */
  const char *s
) {
  if(s==NULL || strnlen(s, 1)<1) return UNIT_UNKNOWN;
  /* Try if string can be found directly in the table */
  int i, n=0;
  while(strlen(tpc_unit[n].name)>0) n++;
  for(i=0; i<n; i++) {
    if(strcasecmp(tpc_unit[i].name, s)==0) return i;
  }
  /* Unit string is not following TPC standard, lets try something else */
  if(     strcasecmp(s, "ms")==0)               return UNIT_MSEC;
  else if(strcasecmp(s, "s")==0)                return UNIT_SEC;
  else if(strcasecmp(s, "second")==0)           return UNIT_SEC;
  else if(strcasecmp(s, "seconds")==0)          return UNIT_SEC;
  else if(strcasecmp(s, "mins")==0)             return UNIT_MIN;
  else if(strcasecmp(s, "minute")==0)           return UNIT_MIN;
  else if(strcasecmp(s, "minutes")==0)          return UNIT_MIN;
  else if(strcasecmp(s, "hours")==0)            return UNIT_HOUR;
  else if(strcasecmp(s, "days")==0)             return UNIT_DAY;
  else if(strcasecmp(s, "months")==0)           return UNIT_MONTH;
  else if(strcasecmp(s, "years")==0)            return UNIT_YEAR;
  else if(strcasecmp(s, "yrs")==0)              return UNIT_YEAR;
  else if(strcasecmp(s, "microm")==0)           return UNIT_UM;
  else if(strcasecmp(s, "millim")==0)           return UNIT_MM;
  else if(strcasecmp(s, "microl")==0)           return UNIT_UL;
  else if(strcasecmp(s, "mm^3")==0)             return UNIT_UL;
  else if(strcasecmp(s, "cc")==0)               return UNIT_ML;
  else if(strcasecmp(s, "liter")==0)            return UNIT_L;
  else if(strcasecmp(s, "microgram")==0)        return UNIT_UG;
  else if(strcasecmp(s, "milligram")==0)        return UNIT_MG;
  else if(strcasecmp(s, "gram")==0)             return UNIT_G;
  else if(strcasecmp(s, "100 g")==0)            return UNIT_100G;
  else if(strcasecmp(s, "kilogram")==0)         return UNIT_KG;
  else if(strcasecmp(s, "micromol")==0)         return UNIT_UMOL;
  else if(strcasecmp(s, "micromoles")==0)       return UNIT_UMOL;
  else if(strcasecmp(s, "millimol")==0)         return UNIT_MMOL;
  else if(strcasecmp(s, "millimoles")==0)       return UNIT_MMOL;
  else if(strcasecmp(s, "mole")==0)             return UNIT_MOL;
  else if(strcasecmp(s, "moles")==0)            return UNIT_MOL;
  else if(strcasecmp(s, "cnts")==0)             return UNIT_COUNTS;
  else if(strcasecmp(s, "kcnts")==0)            return UNIT_KCOUNTS;
  else if(strcasecmp(s, "becquerels")==0)       return UNIT_BQ;
  else if(strcasecmp(s, "kbecquerels")==0)      return UNIT_KBQ;
  else if(strcasecmp(s, "cnts/s")==0)           return UNIT_CPS;
  else if(strcasecmp(s, "counts/s")==0)         return UNIT_CPS;
  else if(strcasecmp(s, "counts/sec")==0)       return UNIT_CPS;
  else if(strcasecmp(s, "ECAT counts/sec")==0)  return UNIT_CPS;
  else if(strcasecmp(s, "kcounts/s")==0)        return UNIT_KCPS;
  else if(strcasecmp(s, "kcounts/sec")==0)      return UNIT_KCPS;
  else if(strcasecmp(s, "counts/min")==0)       return UNIT_CPM;
  else if(strcasecmp(s, "kcounts/min")==0)      return UNIT_KCPM;
  else if(strcasecmp(s, "Bq/cc")==0)            return UNIT_BQ_PER_ML;
  else if(strcasecmp(s, "Bqcc")==0)             return UNIT_BQ_PER_ML;
  else if(strcasecmp(s, "BqmL")==0)             return UNIT_BQ_PER_ML;
  else if(strcasecmp(s, "kBq/cc")==0)           return UNIT_KBQ_PER_ML;
  else if(strcasecmp(s, "kBqcc")==0)            return UNIT_KBQ_PER_ML;
  else if(strcasecmp(s, "kBqmL")==0)            return UNIT_KBQ_PER_ML;
  else if(strcasecmp(s, "MBq/cc")==0)           return UNIT_MBQ_PER_ML;
  else if(strcasecmp(s, "MBqcc")==0)            return UNIT_MBQ_PER_ML;
  else if(strcasecmp(s, "MBqmL")==0)            return UNIT_MBQ_PER_ML;
  else if(strcasecmp(s, "nCi/cc")==0)           return UNIT_NCI_PER_ML;
  else if(strcasecmp(s, "nCicc")==0)            return UNIT_NCI_PER_ML;
  else if(strcasecmp(s, "nCimL")==0)            return UNIT_NCI_PER_ML;
  else if(strcasecmp(s, "uCi/cc")==0)           return UNIT_UCI_PER_ML;
  else if(strcasecmp(s, "uCicc")==0)            return UNIT_UCI_PER_ML;
  else if(strcasecmp(s, "uCimL")==0)            return UNIT_UCI_PER_ML;
  else if(strcasecmp(s, "integral")==0)         return UNIT_SEC_KBQ_PER_ML;
  else if(strcasecmp(s, "percent")==0)          return UNIT_PERCENTAGE;
  else if(strcasecmp(s, "perc")==0)             return UNIT_PERCENTAGE;
  else if(strcasecmp(s, "cc/cc")==0)            return UNIT_ML_PER_ML;
  else if(strcasecmp(s, "mL/cc")==0)            return UNIT_ML_PER_ML;
  else if(strcasecmp(s, "cc/mL")==0)            return UNIT_ML_PER_ML;
  else if(strcasecmp(s, "g/cc")==0)             return UNIT_G_PER_ML;
  else if(strcasecmp(s, "SUV")==0)              return UNIT_G_PER_ML;
  else if(strcasecmp(s, "SUVbw")==0)            return UNIT_G_PER_ML;
  else if(strcasecmp(s, "SUV_bw")==0)           return UNIT_G_PER_ML;
  else if(strcasecmp(s, "cc/g")==0)             return UNIT_ML_PER_G;
  else if(strcasecmp(s, "1/s")==0)              return UNIT_PER_SEC;
  else if(strcasecmp(s, "s-1")==0)              return UNIT_PER_SEC;
  else if(strcasecmp(s, "min-1")==0)            return UNIT_PER_MIN;
  else if(strcasecmp(s, "h-1")==0)              return UNIT_PER_HOUR;
  else if(strcmp(s, "pM")==0)                   return UNIT_PMOL_PER_L;
  else if(strcmp(s, "nM")==0)                   return UNIT_NMOL_PER_L;
  else if(strcmp(s, "uM")==0)                   return UNIT_UMOL_PER_L;
  else if(strcmp(s, "mM")==0)                   return UNIT_MMOL_PER_L;
  else if(strcasecmp(s, "mL/(sec*mL)")==0)      return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "mL/sec/mL")==0)        return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "mL/mL/sec")==0)        return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "cc/sec/cc")==0)        return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "cc/cc/sec")==0)        return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "mL/(sec*cc)")==0)      return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "mL/(cc*sec)")==0)      return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "cc/(cc*sec)")==0)      return UNIT_ML_PER_ML_SEC;
  else if(strcasecmp(s, "mL/(min*mL)")==0)      return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "mL/min/mL")==0)        return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "mL/mL/min")==0)        return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "cc/min/cc")==0)        return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "cc/cc/min")==0)        return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "mL/(min*g)")==0)       return UNIT_ML_PER_G_MIN;
  else if(strcasecmp(s, "mL/(min*cc)")==0)      return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "mL/(cc*min)")==0)      return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "cc/(cc*min)")==0)      return UNIT_ML_PER_ML_MIN;
  else if(strcasecmp(s, "mL/(min*dL)")==0)      return UNIT_ML_PER_DL_MIN;
  else if(strcasecmp(s, "mL/min/dL")==0)        return UNIT_ML_PER_DL_MIN;
  else if(strcasecmp(s, "mL/dL/min")==0)        return UNIT_ML_PER_DL_MIN;
  else if(strcasecmp(s, "umol/(min*mL)")==0)    return UNIT_UMOL_PER_ML_MIN;
  else if(strcasecmp(s, "umol/(min*dL)")==0)    return UNIT_UMOL_PER_DL_MIN;
  else if(strcasecmp(s, "umol/dL/min)")==0)     return UNIT_UMOL_PER_DL_MIN;
  else if(strcasecmp(s, "umol/(min*g)")==0)     return UNIT_UMOL_PER_G_MIN;
  else if(strcasecmp(s, "umol/(min*100g)")==0)  return UNIT_UMOL_PER_100G_MIN;
  else if(strcasecmp(s, "mmol/(min*mL)")==0)    return UNIT_MMOL_PER_ML_MIN;
  else if(strcasecmp(s, "mmol/(min*dL)")==0)    return UNIT_MMOL_PER_DL_MIN;
  else if(strcasecmp(s, "mmol/dL/min)")==0)     return UNIT_MMOL_PER_DL_MIN;
  else if(strcasecmp(s, "mmol/(min*g)")==0)     return UNIT_MMOL_PER_G_MIN;
  else if(strcasecmp(s, "mmol/(min*100g)")==0)  return UNIT_MMOL_PER_100G_MIN;
  else if(strcasecmp(s, "% ID")==0)             return UNIT_PID;
  else if(strcasecmp(s, "%i.d.")==0)            return UNIT_PID;
  else if(strcasecmp(s, "PID")==0)              return UNIT_PID;
  else if(strcasecmp(s, "% ID/mL")==0)          return UNIT_PID_PER_ML;
  else if(strcasecmp(s, "%i.d./mL")==0)         return UNIT_PID_PER_ML;
  else if(strcasecmp(s, "PID/mL")==0)           return UNIT_PID_PER_ML;
  else if(strcasecmp(s, "%ID/cc")==0)           return UNIT_PID_PER_ML;
  else if(strcasecmp(s, "% ID/cc")==0)          return UNIT_PID_PER_ML;
  else if(strcasecmp(s, "PID/cc")==0)           return UNIT_PID_PER_ML;
  else if(strcasecmp(s, "% ID/L")==0)           return UNIT_PID_PER_L;
  else if(strcasecmp(s, "%i.d./L")==0)          return UNIT_PID_PER_L;
  else if(strcasecmp(s, "PID/L")==0)            return UNIT_PID_PER_L;
  else if(strcasecmp(s, "% ID/g")==0)           return UNIT_PID_PER_G;
  else if(strcasecmp(s, "%i.d./g")==0)          return UNIT_PID_PER_G;
  else if(strcasecmp(s, "PID/g")==0)            return UNIT_PID_PER_G;
  else if(strcasecmp(s, "% ID/kg")==0)          return UNIT_PID_PER_KG;
  else if(strcasecmp(s, "%i.d./kg")==0)         return UNIT_PID_PER_KG;
  else if(strcasecmp(s, "PID/kg")==0)           return UNIT_PID_PER_KG;
  else if(strcasecmp(s, "Hounsfield Unit")==0)  return UNIT_HU;
  else if(strcasecmp(s, "HU")==0)               return UNIT_HU;
  else if(strcasecmp(s, "1/1")==0)              return UNIT_UNITLESS;

  return UNIT_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
/** Identify calibration unit based on file name.
 
    @return enum unit, or 0 (enum UNIT_UNKNOWN) if not identified.
    @sa unitName, unitIdentify, unitDividerHasVolume, unitIsRAConc
 */
int unitIdentifyFilename(
  /** File name. */
  const char *s
) {
  if(s==NULL || strnlen(s, 3)<3) return UNIT_UNKNOWN;
  char *cptr;
  for(int i=0; i<2; i++) {
    if(i==0) /* On the first run, look in the extension */
      {cptr=strrchr((char*)s, '.'); if(cptr==NULL) {cptr=(char*)s; i++;}}
    else /* Then, look into whole filename */
      cptr=(char*)s;
    if(strcasestr(cptr, "KBQ")) return UNIT_KBQ_PER_ML;
    if(strcasestr(cptr, "MBQ")) return UNIT_MBQ_PER_ML;
    if(strcasestr(cptr, "BQ")) return UNIT_BQ_PER_ML;
    if(strcasestr(cptr, "NCI")) return UNIT_NCI_PER_ML;
    if(strcasestr(cptr, "KCPS")) return UNIT_KCPS;
    if(strcasestr(cptr, "CPS")) return UNIT_CPS;
    if(strcasestr(cptr, "SUV")) return UNIT_G_PER_ML;
  }
  return UNIT_UNKNOWN;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a unit of distance.
    @return 1 if distance unit, otherwise 0.
    @sa unitIsTime, unitIdentify, unitName
 */
int unitIsDistance(
  /** Enum unit. */
  int u
) {
  switch(u) {
    case UNIT_UM:
    case UNIT_MM:
    case UNIT_CM:
    case UNIT_M:
      return 1;  
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a unit of time.
    @return 1 if time unit, otherwise 0.
    @sa unitIsDistance, unitConversionFactor, unitInverse, unitIsRAConc, unitIdentify
 */
int unitIsTime(
  /** Enum unit. */
  int u
) {
  switch(u) {
    case UNIT_MSEC:
    case UNIT_SEC:
    case UNIT_MIN:
    case UNIT_HOUR:
    case UNIT_DAY:
    case UNIT_MONTH:
    case UNIT_YEAR:
      return 1;    
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a unit of volume.
    @return 1 if volume unit, otherwise 0.
    @sa unitDividerHasVolume, unitIsMass, unitName
 */
int unitIsVolume(
  /** Enum unit. */
  int u
) {
  switch(u) {
    case UNIT_UL:
    case UNIT_ML:
    case UNIT_DL:
    case UNIT_L:
      return 1;    
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a unit of mass.
    @return 1 if mass unit, otherwise 0.
    @sa unitDividerHasMass, unitIsVolume, unitIsMole
 */
int unitIsMass(
  /** Enum unit. */
  int u
) {
  switch(u) {
    case UNIT_UG:
    case UNIT_MG:
    case UNIT_G:
    case UNIT_100G:
    case UNIT_KG:
      return 1;    
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a unit of chemical mass.
    @return 1 if mole unit, otherwise 0.
    @sa unitIsRadioactivity, unitIsMass
 */
int unitIsMole(
  /** Enum unit. */
  int u
) {
  switch(u) {
    case UNIT_PMOL:
    case UNIT_NMOL:
    case UNIT_UMOL:
    case UNIT_MMOL:
    case UNIT_MOL:
      return 1;    
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a unit of radioactivity.
    @return 1 if radioactivity unit, otherwise 0.
    @sa unitIsRAConc, unitIsMole, unitIsMass
 */
int unitIsRadioactivity(
  /** Enum unit */
  int u
) {
  switch(u) {
    case UNIT_BQ:
    case UNIT_KBQ:
    case UNIT_MBQ:
    case UNIT_GBQ:
    case UNIT_NCI:
    case UNIT_UCI:
    case UNIT_MCI:
    case UNIT_CI:
      return 1;    
  }
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a combinatorial unit.
    @return 1 if combination unit, otherwise 0.
    @sa unitConversionFactor, unitIsRadioactivity, unitIsVolume, unitIsMole, unitCombination
 */
int unitIsCombinatorial(
  /** Enum unit */
  int u
) {
  if(u<1 || u>=UNIT_LAST) return 0;
  if(tpc_unit[u].u2==UNIT_UNITLESS && tpc_unit[u].v1==UNIT_UNITLESS &&
     tpc_unit[u].v2==UNIT_UNITLESS) return 0;
  return 1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculates conversion factor between specified two units, if possible.

    Multiply value with factor to get the value in unit2.
    @return Conversion factor (or NaN, if conversion is not possible).
    @sa unitIsCombinatorial, unitIdentify, unitName, unitInverse, unitCombination,
        unitDividerMassVolumeConversion, unitIsRAConc
 */
double unitConversionFactor(
  /** Enum unit 1. */
  const int u1,
  /** Enum unit 2. */
  const int u2
) {
  //printf("unitConversionFactor(%s, %s)\n", unitName(u1), unitName(u2));
  double cf=nan("");
  if(u1==UNIT_UNKNOWN || u2==UNIT_UNKNOWN) return cf;
  if(u1==u2) return 1.0;

  /* If both are time units, then this is easy */
  if(unitIsTime(u1) && unitIsTime(u2)) {
    if(u1==UNIT_MSEC) cf=1.0E-3;
    else if(u1==UNIT_SEC) cf=1.0;
    else if(u1==UNIT_MIN) cf=60.0;
    else if(u1==UNIT_HOUR) cf=3600.0;
    else if(u1==UNIT_DAY) cf=24.0*36000.0;
    else if(u1==UNIT_MONTH) cf=30.0*24.0*36000.0;
    else if(u1==UNIT_YEAR) cf=365.0*30.0*24.0*36000.0;
    if(u2==UNIT_MSEC) cf/=1.0E-3;
    else if(u2==UNIT_SEC) cf/=1.0;
    else if(u2==UNIT_MIN) cf/=60.0;
    else if(u2==UNIT_HOUR) cf/=3600.0;
    else if(u2==UNIT_DAY) cf/=24.0*36000.0;
    else if(u2==UNIT_MONTH) cf/=30.0*24.0*36000.0;
    else if(u2==UNIT_YEAR) cf/=365.0*30.0*24.0*36000.0;
    else cf=nan("");
    return cf;
  }

  /* If both are distances */
  if(unitIsDistance(u1) && unitIsDistance(u2)) {
    if(u1==UNIT_UM) cf=1.0E-6;
    else if(u1==UNIT_MM) cf=1.0E-3;
    else if(u1==UNIT_CM) cf=1.0E-2;
    else if(u1==UNIT_M) cf=1.0;
    if(u2==UNIT_UM) cf/=1.0E-6;
    else if(u2==UNIT_MM) cf/=1.0E-3;
    else if(u2==UNIT_CM) cf/=1.0E-2;
    else if(u2==UNIT_M) cf/=1.0;
    else cf=nan("");
    return cf;
  }

  /* If both are volumes */
  if(unitIsVolume(u1) && unitIsVolume(u2)) {
    if(u1==UNIT_UL) cf=1.0E-6;
    else if(u1==UNIT_ML) cf=1.0E-3;
    else if(u1==UNIT_DL) cf=1.0E-1;
    else if(u1==UNIT_L) cf=1.0;
    if(u2==UNIT_UL) cf/=1.0E-6;
    else if(u2==UNIT_ML) cf/=1.0E-3;
    else if(u2==UNIT_DL) cf/=1.0E-1;
    else if(u2==UNIT_L) cf/=1.0;
    else cf=nan("");
    return cf;
  }

  /* If both are masses */
  if(unitIsMass(u1) && unitIsMass(u2)) {
    if(u1==UNIT_UG) cf=1.0E-6;
    else if(u1==UNIT_MG) cf=1.0E-3;
    else if(u1==UNIT_G) cf=1.0;
    else if(u1==UNIT_100G) cf=1.0E+2;
    else if(u1==UNIT_KG) cf=1.0E+3;
    if(u2==UNIT_UG) cf/=1.0E-6;
    else if(u2==UNIT_MG) cf/=1.0E-3;
    else if(u2==UNIT_G) cf/=1.0;
    else if(u2==UNIT_100G) cf/=1.0E+2;
    else if(u2==UNIT_KG) cf/=1.0E+3;
    else cf=nan("");
    return cf;
  }

  /* If both are moles */
  if(unitIsMole(u1) && unitIsMole(u2)) {
    if(u1==UNIT_PMOL)      cf=1.0E-12;
    else if(u1==UNIT_NMOL) cf=1.0E-9;
    else if(u1==UNIT_UMOL) cf=1.0E-6;
    else if(u1==UNIT_MMOL) cf=1.0E-3;
    else if(u1==UNIT_MOL)  cf=1.0;
    if(u2==UNIT_PMOL)      cf/=1.0E-12;
    else if(u2==UNIT_NMOL) cf/=1.0E-9;
    else if(u2==UNIT_UMOL) cf/=1.0E-6;
    else if(u2==UNIT_MMOL) cf/=1.0E-3;
    else if(u2==UNIT_MOL)  cf/=1.0;
    else cf=nan("");
    return cf;
  }

  /* If both are radioactivities */
  if(unitIsRadioactivity(u1) && unitIsRadioactivity(u2)) {
    if(u1==UNIT_BQ)       cf=1.0;
    else if(u1==UNIT_KBQ) cf=1.0E+3;
    else if(u1==UNIT_MBQ) cf=1.0E+6;
    else if(u1==UNIT_GBQ) cf=1.0E+9;
    else if(u1==UNIT_NCI) cf=37.0;
    else if(u1==UNIT_UCI) cf=3.7E+4;
    else if(u1==UNIT_MCI) cf=3.7E+7;
    else if(u1==UNIT_CI)  cf=3.7E+10;
    if(u2==UNIT_BQ)       cf/=1.0;
    else if(u2==UNIT_KBQ) cf/=1.0E+3;
    else if(u2==UNIT_MBQ) cf/=1.0E+6;
    else if(u2==UNIT_GBQ) cf/=1.0E+9;
    else if(u2==UNIT_NCI) cf/=37.0;
    else if(u2==UNIT_UCI) cf/=3.7E+4;
    else if(u2==UNIT_MCI) cf/=3.7E+7;
    else if(u2==UNIT_CI)  cf/=3.7E+10;
    else cf=nan("");
    return cf;
  }
  
  /* Fraction vs percentage */
  if(u1==UNIT_PERCENTAGE) {
    if(u2==UNIT_PERCENTAGE) cf=1.0;
    else if(u2==UNIT_UNITLESS) cf=0.01;
    return cf;
  }    
  if(u2==UNIT_PERCENTAGE) {
    if(u1==UNIT_PERCENTAGE) cf=1.0;
    else if(u1==UNIT_UNITLESS) cf=100.0;
    return cf;
  }    

  /* Counts */
  if(u1==UNIT_COUNTS) {
    if(u2==UNIT_COUNTS) cf=1.0;
    else if(u2==UNIT_KCOUNTS) cf=0.001;
    return cf;
  }    
  if(u1==UNIT_KCOUNTS) {
    if(u2==UNIT_KCOUNTS) cf=1.0;
    else if(u2==UNIT_COUNTS) cf=1000.0;
    return cf;
  }    
  
  
  /* And last, the combination units, recursively */
  if(!unitIsCombinatorial(u1) || !unitIsCombinatorial(u2)) return cf;

  if(tpc_unit[u1].u1==UNIT_UNITLESS && tpc_unit[u2].u1==UNIT_UNITLESS)
    cf=1.0;
  else
    cf=unitConversionFactor(tpc_unit[u1].u1, tpc_unit[u2].u1);
  if(!isnan(cf)) {
    if(tpc_unit[u1].u2!=UNIT_UNITLESS && tpc_unit[u2].u2!=UNIT_UNITLESS)
      cf*=unitConversionFactor(tpc_unit[u1].u2, tpc_unit[u2].u2);
    if(!isnan(cf)) {
      if(tpc_unit[u1].v1!=UNIT_UNITLESS && tpc_unit[u2].v1!=UNIT_UNITLESS)
        cf/=unitConversionFactor(tpc_unit[u1].v1, tpc_unit[u2].v1);
      if(!isnan(cf)) {
        if(tpc_unit[u1].v2!=UNIT_UNITLESS && tpc_unit[u2].v2!=UNIT_UNITLESS)
          cf/=unitConversionFactor(tpc_unit[u1].v2, tpc_unit[u2].v2);
      }
    }
  }
  
  return cf;
}
/*****************************************************************************/

/*****************************************************************************/
/** Inverse of given unit.
    @return enum unit, or 0 (enum UNIT_UNKNOWN) if not supported.
    @sa unitConversionFactor, unitIsCombinatorial, unitIsTime
 */
int unitInverse(
  /** Unit (enum) */
  int u
) {
  if(u==UNIT_UNKNOWN) return(UNIT_UNKNOWN);
  else if(u==UNIT_UNITLESS) return(UNIT_UNITLESS);
  else if(u==UNIT_SEC) return(UNIT_PER_SEC);
  else if(u==UNIT_PER_SEC) return(UNIT_SEC);
  else if(u==UNIT_MIN) return(UNIT_PER_MIN);
  else if(u==UNIT_PER_MIN) return(UNIT_MIN);
  else if(u==UNIT_HOUR) return(UNIT_PER_HOUR);
  else if(u==UNIT_PER_HOUR) return(UNIT_HOUR);
  else if(u==UNIT_ML_PER_ML) return(UNIT_ML_PER_ML);
  else if(u==UNIT_G_PER_ML) return(UNIT_ML_PER_G);
  else if(u==UNIT_ML_PER_G) return(UNIT_G_PER_ML);
  else return(UNIT_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit has volume in divider.
    @return 1 if volume found in divider, otherwise 0.
    @sa unitIsVolume, unitDividerHasMass, unitInverse, unitDividerMassVolumeConversion
 */
int unitDividerHasVolume(
  /** Enum unit */
  int u
) {
  if(!unitIsCombinatorial(u)) return 0;
  if(unitIsVolume(tpc_unit[u].v1)) return 1;
  if(unitIsVolume(tpc_unit[u].v2)) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit has mass in divider.
    @return 1 if mass found in divider, otherwise 0.
    @sa unitDividerHasVolume, unitIsMass, unitInverse, unitDividerMassVolumeConversion
 */
int unitDividerHasMass(
  /** Enum unit */
  int u
) {
  if(!unitIsCombinatorial(u)) return 0;
  if(unitIsMass(tpc_unit[u].v1)) return 1;
  if(unitIsMass(tpc_unit[u].v2)) return 1;
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit has radioactivity in the dividend.
    @return 1 if radioactivity in dividend, otherwise 0.
    @sa unitIsRadioactivity, unitIsRAConc
 */
int unitDividendHasRadioactivity(
  /** Enum unit */
  int u
) {
  if(unitIsRadioactivity(u)) return(1);
  if(!unitIsCombinatorial(u)) return(0);
  if(unitIsRadioactivity(tpc_unit[u].u1) || unitIsRadioactivity(tpc_unit[u].u2)) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the unit is a known unit of radioactivity per mass or volume.
    @return 1 if radioactivity concentration unit, otherwise 0.
    @sa unitDividendHasRadioactivity, unitDividerHasVolume, unitDividerHasMass
 */
int unitIsRAConc(
  /** Enum unit */
  int u
) {
  if(!unitIsCombinatorial(u)) return(0);
  if(!unitDividendHasRadioactivity(u)) return(0);
  if(!unitDividerHasMass(u) && !unitDividerHasVolume(u)) return(0);
  if(!unitIsRadioactivity(tpc_unit[u].u1)) return(0);
  if(!unitIsMass(tpc_unit[u].v1) && !unitIsVolume(tpc_unit[u].v1)) return(0);
  if(tpc_unit[u].u2!=UNIT_UNITLESS || tpc_unit[u].v2!=UNIT_UNITLESS) return(0);
  return(1);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether the combination of units is an identifiable combinatorial unit.
    @return enum unit, or 0 (enum UNIT_UNKNOWN) if not supported.
    @sa unitConversionFactor, unitIsCombinatorial, unitIdentify, unitName
 */
int unitCombination(
  /** Unit #1 of dividend */
  const int u1,
  /** Unit #2 of dividend */
  const int u2,
  /** Unit #1 of divider */
  const int v1,
  /** Unit #2 of divider */
  const int v2
) {
  /* Try if unit table has this combination */
  int i, n=0;
  while(strlen(tpc_unit[n].name)>0) n++;
  for(i=0; i<n; i++) {
    if(!((u1==tpc_unit[i].u1 && u2==tpc_unit[i].u2) || (u1==tpc_unit[i].u2 && u2==tpc_unit[i].u1)))
      continue;
    if(!((v1==tpc_unit[i].v1 && v2==tpc_unit[i].v2) || (v1==tpc_unit[i].v2 && v2==tpc_unit[i].v1)))
      continue;
    return i;
  }
  return(UNIT_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Convert unit that has either volume or mass unit in divider to corresponding
    unit (assuming density of 1 g/mL) with either mass or volume in divider.
    @return enum unit, or 0 (enum UNIT_UNKNOWN) if not supported.
    @sa unitConversionFactor, unitDividerHasVolume, unitDividerHasMass
 */
int unitDividerMassVolumeConversion(
  /** Unit (enum) */
  int u
) {
  if(unitDividerHasVolume(u)) {
    int uo=UNIT_UNKNOWN, uu=UNIT_UNKNOWN, uoo=UNIT_UNKNOWN;
    if(unitIsVolume(tpc_unit[u].v1)) {uo=tpc_unit[u].v1; uoo=tpc_unit[u].v2;} 
    else if(unitIsVolume(tpc_unit[u].v2)) {uo=tpc_unit[u].v2; uoo=tpc_unit[u].v1;} 
    if(uo==UNIT_ML) uu=UNIT_G;
    else if(uo==UNIT_DL) uu=UNIT_100G;
    else if(uo==UNIT_L) uu=UNIT_KG;
    else if(uo==UNIT_UL) uu=UNIT_MG;
    if(uu==UNIT_UNKNOWN) return(UNIT_UNKNOWN);
    return(unitCombination(tpc_unit[u].u1, tpc_unit[u].u2, uoo, uu));
  } else if(unitDividerHasMass(u)) {
    int uo=UNIT_UNKNOWN, uu=UNIT_UNKNOWN, uoo=UNIT_UNKNOWN;
    if(unitIsMass(tpc_unit[u].v1)) {uo=tpc_unit[u].v1; uoo=tpc_unit[u].v2;} 
    else if(unitIsMass(tpc_unit[u].v2)) {uo=tpc_unit[u].v2; uoo=tpc_unit[u].v1;} 
    if(uo==UNIT_G) uu=UNIT_ML;
    else if(uo==UNIT_100G) uu=UNIT_DL;
    else if(uo==UNIT_KG) uu=UNIT_L;
    else if(uo==UNIT_MG) uu=UNIT_UL;
    if(uu==UNIT_UNKNOWN) return(UNIT_UNKNOWN);
    return(unitCombination(tpc_unit[u].u1, tpc_unit[u].u2, uoo, uu));
  }
  return(UNIT_UNKNOWN);
}
/*****************************************************************************/

/*****************************************************************************/
/** Multiply two units.
    @return enum unit, or 0 (enum UNIT_UNKNOWN) if combination is not supported.
    @sa unitConversionFactor, unitCombination
 */
int unitMultiply(
  /** Unit A */
  int ua,
  /** Unit B */
  int ub
) {
  /* Verify validity of input units */
  if(ua<1 || ub<1) return(UNIT_UNKNOWN);
  int n=0; while(strlen(tpc_unit[n].name)>0) n++;
  if(ua>n-1 || ub>n-1) return(UNIT_UNKNOWN);

  /* Collect actual components of units */
  int u[4], v[4], nu=0, nv=0;
  for(int i=0; i<4; i++) u[i]=v[i]=UNIT_UNITLESS;
  if(tpc_unit[ua].u1!=UNIT_UNITLESS) u[nu++]=tpc_unit[ua].u1;
  if(tpc_unit[ua].u2!=UNIT_UNITLESS) u[nu++]=tpc_unit[ua].u2;
  if(tpc_unit[ua].v1!=UNIT_UNITLESS) v[nv++]=tpc_unit[ua].v1;
  if(tpc_unit[ua].v2!=UNIT_UNITLESS) v[nv++]=tpc_unit[ua].v2;
  if(tpc_unit[ub].u1!=UNIT_UNITLESS) u[nu++]=tpc_unit[ub].u1;
  if(tpc_unit[ub].u2!=UNIT_UNITLESS) u[nu++]=tpc_unit[ub].u2;
  if(tpc_unit[ub].v1!=UNIT_UNITLESS) v[nv++]=tpc_unit[ub].v1;
  if(tpc_unit[ub].v2!=UNIT_UNITLESS) v[nv++]=tpc_unit[ub].v2;
#if(0)
  printf("units: (");
  for(int i=0; i<nu; i++) printf(" %s", unitName(u[i]));
  printf(" ) / (");
  for(int j=0; j<nv; j++) printf(" %s", unitName(v[j]));
  printf(" )\n");
#endif

  /* Cancel out units if possible */
  int cn=0;
  for(int i=0; i<nu; i++) for(int j=0; j<nv; j++) if(u[i]==v[j]) {u[i]=v[i]=UNIT_UNITLESS; cn++;}
  if(cn>0) {
#if(0)
    printf("units after cancelling: (");
    for(int i=0; i<nu; i++) printf(" %s", unitName(u[i]));
    printf(" ) / (");
    for(int j=0; j<nv; j++) printf(" %s", unitName(v[j]));
    printf(" )\n");
#endif
    for(int i=0; i<nu-1; i++) if(u[i]==UNIT_UNITLESS)
      for(int j=i+1; j<nu; i++) if(u[j]!=UNIT_UNITLESS) {
        int s=u[i]; u[i]=u[j]; u[j]=s;
      }
    for(int i=0; i<nv-1; i++) if(v[i]==UNIT_UNITLESS)
      for(int j=i+1; j<nv; i++) if(v[j]!=UNIT_UNITLESS) {
        int s=v[i]; v[i]=v[j]; v[j]=s;
      }
    nu-=cn; nv-=cn;
#if(0)
    printf("units after removing gaps: (");
    for(int i=0; i<nu; i++) printf(" %s", unitName(u[i]));
    printf(" ) / (");
    for(int j=0; j<nv; j++) printf(" %s", unitName(v[j]));
    printf(" )\n");
#endif
  }

  /* Is the new combination a valid unit? */
  if(nu>2 || nv>2) return(UNIT_UNKNOWN);
  return(unitCombination(u[0], u[1], v[0], v[1]));
}
/*****************************************************************************/

/*****************************************************************************/
