/** @file filexist.c
 *  @brief Functions for checking if file exists or how many files do exist.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcfileutil.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check if specified file (not path) exists.
   @sa pathExist, pathFileNr, pathFileList, filenameRmPath, fncasematch, asciiFileRead
   @return Returns 1, if file exists, and 0 if it does not or is not a regular file.
 */
int fileExist(
  /** Name of file to check. */
  const char *filename
) {
  if(filename==NULL || strnlen(filename, 2)<1) return(0);
  struct stat fst;
  if(stat(filename, &fst)!=0) return(0);
  if(S_ISREG(fst.st_mode)) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** @return Returns the number of regular files in specified path.
    @sa pathExist, fileExist, pathFileList
 */
unsigned short int pathFileNr(
  /** Name of path. */
  const char *pathname
) {
  if(pathname==NULL || strnlen(pathname, 2)<1) return(0);
  /* Check that path exists */
  if(!pathExist(pathname)) return(0);
  /* Open the directory */
  DIR *dp;
  dp=opendir(pathname); if(dp==NULL) return(0);
#if(1)
  char tempname[2*FILENAME_MAX];
  unsigned short int n=0;
  struct dirent *de;
  /* Go through the directory */
  while((de=readdir(dp))!=NULL) {
    //printf("d_name='%s'\n", de.d_name);
    if(de->d_name[0]=='.') continue; /* Ignore hidden and 'system' dirs */
    /* Combine path and name */
    sprintf(tempname, "%s/%s", pathname, de->d_name);
    //printf("name='%s'\n", tempname);
    /* Check that it is regular file */
    if(fileExist(tempname)) n++;
  }
#else  // readdir_r deprecated
  struct dirent de, *deptr;
  char tempname[2*FILENAME_MAX];
  unsigned short int n=0;
  /* Go through the directory */
  while(readdir_r(dp, &de, &deptr)==0 && deptr!=NULL) {
    //printf("d_name='%s'\n", de.d_name);
    if(de.d_name[0]=='.') continue; /* Ignore hidden and 'system' dirs */
    /* Combine path and name */
    sprintf(tempname, "%s/%s", pathname, de.d_name);
    //printf("name='%s'\n", tempname);
    /* Check that it is regular file */
    if(fileExist(tempname)) n++;
  }
#endif
  closedir(dp);
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
/** List the names regular files in specified path into IFT structure.
    File names will contain the directory path. 
    @return Returns the number of stored regular file names.
    @sa pathFileNr, iftInit, iftFree
 */
unsigned short int pathFileList(
  /** Name of path. */
  const char *pathname,
  /** Pointer to initiated IFT structure; any previous contents are kept. */
  IFT *ift
) {
  if(pathname==NULL || strnlen(pathname, 2)<1) return(0);
  if(ift==NULL) return(0);
  /* Check that path exists */
  if(!pathExist(pathname)) return(0);
  /* Open the directory */
  DIR *dp;
  dp=opendir(pathname); if(dp==NULL) return(0);
#if(1)
  char tempname[2*FILENAME_MAX];
  unsigned short int n=0;
  struct dirent *de;
  /* Go throught the directory */
  while((de=readdir(dp))!=NULL) {
    if(de->d_name[0]=='.') continue; /* Ignore hidden and 'system' dirs */
    /* Combine path and name */
    sprintf(tempname, "%s/%s", pathname, de->d_name);
    /* Check that it is regular file */
    if(!fileExist(tempname)) continue;
    /* Add to IFT */
    if(iftPut(ift, NULL, tempname, 0, NULL)) break;
    n++;
  }
#else  // readdir_r deprecated
  struct dirent de, *deptr;
  char tempname[2*FILENAME_MAX];
  unsigned short int n=0;
  /* Go throught the directory */
  while(readdir_r(dp, &de, &deptr)==0 && deptr!=NULL) {
    if(de.d_name[0]=='.') continue; /* Ignore hidden and 'system' dirs */
    /* Combine path and name */
    sprintf(tempname, "%s/%s", pathname, de.d_name);
    /* Check that it is regular file */
    if(!fileExist(tempname)) continue;
    /* Add to IFT */
    if(iftPut(ift, NULL, tempname, 0, NULL)) break;
    n++;
  }
#endif
  closedir(dp);
  return(n);
}
/*****************************************************************************/

/*****************************************************************************/
