/** @file func.c
 *  @brief Mathematical functions evaluation library C file.
 *  @todo Add evaluation functions. 
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcfunc.h"
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate the function values at specified x values.
    @return Returns non-zero in case of an error.
    @author Vesa Oikonen
    @todo Add support for more functions.
    @sa mfEvalInt, mfEvalFrameY, parExampleTTACs, parExamplePerfectBolus
 */
int mfEvalY(
  /** Function code. */
  const char *fid,
  /** Nr of function parameters. */
  const int parNr, 
  /** Array of function parameters. */
  double *p,
  /** Size of x and y arrays. */
  const int sampleNr, 
  /** Array of x values where function values are to be evaluated. */
  double *x,
  /** Pointer to allocated array of y values where function values will be written. */
  double *y,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) {
    printf("%s('%s', %d, p[], %d, x[], y[], %d)\n", __func__, fid, parNr, sampleNr, verbose);
    fflush(stdout);
  }
  if(parNr<1 || p==NULL || sampleNr<1 || x==NULL || y==NULL) return(1);

  if(strcasecmp(fid, "step")==0) {
    /* Parameter number must be an even number and >=2 */
    if(parNr<2 || parNr&1) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0;
      for(int j=0; j<parNr; j+=2) if(x[i]>=p[j]) y[i]=p[j+1];
    }
    return(0);
  }

  if(strcasecmp(fid, "level")==0) {
    for(int i=0; i<sampleNr; i++) y[i]=p[0];
    return(0);
  }

  if(strcasecmp(fid, "fengm2")==0) {
    if(parNr<6) return(2);
    double delayt=0.0; if(parNr>6) delayt=p[6];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; if(!(xt>0.0)) continue;
      y[i] += (p[0]*xt-p[2]-p[4])*exp(p[1]*xt);
      y[i] += p[2]*exp(p[3]*xt);
      y[i] += p[4]*exp(p[5]*xt);
    }
    return(0);
  }
  if(strcasecmp(fid, "fengm2s")==0) {
    if(parNr<4) return(2);
    double delayt=0.0; if(parNr>4) delayt=p[4];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; if(!(xt>0.0)) continue;
      y[i] += (p[0]*xt-p[2])*exp(p[1]*xt);
      y[i] += p[2]*exp(p[3]*xt);
    }
    return(0);
  }
  if(strcasecmp(fid, "fengm2e")==0) {
    if(parNr<8) return(2);
    double delayt=0.0; if(parNr>8) delayt=p[8];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; if(!(xt>0.0)) continue;
      y[i] += (p[0]*xt-p[2]-p[4]-p[6])*exp(p[1]*xt);
      y[i] += p[2]*exp(p[3]*xt);
      y[i] += p[4]*exp(p[5]*xt);
      y[i] += p[6]*exp(p[7]*xt);
    }
    return(0);
  }
  /* Gamma variate function */
  if(strcasecmp(fid, "gammav")==0) {
    if(parNr<3) return(2);
    double delayt=0.0; if(parNr>3) delayt=p[3];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; 
      if(!(xt>0.0) || fabs(p[2])<1.0E-10) continue;
      y[i] = p[0]*pow(xt, p[1])*exp(-xt/p[2]);
    }
    return(0);
  }
  /* Exponential function */
  if(strcasecmp(fid, "1exp")==0) {
    if(parNr<2) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0; 
      y[i] += p[0]*exp(p[1]*x[i]);
    }
    return(0);
  }
  /* Sum of two exponentials */
  if(strcasecmp(fid, "2exp")==0) {
    if(parNr<4) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0; 
      y[i] += p[0]*exp(p[1]*x[i]);
      y[i] += p[2]*exp(p[3]*x[i]);
    }
    return(0);
  }
  /* Sum of three exponentials */
  if(strcasecmp(fid, "3exp")==0) {
    if(parNr<6) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0; 
      y[i] += p[0]*exp(p[1]*x[i]);
      y[i] += p[2]*exp(p[3]*x[i]);
      y[i] += p[4]*exp(p[5]*x[i]);
    }
    return(0);
  }
  /* Sum of four exponentials */
  if(strcasecmp(fid, "4exp")==0) {
    if(parNr<8) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0; 
      y[i] += p[0]*exp(p[1]*x[i]);
      y[i] += p[2]*exp(p[3]*x[i]);
      y[i] += p[4]*exp(p[5]*x[i]);
      y[i] += p[6]*exp(p[7]*x[i]);
    }
    return(0);
  }
  /* Sum of five exponentials */
  if(strcasecmp(fid, "5exp")==0) {
    if(parNr<10) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0; 
      y[i] += p[0]*exp(p[1]*x[i]);
      y[i] += p[2]*exp(p[3]*x[i]);
      y[i] += p[4]*exp(p[5]*x[i]);
      y[i] += p[6]*exp(p[7]*x[i]);
      y[i] += p[8]*exp(p[9]*x[i]);
    }
    return(0);
  }

  /* Inverted gamma cdf for plasma parent fraction */
  if(strcasecmp(fid, "ppfigamc")==0) {
    if(parNr<4) return(2);
    double delayt=0.0; if(parNr>4) delayt=p[4];
    double a=p[0];
    double b=p[1];
    double c=p[2]; if(c<0.0) return(2);
    double d=p[3]; if(d<=0.0) return(2);
    for(int i=0; i<sampleNr; i++) {
      double t=x[i]-delayt; if(t<0.0) t=0.0;
      y[i]=a*(1.0-b*igam(d, c*t));
    }
    return(0);
  }

  /* Weibull cdf with time delay */
  if(strcasecmp(fid, "weibullcdfd")==0) {
    if(parNr<4) return(2);
    double delayt=p[0];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      y[i] = p[1]*(1.0-exp(-pow(xt/p[2], p[3])));
    }
    return(0);
  }

  /* Weibull cdf with time delay and cdf derivative */
  if(strcasecmp(fid, "weibullcdfdd")==0) {
    if(parNr<5) return(2);
    double delayt=p[0];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      double a=exp(-pow(xt/p[2], p[3]));
      double b=(p[3]/p[2]) * pow(xt/p[2], p[3]-1.0) * a;
      y[i] = p[1]*(b + p[4]*(1.0-a));
    }
    return(0);
  }

  /* Surge function in form f(x)=a*x*exp(-b*x)*b^2 to give AUC=a from 0 to infinity.
     Maximum value is at 1/b. */
  if(strcasecmp(fid, "surge")==0) {
    if(parNr<2) return(2);
    double f=p[0]*(p[1]*p[1]);
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      y[i] = f*xt*exp(-p[1]*xt);
    }
    return(0);
  }

  /* Surge function in form f(x)=a*x*exp(-b*x). Maximum value is at 1/b. */
  if(strcasecmp(fid, "tradsurge")==0) {
    if(parNr<2) return(2);
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      y[i] = p[0]*xt*exp(-p[1]*xt);
    }
    return(0);
  }

  /* Surge function with recirculation, in form
     f(x)=a*(x*exp(-b*x) + (c/(b*b))*(1-(b*x+1)*exp(-b*x))). */
  if(strcasecmp(fid, "surgerecirc")==0) {
    if(parNr<2) return(2);
    double c=0.0;
    if(parNr>=3) c=p[2];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      double e=exp(-p[1]*xt);
      y[i] = xt*e + (c/(p[1]*p[1]))*(1.0 - (p[1]*xt+1.0)*e);
      y[i] *= p[0];
    }
    return(0);
  }

  /* Surge function with recirculation and delay, in form
     f(x)=a*((x-d)*exp(-b*(x-d)) + (c/(b*b))*(1-(b*(x-d)+1)*exp(-b*(x-d)))). */
  if(strcasecmp(fid, "surgerecircd")==0) {
    if(parNr<2) return(2);
    double c=0.0; if(parNr>=3) c=p[2];
    double delayt=0.0; if(parNr>=4) delayt=p[3];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      double e=exp(-p[1]*xt);
      y[i] = xt*e + (c/(p[1]*p[1]))*(1.0 - (p[1]*xt+1.0)*e);
      y[i] *= p[0];
    }
    return(0);
  }

  /* Surge function with recirculation, for plasma-to-blood ratio.
     RBC-to-plasma r(x)=a*(x*exp(-b*x) + (c/(b*b))*(1-(b*x+1)*exp(-b*x)))
     Plasma-to-blood f(x)=1/(1-HCT*(1-r(x))). */
  if(strcasecmp(fid, "p2bsrc")==0) {
    if(parNr<3) return(2);
    double c=0.0; if(parNr>=4) c=p[3];
    double HCT=p[0];
    double a=p[1];
    double b=p[2];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; if(!(xt>0.0)) {y[i]=1.0/(1.0-HCT); continue;}
      double e=exp(-b*xt);
      double r=a*(xt*e + (c/(b*b))*(1.0 - (b*xt+1.0)*e));
      y[i] = 1.0/(1.0-HCT*(1.0-r));
    }
    return(0);
  }

  /* Feng M2 function for plasma-to-blood ratio.
     RBC-to-plasma ratio r(x) using Feng M2, and then Plasma-to-blood f(x)=1/(1-HCT*(1-r(x))). */
  if(strcasecmp(fid, "p2bfm2")==0) {
    if(parNr<3) return(2);
    double HCT=p[0];
    double A1=p[1];
    double L1=p[2];
    double A2=0.0; if(parNr>3) A2=p[3];
    double L2=0.0; if(parNr>4) L2=p[4];
    double A3=0.0; if(parNr>5) A3=p[5];
    double L3=0.0; if(parNr>6) L3=p[6];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; if(!(xt>0.0)) {y[i]=1.0/(1.0-HCT); continue;}
      double r=(A1*xt - A2 - A3)*exp(-L1*xt) + A2*exp(-L2*xt) + A3*exp(-L3*xt);
      y[i] = 1.0/(1.0-HCT*(1.0-r));
    }
    return(0);
  }

  /* Surge function for late FDG AIF with time delay */
  if(strcasecmp(fid, "surgefdgaif")==0) {
    if(parNr<5) return(2);
    double delayt=p[0];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; y[i]=0.0; 
      if(!(xt>0.0)) continue;
      y[i] = p[2]*(p[3]*xt*exp(-p[4]*p[1]*xt) + exp(-p[1]*xt));
    }
    return(0);
  }

  /* Probability density function of Erlang distribution */
  if(strcasecmp(fid, "erlangpdf")==0) {
    if(parNr<3) return(2);
    double A=p[0], k=p[1]; if(!(k>=0.0)) return(2);
    int N=(int)round(p[2]); if(!(N>0)) return(2);
    unsigned long long int f=lfactorial(N-1);
    for(int i=0; i<sampleNr; i++) {
      y[i]=0.0; if(!(x[i]>=0.0)) continue;
      y[i] = A*pow(k, N)*pow(x[i], N-1)*exp(-k*x[i])/f;
    }
    return(0);
  }

  /* Rational functions */
  if(strcasecmp(fid, "ratf11")==0) {
    if(parNr<4) return(2);
    for(int i=0; i<sampleNr; i++) {
      double a=p[0]+p[2]*x[i];
      double b=p[1]+p[3]*x[i];
      y[i]=a/b;
    }
    return(0);
  }
  if(strcasecmp(fid, "ratf21")==0) {
    if(parNr<5) return(2);
    for(int i=0; i<sampleNr; i++) {
      double a=p[0]+p[2]*x[i]+p[4]*x[i]*x[i];
      double b=p[1]+p[3]*x[i];
      y[i]=a/b;
    }
    return(0);
  }
  if(strcasecmp(fid, "ratf22")==0) {
    if(parNr<6) return(2);
    for(int i=0; i<sampleNr; i++) {
      double sqrx=x[i]*x[i];
      double a=p[0]+p[2]*x[i]+p[4]*sqrx;
      double b=p[1]+p[3]*x[i]+p[5]*sqrx;
      y[i]=a/b;
    }
    return(0);
  }
  if(strcasecmp(fid, "ratf32")==0) {
    if(parNr<7) return(2);
    for(int i=0; i<sampleNr; i++) {
      double sqrx=x[i]*x[i];
      double a=p[0]+p[2]*x[i]+p[4]*sqrx+p[6]*sqrx*x[i];
      double b=p[1]+p[3]*x[i]+p[5]*sqrx;
      y[i]=a/b;
    }
    return(0);
  }
  if(strcasecmp(fid, "ratf33")==0) {
    if(parNr<8) return(2);
    for(int i=0; i<sampleNr; i++) {
      double sqrx=x[i]*x[i];
      double cubx=sqrx*x[i];
      double a=p[0]+p[2]*x[i]+p[4]*sqrx+p[6]*cubx;
      double b=p[1]+p[3]*x[i]+p[5]*sqrx+p[7]*cubx;
      y[i]=a/b;
    }
    return(0);
  }
  if(strcasecmp(fid, "p2brf")==0) {
    if(parNr<7) return(2);
    for(int i=0; i<sampleNr; i++) {
      if(x[i]<=0.0) {
        y[i]=1.0/(1.0-p[0]);
      } else {
        double sqrx=x[i]*x[i];
        double cubx=sqrx*x[i];
        double a=0.0+p[1]*x[i]+p[3]*sqrx+p[5]*cubx;
        double b=1.0+p[2]*x[i]+p[4]*sqrx+p[6]*cubx;
        y[i]=1.0/(1.0-p[0]*(1.0-(a/b)));
      }
    }
    return(0);
  }

  /* Hill functions */
  if(strcasecmp(fid, "mpfhill")==0) {
    if(parNr<3) return(2);
    for(int i=0; i<sampleNr; i++) {
      y[i]=p[0]*pow(x[i], p[1]) / (pow(x[i], p[1]) + p[2]);
    }
    return(0);
  }
  if(strcasecmp(fid, "p2bhill")==0) {
    if(parNr<4) return(2);
    for(int i=0; i<sampleNr; i++) {
      double cpr=p[1]*pow(x[i], p[2]) / (pow(x[i], p[2]) + p[3]);
      y[i]=1.0/(1.0-p[0]*(1.0-cpr));
    }
    return(0);
  }


  if(verbose>1) printf("function '%s' not supported by %s()\n", fid, __func__);
  return(10);
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate the function integral from 0 to specified x values.
    @return Returns non-zero in case of an error.
    @author Vesa Oikonen
    @todo Only MF_FENGM2, MF_FENGM2E, erlangpdf, and exponential functions are tested.
    @sa mfEvalY, mfEvalFrameY, mfEval2ndInt, mfEvalIntToInf
 */
int mfEvalInt(
  /** Function code. */
  const char *fid,
  /** Nr of function parameters. */
  const int parNr, 
  /** Array of function parameters. */
  double *p,
  /** Size of x and y arrays. */
  const int sampleNr, 
  /** Array of x values where function values are to be evaluated. */
  double *x,
  /** Pointer to allocated array of integral values where results will be written. */
  double *v,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) {
    printf("%s('%s', %d, p[], %d, x[], y[], %d)\n", __func__, fid, parNr, sampleNr, verbose);
    fflush(stdout);
  }
  if(parNr<1 || p==NULL || sampleNr<1 || x==NULL || v==NULL) return(1);

  if(strcasecmp(fid, "fengm2")==0) {
    if(parNr<6) return(2);
    double delayt=0.0; if(parNr>6) delayt=p[6];
    double a;
    double A1, A2, A3, L1, L2, L3; // lambdas are negative
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3]; A3=p[4]; L3=p[5];
    if(fabs(L1)<1.0E-20 || fabs(L2)<1.0E-20 || fabs(L3)<1.0E-20) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; v[i]=0.0; if(!(xt>0.0)) continue;
      if(L1!=0.0) {
        a=exp(L1*xt);
        v[i] += (A1+L1*(A2+A3))*(1.0-a)/(L1*L1);
        v[i] += (A1/L1)*xt*a;
      }
      if(L2!=0.0) v[i]-=(A2/L2)*(1.0-exp(L2*xt)); else v[i]+=A2*xt;
      if(L3!=0.0) v[i]-=(A3/L3)*(1.0-exp(L3*xt)); else v[i]+=A3*xt;
    }
    return(0);
  }
  if(strcasecmp(fid, "fengm2e")==0) {
    if(parNr<8) return(2);
    double delayt=0.0; if(parNr>8) delayt=p[8];
    double a;
    double A1, A2, A3, A4, L1, L2, L3, L4; // lambdas are negative
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3]; A3=p[4]; L3=p[5]; A4=p[6]; L4=p[7];
    if(fabs(L1)<1.0E-20 || fabs(L2)<1.0E-20 || fabs(L3)<1.0E-20 || fabs(L4)<1.0E-20) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; v[i]=0.0; if(!(xt>0.0)) continue;
      if(L1!=0.0) {
        a=exp(L1*xt);
        v[i] += (A1+L1*(A2+A3+A4))*(1.0-a)/(L1*L1);
        v[i] += (A1/L1)*xt*a;
      }
      if(L2!=0.0) v[i]-=(A2/L2)*(1.0-exp(L2*xt)); else v[i]+=A2*xt;
      if(L3!=0.0) v[i]-=(A3/L3)*(1.0-exp(L3*xt)); else v[i]+=A3*xt;
      if(L4!=0.0) v[i]-=(A4/L4)*(1.0-exp(L4*xt)); else v[i]+=A4*xt;
    }
    return(0);
  }
  if(strcasecmp(fid, "fengm2s")==0) {
    if(parNr<4) return(2);
    double delayt=0.0; if(parNr>4) delayt=p[4];
    double a;
    double A1, A2, L1, L2; // lambdas are negative
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3];
    if(fabs(L1)<1.0E-20 || fabs(L2)<1.0E-20) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; v[i]=0.0; if(!(xt>0.0)) continue;
      if(L1!=0.0) {
        a=exp(L1*xt);
        v[i] += (A1+L1*A2)*(1.0-a)/(L1*L1);
        v[i] += (A1/L1)*xt*a;
      }
      if(L2!=0.0) v[i]-=(A2/L2)*(1.0-exp(L2*xt)); else v[i]+=A2*xt;
    }
    return(0);
  }
  /* Gamma variate function when p[1]==1 */
  if(strcasecmp(fid, "gammav")==0 && parNr>=3 && fabs(p[1]-1.0)<1.0E-10) {
    if(parNr<3) return(2);
    double delayt=0.0; if(parNr>3) delayt=p[3];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; v[i]=0.0; 
      if(!(xt>0.0) || fabs(p[2])<1.0E-10) continue;
      v[i] = p[0]*p[2]*(p[2] - (p[2]+xt)*exp(-xt/p[2]));
    }
    return(0);
  }

  /* Surge function in form f(x)=a*x*exp(-b*x)*b^2, to give AUC=a from 0 to infinity. */
  if(strcasecmp(fid, "surge")==0) {
    if(parNr<2) return(2);
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; v[i]=0.0; 
      if(!(xt>0.0)) continue;
      v[i] = p[0]*(1.0 - (p[1]*xt + 1.0)*exp(-p[1]*xt));
    }
    return(0);
  }

  /* Surge function in form f(x)=a*x*exp(-b*x). Maximum value is at 1/b. */
  if(strcasecmp(fid, "tradsurge")==0) {
    if(parNr<2) return(2);
    double f=p[0]/(p[1]*p[1]);
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]; v[i]=0.0; 
      if(!(xt>0.0)) continue;
      v[i] = f*(1.0 - (p[1]*xt + 1.0)*exp(-p[1]*xt));
    }
    return(0);
  }

  /* Surge function for late FDG AIF with time delay */
  if(strcasecmp(fid, "surgefdgaif")==0) {
    if(parNr<5) return(2);
    double delayt=p[0];
    for(int i=0; i<sampleNr; i++) {
      double xt=x[i]-delayt; v[i]=0.0; 
      if(!(xt>0.0)) continue;
      v[i] += p[3]*(1.0-(p[4]*p[1]*xt + 1.0)*exp(-p[4]*p[1]*xt))/(p[1]*p[1]*p[4]*p[4]);
      v[i] += (1.0-exp(-p[1]*xt))/p[1];
      v[i] *= p[2];
    }
    return(0);
  }

  /* Probability density function of Erlang distribution */
  if(strcasecmp(fid, "erlangpdf")==0 && p[2]<20.5) { // Supports only N=0,1,2,...,20
    if(parNr<3) return(2);
    double A=p[0], k=p[1]; if(!(k>=0.0)) return(2);
    int N=(int)round(p[2]); if(!(N>=0) || N>20) return(2);
    if(N==0)
      for(int i=0; i<sampleNr; i++) {
        if(x[i]>=0.0) v[i]=A; else v[i]=0.0;
      }
    else if(N==1)
      for(int i=0; i<sampleNr; i++) {
        if(x[i]>=0.0) v[i]=A*(1.0-exp(-k*x[i])); else v[i]=0.0;
      }
    else if(N==2)
      for(int i=0; i<sampleNr; i++) {
        if(x[i]>=0.0) v[i]=A*(1.0 - exp(-k*x[i]) - k*x[i]*exp(-k*x[i])); else v[i]=0.0;
      }
    else {
      for(int i=0; i<sampleNr; i++) {
        if(!(x[i]>=0.0)) {v[i]=0.0; continue;}
        double s=1.0+k*x[i];
        for(int j=2; j<N; j++) s+=pow(k*x[i], j)/lfactorial(j);
        v[i]=A*(1.0-s*exp(-k*x[i]));
      }
    }

    return(0);
  }

  /* Exponential functions */
  if(strcasecmp(fid, "1exp")==0) {
    if(parNr<2) return(2);
    double A=p[0], k=p[1];
    if(fabs(k)<1.0E-20) {
      for(int i=0; i<sampleNr; i++) v[i]=A*x[i];
    } else {
      for(int i=0; i<sampleNr; i++) v[i]=(A/k)*(exp(k*x[i]) - 1.0);
    }
    return(0);
  }
  if(strcasecmp(fid, "2exp")==0) {
    if(parNr<4) return(2);
    for(int i=0; i<sampleNr; i++) v[i]=0.0;
    for(int n=0; n<=1; n++) {
      double A=p[n*2], k=p[n*2+1];
      if(fabs(k)<1.0E-20) {
        for(int i=0; i<sampleNr; i++) v[i]+=A*x[i];
      } else {
        for(int i=0; i<sampleNr; i++) v[i]+=(A/k)*(exp(k*x[i]) - 1.0);
      }
    }
    return(0);
  }
  if(strcasecmp(fid, "3exp")==0) {
    if(parNr<6) return(2);
    for(int i=0; i<sampleNr; i++) v[i]=0.0;
    for(int n=0; n<=2; n++) {
      double A=p[n*2], k=p[n*2+1];
      if(fabs(k)<1.0E-20) {
        for(int i=0; i<sampleNr; i++) v[i]+=A*x[i];
      } else {
        for(int i=0; i<sampleNr; i++) v[i]+=(A/k)*(exp(k*x[i]) - 1.0);
      }
    }
    return(0);
  }
  if(strcasecmp(fid, "4exp")==0) {
    if(parNr<8) return(2);
    for(int i=0; i<sampleNr; i++) v[i]=0.0;
    for(int n=0; n<=3; n++) {
      double A=p[n*2], k=p[n*2+1];
      if(fabs(k)<1.0E-20) {
        for(int i=0; i<sampleNr; i++) v[i]+=A*x[i];
      } else {
        for(int i=0; i<sampleNr; i++) v[i]+=(A/k)*(exp(k*x[i]) - 1.0);
      }
    }
    return(0);
  }
  if(strcasecmp(fid, "5exp")==0) {
    if(parNr<10) return(2);
    for(int i=0; i<sampleNr; i++) v[i]=0.0;
    for(int n=0; n<=4; n++) {
      double A=p[n*2], k=p[n*2+1];
      if(fabs(k)<1.0E-20) {
        for(int i=0; i<sampleNr; i++) v[i]+=A*x[i];
      } else {
        for(int i=0; i<sampleNr; i++) v[i]+=(A/k)*(exp(k*x[i]) - 1.0);
      }
    }
    return(0);
  }

  if(verbose>1) printf("function '%s' not supported by %s()\n", fid, __func__);
  return(10);
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate the 2nd integral of function from 0 to specified x values.
    @return Returns non-zero in case of an error.
    @todo Only MF_FENGM2 and MF_FENGM2E are functional and tested.
    @author Vesa Oikonen
    @sa mfEvalInt, mfEvalY, mfEvalFrameY
 */
int mfEval2ndInt(
  /** Function code. */
  const char *fid,
  /** Nr of function parameters. */
  const int parNr, 
  /** Array of function parameters. */
  double *p,
  /** Size of x and y arrays. */
  const int sampleNr, 
  /** Array of x values where function values are to be evaluated. */
  double *x,
  /** Pointer to allocated array of 2nd integral values where results will be written. */
  double *v,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  const int verbose
) {
  if(verbose>0) {
    printf("%s('%s', %d, p[], %d, x[], y[], %d)\n", __func__, fid, parNr, sampleNr, verbose);
    fflush(stdout);
  }
  if(parNr<1 || p==NULL || sampleNr<1 || x==NULL || v==NULL) return(1);

  int i;
  double xt, delayt=0.0;

  if(strcasecmp(fid, "fengm2")==0) {
    if(parNr<6) return(2);
    if(parNr>6) delayt=p[6];
    double A1, A2, A3, L1, L2, L3;
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3]; A3=p[4]; L3=p[5];
    if(fabs(L1)<1.0E-06 || fabs(L2)<1.0E-06 || fabs(L3)<1.0E-06) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(i=0; i<sampleNr; i++) {
      xt=x[i]-delayt; v[i]=0.0; if(!(xt>0.0)) continue;
      if(L1!=0.0) {
        v[i] += (((A1*xt-A2-A3)*L1 - 2.0*A1) * exp(L1*xt) + 2.0*A1)/(L1*L1*L1);
        v[i] += (A1*xt+A2+A3)/(L1*L1);
        v[i] += (A2*xt+A3*xt)/L1;
      }
      if(L2!=0.0) v[i] -= (A2/(L2*L2))*(1.0 - exp(L2*xt)) + A2*xt/L2;
      if(L3!=0.0) v[i] -= (A3/(L3*L3))*(1.0 - exp(L3*xt)) + A3*xt/L3;
    }
    return(0);
  }
  if(strcasecmp(fid, "fengm2e")==0) {
    if(parNr<8) return(2);
    if(parNr>8) delayt=p[8];
    double A1, A2, A3, A4, L1, L2, L3, L4;
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3]; A3=p[4]; L3=p[5]; A4=p[6]; L4=p[7];
    if(fabs(L1)<1.0E-15 || fabs(L2)<1.0E-15 || fabs(L3)<1.0E-15 || fabs(L4)<1.0E-15) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(i=0; i<sampleNr; i++) {
      xt=x[i]-delayt; v[i]=0.0; if(!(xt>0.0)) continue;
      v[i] = A4*L1*L1*L1*L2*L2*L3*L3*L4*xt +
        A4*L1*L1*L1*L2*L2*L3*L3*(1-exp(L4*xt)) +
        L4*L4*(A3*L1*L1*L1*L2*L2*L3*xt + A3*L1*L1*L1*L2*L2*(1.0-exp(L3*xt)) +
          L3*L3*(A2*L1*L1*L1*L2*xt + A2*L1*L1*L1*(1.0-exp(L2*xt)) -
            L2*L2*((A2+A3+A4)*xt*L1*L1 + (A1*xt+A2+A3+A4)*L1 + 
                   ((A1*xt-A2-A3-A4)*L1-2.0*A1)*exp(L1*xt) + 2.0*A1)));
      v[i] /= -(L1*L1*L1*L2*L2*L3*L3*L4*L4);
    }
    return(0);
  }
  /* Gamma variate function when p[1]==1 */
  if(strcasecmp(fid, "gammav")==0 && parNr>=3 && fabs(p[1]-1.0)<1.0E-10) {
    if(parNr<3) return(2);
    if(parNr>3) delayt=p[3];
    for(i=0; i<sampleNr; i++) {
      xt=x[i]-delayt; v[i]=0.0; 
      if(!(xt>0.0) || fabs(p[2])<1.0E-10) continue;
      v[i] = p[0]*p[2]*p[2]*((p[2]+p[2]+xt)*exp(-xt/p[2]) - p[2] - p[2] + xt);
    }
    return(0);
  }

  if(verbose>1) printf("function '%s' not supported by %s()\n", fid, __func__);
  return(10);
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate the PET time frame mean of function values.
    @return Returns non-zero in case of an error.
    @author Vesa Oikonen
    @todo Only MF_FENGM2 and MF_FENGM2E are functional and tested.
 */
int mfEvalFrameY(
  /** Function code. */
  const char *fid,
  /** Nr of function parameters. */
  const int parNr, 
  /** Array of function parameters. */
  double *p,
  /** Size of x and y arrays. */
  const int sampleNr, 
  /** Array of frame start times. */
  double *x1,
  /** Array of frame end times. */
  double *x2,
  /** Pointer to allocated array of y values where function values will be written. */
  double *y,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) {
    printf("%s('%s', %d, p[], %d, x1[], x2[], y[], %d)\n", __func__, fid, parNr, sampleNr, verbose);
    fflush(stdout);
  }
  if(parNr<1 || p==NULL || sampleNr<1 || x1==NULL || x2==NULL || y==NULL) return(1);

  int i, ret;
  double xt1, xt2, v, fd, delayt=0.0;

  if(strcasecmp(fid, "fengm2")==0) {
    if(parNr<6) return(2);
    if(parNr>6) delayt=p[6];
    double A1, A2, A3, L1, L2, L3;
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3]; A3=p[4]; L3=p[5];
    if(fabs(L1)<1.0E-12 || fabs(L2)<1.0E-12 || fabs(L3)<1.0E-12) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(i=0; i<sampleNr; i++) {
      xt1=x1[i]-delayt; xt2=x2[i]-delayt; y[i]=0.0; 
      if(xt1>xt2) {v=xt1; xt1=xt2; xt2=v;}
      if(!(xt2>0.0)) continue;
      fd=xt2-xt1; // requested frame duration
      if(fd<1.0E-025) {
        // if frame duration is about zero, then use the method for that
        ret=mfEvalY(fid, parNr, p, 1, x2+i, y+i, verbose);
        if(ret!=0) return(ret);
      }
      if(!(xt1>0.0)) xt1=0.0; // integral start time must not be <0
      /* Calculate integral between xt1 and xt2 */
      y[i] = A3*L1*L1*L2*(exp(L3*xt1) - exp(L3*xt2)) +
          L3*( A2*L1*L1*(exp(L2*xt1) - exp(L2*xt2)) +
            L2*(((A1*xt1-A2-A3)*L1-A1)*exp(L1*xt1) -  
                ((A1*xt2-A2-A3)*L1-A1)*exp(L1*xt2)
               )
          );
      y[i] /= -(L1*L1*L2*L3);
      /* Divide integral with the original frame duration */
      y[i]/=fd; 
    }
    return(0);
  }
  if(strcasecmp(fid, "fengm2e")==0) {
    if(parNr<8) return(2);
    if(parNr>8) delayt=p[8];
    double A1, A2, A3, A4, L1, L2, L3, L4;
    A1=p[0]; L1=p[1]; A2=p[2]; L2=p[3]; A3=p[4]; L3=p[5]; A4=p[6]; L4=p[7];
    if(fabs(L1)<1.0E-20 || fabs(L2)<1.0E-20 || fabs(L3)<1.0E-20 || fabs(L4)<1.0E-20) {
      if(verbose>1) printf("too small L parameter(s)\n");
      return(3);
    }
    for(i=0; i<sampleNr; i++) {
      xt1=x1[i]-delayt; xt2=x2[i]-delayt; y[i]=0.0; 
      if(xt1>xt2) {v=xt1; xt1=xt2; xt2=v;}
      if(!(xt2>0.0)) continue;
      fd=xt2-xt1; // requested frame duration
      if(fd<1.0E-025) {
        // if frame duration is about zero, then use the method for that
        ret=mfEvalY(fid, parNr, p, 1, x2+i, y+i, verbose);
        if(ret!=0) return(ret);
      }
      if(!(xt1>0.0)) xt1=0.0; // integral start time must not be <0
      /* Calculate integral between xt1 and xt2 */
      y[i] = A4*L1*L1*L2*L3*(exp(L4*xt1) - exp(L4*xt2)) +
        L4*( A3*L1*L1*L2*(exp(L3*xt1) - exp(L3*xt2)) +
          L3*( A2*L1*L1*(exp(L2*xt1) - exp(L2*xt2)) +
            L2*(((A1*xt1-A2-A3-A4)*L1-A1)*exp(L1*xt1) -  
                ((A1*xt2-A2-A3-A4)*L1-A1)*exp(L1*xt2)
               )));
      y[i] /= -(L1*L1*L2*L3*L4);
      /* Divide integral with the original frame duration */
      y[i]/=fd; 
    }
    return(0);
  }
  /* Gamma variate function when p[1]==1 */
  if(strcasecmp(fid, "gammav")==0 && parNr>=3 && fabs(p[1]-1.0)<1.0E-10) {
    if(parNr<3) return(2);
    if(parNr>3) delayt=p[3];
    for(i=0; i<sampleNr; i++) {
      xt1=x1[i]-delayt; xt2=x2[i]-delayt; y[i]=0.0; 
      if(fabs(p[2])<1.0E-10) continue;
      if(xt1>xt2) {v=xt1; xt1=xt2; xt2=v;}
      if(!(xt2>0.0)) continue;
      fd=xt2-xt1; // requested frame duration
      if(fd<1.0E-025) {
        // if frame duration is about zero, then use the method for that
        ret=mfEvalY(fid, parNr, p, 1, x2+i, y+i, verbose);
        if(ret!=0) return(ret);
      }
      if(!(xt1>0.0)) xt1=0.0; // integral start time must not be <0
      /* Calculate integral between xt1 and xt2 */
      y[i] = p[0]*p[2]*( (p[2]+xt1)*exp(-xt1/p[2]) - (p[2]+xt2)*exp(-xt2/p[2]) );
      /* Divide integral with the original frame duration */
      y[i]/=fd; 
    }
    return(0);
  }

  if(verbose>1) printf("function '%s' not supported by %s()\n", fid, __func__);
  return(10);
}
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate the function integral from specified x to infinity.
    @return Returns non-zero in case of an error.
    @author Vesa Oikonen
    @todo Only exponential functions are functional and tested.
    @sa mfEvalInt, mfEvalY
 */
int mfEvalIntToInf(
  /** Function code. */
  const char *fid,
  /** Nr of function parameters. */
  const int parNr, 
  /** Array of function parameters. */
  double *p,
  /** X value; must be 0 or larger. */
  double x,
  /** Pointer for result integral. */
  double *v,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) {
    printf("%s('%s', %d, p[], %g, y, %d)\n", __func__, fid, parNr, x, verbose);
    fflush(stdout);
  }
  if(parNr<1 || p==NULL || v==NULL) return(1);
  if(!(x>=0.0)) return(2);
  *v=0.0;

  /* Exponential functions */
  if(strcasecmp(fid, "1exp")==0) {
    if(parNr<2) return(2);
    double A=p[0], k=p[1];
    *v=(-A/k)*exp(k*x);
    return(0);
  }
  if(strcasecmp(fid, "2exp")==0) {
    if(parNr<4) return(2);
    for(int n=0; n<=1; n++) {
      double A=p[n*2], k=p[n*2+1];
      *v+=(-A/k)*exp(k*x);
    }
    return(0);
  }
  if(strcasecmp(fid, "3exp")==0) {
    if(parNr<6) return(2);
    for(int n=0; n<=2; n++) {
      double A=p[n*2], k=p[n*2+1];
      *v+=(-A/k)*exp(k*x);
    }
    return(0);
  }
  if(strcasecmp(fid, "4exp")==0) {
    if(parNr<8) return(2);
    for(int n=0; n<=3; n++) {
      double A=p[n*2], k=p[n*2+1];
      *v+=(-A/k)*exp(k*x);
    }
    return(0);
  }
  if(strcasecmp(fid, "5exp")==0) {
    if(parNr<10) return(2);
    for(int n=0; n<=4; n++) {
      double A=p[n*2], k=p[n*2+1];
      *v+=(-A/k)*exp(k*x);
    }
    return(0);
  }



  if(verbose>1) printf("function '%s' not supported by %s()\n", fid, __func__);
  return(10);
}
/*****************************************************************************/

/*****************************************************************************/
