/** @file ift.c
 *  @brief IFT struct processing.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the IFT struct before any use.
    @sa iftFree, iftDuplicate
    @author Vesa Oikonen
 */
void iftInit(
  /** Pointer to IFT */
  IFT *ift
) {
  if(ift==NULL) return;
  ift->_memNr=ift->keyNr=0; ift->item=NULL; ift->type=0;
  ift->space_before_eq=ift->space_after_eq=1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for IFT data. All contents are destroyed.
    @pre Before first use initialize the IFT struct with iftInit().
    @sa iftInit
    @author Vesa Oikonen
 */
void iftFree(
  /** Pointer to IFT */
  IFT *ift
) {
  int i;

  if(ift==NULL) return;
  for(i=0; i<ift->_memNr; i++) {
    free(ift->item[i].key);
    free(ift->item[i].value);
  }
  free(ift->item); ift->item=NULL; ift->_memNr=ift->keyNr=0;
  iftInit(ift);
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified key and its value to the IFT.

    Either key or value can be empty, but not both of them.

    @sa iftInit, iftFree, iftDelete, iftDeleteKey, iftPutDouble, iftCopyItems
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT struct with iftInit().
    @author Vesa Oikonen
 */
int iftPut(
  /** Pointer to initiated IFT; previous contents are not changed */
  IFT *ift,
  /** Key string; can be empty ("" or NULL) */
  const char *key,
  /** Value string; can be empty ("" or NULL) */
  const char *value,
  /** Is this comment line, or line that is commented out?
   *  - 0: not a comment line
   *  - 1: line will be started with '#' */
  char comment,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL) return TPCERROR_FAIL;
  if((key==NULL || strlen(key)<1) && (value==NULL || strlen(value)<1)) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>10) {
    printf("iftPut(ift, ");
    if(key!=NULL) printf("\"%s\", ", key); else printf("NULL, ");
    if(value!=NULL) printf("\"%s\", ", value); else printf("NULL, ");
    printf("%d)\n", comment);
    fflush(stdout);
  }

  /* If necessary, allocate more memory for items */
  if(ift->_memNr<=ift->keyNr) {
    int i, add_nr=20;
    ift->item=(IFT_ITEM*)realloc(ift->item, (ift->_memNr+add_nr)*sizeof(IFT_ITEM));
    if(ift->item==NULL) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OUT_OF_MEMORY);
      return TPCERROR_OUT_OF_MEMORY;
    }
    for(i=ift->_memNr; i<ift->_memNr+add_nr; i++) {
      ift->item[i].comment=(char)0;
      ift->item[i].sw=(short int)0;
      ift->item[i].key=NULL;
      ift->item[i].value=NULL;
    }
    ift->_memNr+=add_nr;
  }
 
  /* Set the contents */
  /* type */
  if(comment!=0) ift->item[ift->keyNr].comment=(char)1;
  /* key (do not put NULL because it would segfault in std functions) */
  if(key!=NULL) ift->item[ift->keyNr].key=strdup(key);
  else ift->item[ift->keyNr].key=strdup("");
  /* value (do not put NULL because it would segfault in std functions) */
  if(value!=NULL) ift->item[ift->keyNr].value=strdup(value);
  else ift->item[ift->keyNr].value=strdup("");

  ift->keyNr++;
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified key and its floating point (double) value to the IFT.
    @sa iftInit, iftPut, iftPutFromString, iftPutInt, iftGetDouble
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT structure with iftInit().
 */
int iftPutDouble(
  /** Pointer to initiated IFT; previous contents are not changed */
  IFT *ift,
  /** Key string; can be empty ("" or NULL). */
  const char *key,
  /** Value as double. */
  const double value,
  /** Is this comment line, or line that is commented out?
      - 0: not a comment line
      - 1: line will be started with '#' */
  char comment,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  char dstr[128];
  sprintf(dstr, "%g", value);
  return(iftPut(ift, key, dstr, comment, status));
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified key and its integer (signed) value to the IFT.
    @sa iftInit, iftPut, iftPutFromString, iftPutUint, iftPutDouble, iftGetInt
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT structure with iftInit().
 */
int iftPutInt(
  /** Pointer to initiated IFT; previous contents are not changed */
  IFT *ift,
  /** Key string; can be empty ("" or NULL). */
  const char *key,
  /** Value as int. */
  const int value,
  /** Is this comment line, or line that is commented out?
      - 0: not a comment line
      - 1: line will be started with '#' */
  char comment,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  char dstr[128];
  sprintf(dstr, "%d", value);
  return(iftPut(ift, key, dstr, comment, status));
}
/*****************************************************************************/

/*****************************************************************************/
/** Add specified key and its integer (unsigned) value to the IFT.
    @sa iftInit, iftPut, iftPutFromString, iftPutInt, iftPutDouble, iftGetUint
    @return tpcerror (TPCERROR_OK when successful).
    @pre Before first use initialize the IFT structure with iftInit().
 */
int iftPutUInt(
  /** Pointer to initiated IFT; previous contents are not changed */
  IFT *ift,
  /** Key string; can be empty ("" or NULL). */
  const char *key,
  /** Value as int. */
  const unsigned int value,
  /** Is this comment line, or line that is commented out?
      - 0: not a comment line
      - 1: line will be started with '#' */
  char comment,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  char dstr[128];
  sprintf(dstr, "%u", value);
  return(iftPut(ift, key, dstr, comment, status));
}
/*****************************************************************************/

/*****************************************************************************/
/** Remove the specified item from IFT struct.
    @sa iftFree, iftPut, iftDeleteKey, iftFindKey, iftDeleteDuplicateKeys
    @return tpcerror, TPCERROR_OK (0) when successful.
    @author Vesa Oikonen
 */
int iftDelete(
  /** Pointer to existing IFT */
  IFT *ift,
  /** Index [0..keyNr-1] of key and value */
  int index
) {
  if(ift==NULL) return TPCERROR_FAIL;
  if(index<0 || index>=ift->keyNr) return TPCERROR_NO_KEY;

  free(ift->item[index].key); ift->item[index].key=NULL;
  free(ift->item[index].value); ift->item[index].value=NULL;
  ift->item[index].key=ift->item[index].value=NULL;
  for(int i=index+1; i<ift->keyNr; i++) {
    ift->item[i-1].comment=ift->item[i].comment;
    ift->item[i-1].sw=ift->item[i].sw;
    ift->item[i-1].key=ift->item[i].key;
    ift->item[i-1].value=ift->item[i].value;
    ift->item[i].key=ift->item[i].value=NULL;
  }
  ift->keyNr--;
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Make a copy (duplicate) of an IFT struct.
    @sa iftInit, iftDeleteKey, iftDeleteDuplicateKeys
    @return tpcerror, TPCERROR_OK (0) when successful.
    @author Vesa Oikonen
 */
int iftDuplicate(
  /** Pointer to source IFT struct to be copied */
  IFT *ift1,
  /** Pointer to initiated target IFT struct; any previous contents are deleted. */
  IFT *ift2
) {
  if(ift1==NULL || ift2==NULL) return TPCERROR_FAIL;
  /* Empty the new IFT */
  iftFree(ift2);

  /* Copy the contents */
  ift2->type=ift1->type;
  ift2->space_before_eq=ift1->space_before_eq;
  ift2->space_after_eq=ift1->space_after_eq;
  int ret;
  for(int i=0; i<ift1->keyNr; i++) {
    ret=iftPut(ift2, ift1->item[i].key, ift1->item[i].value, ift1->item[i].comment, NULL);
    if(ret!=TPCERROR_OK) {iftFree(ift2); return(ret);} 
    ift2->item[i].sw=ift1->item[i].sw;
  }
  // keyNr was set by iftPut()
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Replace the value of specified IFT item.

    @sa iftReplaceKey, iftDelete, iftDeleteKey, iftPut
    @return tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int iftReplaceValue(
  /** Pointer to IFT containing at least the item to be edited. */
  IFT *ift,
  /** IFT item index [0..keyNr-1] to be edited. */
  int i,
  /** New value string; can be empty ("" or NULL) */
  const char *value,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>10) {
    printf("iftReplaceValue(ift, %d", i);
    if(value!=NULL) printf("\"%s\")\n", value); else printf("NULL)\n");
    fflush(stdout);
  }
  if(i<0 || i>=ift->keyNr) return TPCERROR_FAIL;

  /* Delete the previous value */
  free(ift->item[i].value);
  /* Set the new value;
     do not put NULL because it would segfault in std functions */
  if(value!=NULL) ift->item[i].value=strdup(value);
  else ift->item[i].value=strdup("");

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Replace the key name of a specified IFT item.

    @sa iftReplaceValue, iftDelete, iftDeleteKey, iftPut
    @return tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int iftReplaceKey(
  /** Pointer to IFT containing at least the item to be edited. */
  IFT *ift,
  /** IFT item index [0..keyNr-1] to be edited. */
  int i,
  /** New key string; can be empty ("" or NULL) */
  const char *key,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>10) {
    printf("iftReplaceKey(ift, %d", i);
    if(key!=NULL) printf("\"%s\")\n", key); else printf("NULL)\n");
    fflush(stdout);
  }
  if(i<0 || i>=ift->keyNr) return TPCERROR_FAIL;

  /* Delete the previous value */
  free(ift->item[i].key);
  /* Set the new value;
     do not put NULL because it would segfault in std functions */
  if(key!=NULL) ift->item[i].key=strdup(key);
  else ift->item[i].key=strdup("");

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find and delete items that have similar key names. 

    The first occurrence of the key is kept.
    Search is case-insensitive, but otherwise key name match must be exact. 

    @sa iftDeleteKey, iftFindKey, iftCopyItems
    @return tpcerror (TPCERROR_OK when successful).
 */
int iftDeleteDuplicateKeys(
  /** Pointer to IFT. */
  IFT *ift,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift==NULL) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>0) printf("%s()\n", __func__);
  if(ift->keyNr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }

  /* Go through the (remaining) items */
  int i=0;
  while(i<ift->keyNr-1) {
    int j=i+1;
    while(j<ift->keyNr) {
      int k=iftFindKey(ift, ift->item[i].key, j);
      if(k>=0) iftDelete(ift, k); else j++;
    }
    i++;
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy items from one IFT struct into another. 

    Duplicates, empty source, or no copied items are not considered as errors.

    @sa iftDeleteDuplicateKeys, iftPut
    @return tpcerror (TPCERROR_OK when successful).
 */
int iftCopyItems(
  /** Pointer to target IFT. */
  IFT *ift1,
  /** Pointer to source IFT. */
  IFT *ift2,
  /** Specifies whether items without key name are copied (0) or not (1). */
  int is_key_required,
  /** Specifies whether items without value are copied (0) or not (1). */
  int is_value_required,
  /** Specifies whether comment items are copied or not.
      - 0 = comment lines are not copied,
      - 1 = also comment lines are copied,
      - 2 = only comment lines are copied. */
  int is_comment_accepted,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
  if(ift1==NULL) return TPCERROR_FAIL;
  int verbose=0; if(status!=NULL) verbose=status->verbose-1;
  if(verbose>0) 
    printf("%s(*ift1, *ift2, %d, %d, %d, status)\n", __func__, 
           is_key_required, is_value_required, is_comment_accepted);
  if(ift2==NULL || ift2->keyNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    return TPCERROR_OK;
  }

  /* Copy the appropriate items */
  int n=0;
  for(int i=0; i<ift2->keyNr; i++) {
    if(is_key_required && (ift2->item[i].key==NULL || strlen(ift2->item[i].key)<1)) continue;
    if(is_value_required && (ift2->item[i].value==NULL || strlen(ift2->item[i].value)<1)) continue;
    if(ift2->item[i].comment!=0 && is_comment_accepted==0) continue;
    if(ift2->item[i].comment==0 && is_comment_accepted==2) continue;
    int ret=iftPut(ift1, ift2->item[i].key, ift2->item[i].value, ift2->item[i].comment, status);
    if(ret!=TPCERROR_OK) return(status->error);
    n++;
  }
  if(verbose>1) printf("  %d item(s) copied.\n", n);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
