/** @file iftfind.c
 *  @brief Find keys and values in IFT.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/

/*****************************************************************************/
/** Find the IFT item with specified key. 

    Search is case-insensitive, but otherwise match must be exact. 

    @sa iftInit, iftRead, iftFindPair, iftSearchKey, iftSearchValue,
        iftFindNrOfKeys, iftGetDoubleWithUnit
    @return item index, or <0 if not found.
    @author Vesa Oikonen
 */
int iftFindKey(
  /** Pointer to IFT. */
  IFT *ift,
  /** Key to be searched for. */
  const char *key,
  /** IFT item index [0..keyNr-1] from which search is started. */
  int start_index
) {
  if(ift==NULL || key==NULL) return -2;
  if(start_index<0 || start_index>=ift->keyNr) return -2;
  for(int i=start_index; i<ift->keyNr; i++) if(ift->item[i].key!=NULL)
    if(strcasecmp(ift->item[i].key, key)==0) return i;
  return -1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the IFT item with specified key and value.

    Search is case-insensitive, but otherwise match must be exact. 

    @sa iftSearchKey, iftSearchValue, iftFindNrOfKeys, iftDeleteKey
    @return item index, or <0 if not found.
    @author Vesa Oikonen
 */
int iftFindPair(
  /** Pointer to IFT. */
  IFT *ift,
  /** Key to be searched for. */
  const char *key,
  /** Value to be searched for. */
  const char *value,
  /** IFT item index [0..keyNr-1] from which search is started. */
  int start_index
) {
  if(ift==NULL || key==NULL || value==NULL) return -2;
  if(start_index<0 || start_index>=ift->keyNr) return -2;
  for(int i=start_index; i<ift->keyNr; i++) {
    if(ift->item[i].key==NULL || ift->item[i].value==NULL) continue;
    if(strcasecmp(ift->item[i].key, key)!=0) continue;
    if(strcasecmp(ift->item[i].value, value)!=0) continue;
    return i;
  }
  return -1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the IFT item where key contains the given search string.

    Search is case-insensitive. 

    @sa iftFindKey, iftSearchValue, iftFindNrOfKeys, iftDeleteKey
    @return item index, or <0 if not found.
    @author Vesa Oikonen
 */
int iftSearchKey(
  /** Pointer to IFT. */
  IFT *ift,
  /** String to be searched for in keys. */
  const char *s,
  /** IFT item index [0..keyNr-1] from which search is started. */
  int start_index
) {
  if(ift==NULL || s==NULL) return -2;
  if(start_index<0 || start_index>=ift->keyNr) return -2;
  for(int i=start_index; i<ift->keyNr; i++) {
    if(*s!='\0') {if(strcasestr(ift->item[i].key, s)!=NULL) return i; else continue;}
    // s is empty
    if(ift->item[i].key==NULL || ift->item[i].key[0]=='\0') return i;
  }
  return -1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the IFT item where value contains the given search string.

    Search is case-insensitive. 

    @sa iftSearchKey, iftFindPair, iftFindKey, iftDeleteKey
    @return item index, or <0 if not found.
    @author Vesa Oikonen
 */
int iftSearchValue(
  /** Pointer to IFT. */
  IFT *ift,
  /** String to be searched for in values. */
  const char *s,
  /** IFT item index [0..keyNr-1] from which search is started. */
  int start_index
) {
  if(ift==NULL || s==NULL) return -2;
  if(start_index<0 || start_index>=ift->keyNr) return -2;
  for(int i=start_index; i<ift->keyNr; i++) {
    if(*s!='\0') {if(strcasestr(ift->item[i].value, s)!=NULL) return i; else continue;}
    // s is empty
    if(ift->item[i].value==NULL || ift->item[i].value[0]=='\0') return i;
  }
  return -1;
}
/*****************************************************************************/

/*****************************************************************************/
/** Find the nr of exact occurrences of the specified key in the IFT.

    Key is case insensitive.

    @sa iftSearchKey, iftFindPair, iftFindKey, iftDeleteKey
    @return nr of occurrences of the key.
    @author Vesa Oikonen
 */
int iftFindNrOfKeys(
  /** Pointer to existing IFT. */
  IFT *ift,
  /** Pointer to the key string; if empty or NULL, then the nr of empty keys is returned. */
  const char *key
) {
  if(ift==NULL) return(0);
  int i, found_nr=0;
  if(key==NULL || strlen(key)<1) {
    for(i=0; i<ift->keyNr; i++)
      if(ift->item[i].key==NULL || strlen(ift->item[i].key)<1) found_nr++;
    return found_nr;
  }
  for(i=0; i<ift->keyNr; i++) if(ift->item[i].key!=NULL)
    if(strcasecmp(ift->item[i].key, key)==0) found_nr++;
  return found_nr;
}
/*****************************************************************************/

/*****************************************************************************/
/** Delete all items which have the the specified key in the IFT.

    Key is case insensitive.

    @sa iftSearchKey, iftFindPair, iftFindKey, iftDelete, iftDeleteDuplicateKeys
    @author Vesa Oikonen
 */
void iftDeleteKey(
  /** Pointer to existing IFT. */
  IFT *ift,
  /** Pointer to the key string. */
  const char *key
) {
  if(ift==NULL || key==NULL || *key=='\0') return;
  int i=0, start=0;
  while(1) {
    i=iftFindKey(ift, key, start); if(i<0) break;
    iftDelete(ift, i); if(i>start) start=i;
  }
  return;
}
/*****************************************************************************/

/*****************************************************************************/
/** Finds the specified key string from IFT structure, and reads the corresponding value as double.
    @return Returns the index of key/value, -1 if key or value was not found,
    and <-1 in case of an error.
    @sa iftGetDoubleWithUnit, iftGetDouble, iftGetInt, iftFindKey
 */
int iftGetDoubleValue(
  /** Pointer to existing IFT. */
  IFT *ift,
  /** Pointer to the key string; search is case-insensitive. */
  const char *key,
  /** Index [0..keyNr-1] from which the search is started. */
  int index,
  /** Pointer to double variable where value is written; NaN is written in case of an error. */
  double *v
) {
  if(v!=NULL) *v=nan(""); else return(-10);
  int li=iftFindKey(ift, key, index); if(li<0) return(li);
  if(atofCheck(ift->item[li].value, v)!=0) return(-2);
  return(li);
}
/*****************************************************************************/

/*****************************************************************************/
