/** @file imagecomp.c
    @brief Comparison of IMG structure data.
    @copyright (c) Turku PET Centre
    @author Vesa Oikonen
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcimage.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check whether two IMG data have the same matrix (x,y,z) size.
    @return 0 in case of match, 1 otherwise.
    @sa imgCompareConc, imgCompareTimes
    @author Vesa Oikonen
 */
int imgCompareMatrixSize(
  /** Pointer to IMG structure. */
  IMG *d1,
  /** Pointer to IMG structure. */
  IMG *d2
) {
  if(d1==NULL || d2==NULL) return(1);
  if(d1->dimz!=d2->dimz) return(1);
  if(d1->dimy!=d2->dimy) return(1);
  if(d1->dimx!=d2->dimx) return(1);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether time and concentration units are the same in two IMG data.
    @return 0 in case of match, 1 if y (concentration) unit is not matching,
    2 if x (time) unit is not matching, and 3 if neither is matching.
    @sa tacCompareUnit, imgCompareConc, imgCompareTimes
    @author Vesa Oikonen
 */
int imgCompareUnit(
  /** Pointer to IMG structure. */
  IMG *d1,
  /** Pointer to IMG structure. */
  IMG *d2,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(3);
  }

  int ret=0;
  if(d1->cunit!=d2->cunit) ret+=1;
  if(d1->tunit!=d2->tunit) ret+=2;
  if(ret>0 && verbose>0) {
    printf("img1.cunit := %s\n", unitName(d1->cunit));
    printf("img2.cunit := %s\n", unitName(d2->cunit));
    printf("img1.tunit := %s\n", unitName(d1->tunit));
    printf("img2.tunit := %s\n", unitName(d2->tunit));
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether IMG pixel concentrations are the same.
    Note that units are ignored here.

    If either absolute or relative difference is below the limit, the test is reported as passed.
    @sa tacCompareConc, imgCompareUnit, imgCompareTimes.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int imgCompareConc(
  /** Pointer to IMG structure. */
  IMG *d1,
  /** Pointer to IMG structure. */
  IMG *d2,
  /** Limit for accepted absolute difference; obligatory. */
  const float test_abs,
  /** Optional limit for accepted relative difference |2*(x1-x2)/(x1+x2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const float test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->dimz<1 || d2->dimz<1 || d1->dimy<1 || d2->dimy<1
     || d1->dimx<1 || d2->dimx<1 || d1->dimt<1 || d2->dimt<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }

  /* Dimensions must match */
  if(d1->dimz!=d2->dimz || d1->dimy!=d2->dimy || d1->dimx!=d2->dimx || d1->dimt!=d2->dimt) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different IMG dimension.\n");
    return(2);
  }
 
  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int zi=0; zi<d1->dimz; zi++)
    for(int yi=0; yi<d1->dimy; yi++)
      for(int xi=0; xi<d1->dimx; xi++)
        for(int ti=0; ti<d1->dimt; ti++) {
          if(floatMatch(d1->m[zi][yi][xi][ti], d2->m[zi][yi][xi][ti], test_abs)==1)
            continue;
          if(test_rel>0.0 && floatMatchRel(d1->m[zi][yi][xi][ti], d2->m[zi][yi][xi][ti], test_rel)==1)
            continue;
          if(verbose>0) {
            float s;
            s=fabsf(d1->m[zi][yi][xi][ti] - d2->m[zi][yi][xi][ti]);
            printf("img1.m[%d][%d][%d][%d] := %g\n", zi, yi, xi, ti, d1->m[zi][yi][xi][ti]);
            printf("img2.m[%d][%d][%d][%d] := %g\n", zi, yi, xi, ti, d2->m[zi][yi][xi][ti]);
            printf("|diff| := %g\n", s);
            printf("diff_limit := %g\n", test_abs);
            if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
          }
          return(10);
        }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether IMG frame times (x values) are the same.
    Note that units are ignored here.

    If either absolute or relative difference is below the limit, the test is reported as passed.
    @sa tacCompareTimes, imgCompareConc, imgCompareUnit.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int imgCompareTimes(
  /** Pointer to IMG structure. */
  IMG *d1,
  /** Pointer to IMG structure. */
  IMG *d2,
  /** Limit for accepted absolute difference. */
  const float test_abs,
  /** Optional limit for accepted relative difference |2*(x1-x2)/(x1+x2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const float test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->dimt<1 || d2->dimt<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }

  /* Frame nr must match */
  if(d1->dimt!=d2->dimt) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different frame nr.\n");
    return(3);
  }
  
  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int fi=0; fi<d1->dimt; fi++) {
    if(floatMatch(d1->x[fi], d2->x[fi], test_abs)==1)
      continue;
    if(floatMatch(d1->x1[fi], d2->x1[fi], test_abs)==1)
      continue;
    if(floatMatch(d1->x2[fi], d2->x2[fi], test_abs)==1)
      continue;
    if(test_rel>0.0 && 
       floatMatchRel(d1->x[fi], d2->x[fi], test_rel)==1 &&
       floatMatchRel(d1->x1[fi], d2->x1[fi], test_rel)==1 &&
       floatMatchRel(d1->x2[fi], d2->x2[fi], test_rel)==1)
      continue;
    if(verbose>0) {
      printf("img1.x1[%d] := %g\n", fi, d1->x1[fi]);
      printf("img2.x1[%d] := %g\n", fi, d2->x1[fi]);
      printf("|diff| := %g\n", fabsf(d1->x1[fi]-d2->x1[fi]));
      printf("img1.x2[%d] := %g\n", fi, d1->x2[fi]);
      printf("img2.x2[%d] := %g\n", fi, d2->x2[fi]);
      printf("|diff| := %g\n", fabsf(d1->x2[fi]-d2->x2[fi]));
      printf("img1.x[%d] := %g\n", fi, d1->x[fi]);
      printf("img2.x[%d] := %g\n", fi, d2->x[fi]);
      printf("|diff| := %g\n", fabsf(d1->x[fi]-d2->x[fi]));
      printf("diff_limit := %g\n", test_abs);
      if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
    }
    return(11);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
