/** @file tpcisotope.h
    @brief Header file for library libtpcisotope.
    @details Isotope half-lives and branching fractions are based on
    the following references, except isotopes
    Br-75, Br-76, Cu-62, Cu-64, Fe-52, O-14, Rb-82, Zn-62, I-124, which thus
    should be treated with caution. 
    <br>References:
    1. Table of Isotopes, Sixth edition, edited by C.M. Lederer,
       J.M. Hollander, I. Perlman. WILEY, 1967.
    2. Monographie BIPM-5, Table of Radionuclides, Vol. 6, 2011,
       <http://www.bipm.org/en/publications/monographies-ri.html>.
  
    @author Vesa Oikonen
 */
#ifndef _TPCISOTOPE_H_
#define _TPCISOTOPE_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <ctype.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
/* Max atomic number for elements */
#ifndef MAX_ATOMIC_NUMBER
#define MAX_ATOMIC_NUMBER 118 ///< Max atomic number, and the size of element_symbol and element_name lists.
#endif
/* Define ln(2) if it is not defined */
#ifndef M_LN2
#define M_LN2 0.69314718055994530942 ///< Defined ln(2) for faster computations
#endif
/* Define max isotope code length */
#ifndef MAX_ISOTOPE_LEN
#define MAX_ISOTOPE_LEN 16 ///<  Max string length for PET isotope
#endif
/*****************************************************************************/

/*****************************************************************************/
/** isotope_code, same as isotopes position in isotope table (table can not
    be accessed directly outside the c file).
    @sa isotope.c, isotopeName, isotopeIdentify
 */
typedef enum {
  ISOTOPE_UNKNOWN, ///< Unknown
  ISOTOPE_BR_75,   ///< Br-75
  ISOTOPE_BR_76,   ///< Br-76
  ISOTOPE_CU_62,   ///< Cu-62
  ISOTOPE_CU_64,   ///< Cu-64
  ISOTOPE_FE_52,   ///< Fe-52
  ISOTOPE_GA_68,   ///< Ga-68
  ISOTOPE_GE_68,   ///< Ge-68
  ISOTOPE_NA_22,   ///< Na-22
  ISOTOPE_RB_82,   ///< Rb-82
  ISOTOPE_SC_44,   ///< Sc-44
  ISOTOPE_TB_152,  ///< Tb-152
  ISOTOPE_ZN_62,   ///< Zn-62
  ISOTOPE_ZR_89,   ///< Zr-89
  ISOTOPE_C_11,    ///< C-11
  ISOTOPE_F_18,    ///< F-18
  ISOTOPE_I_124,   ///< I-124
  ISOTOPE_N_13,    ///< N-13
  ISOTOPE_O_15,    ///< O-15
  ISOTOPE_O_14     ///< O-14
} isotope;
/*****************************************************************************/

/*****************************************************************************/
/** Is data corrected for physical decay or not? 
   @sa decay.c, decayDescr
 */
typedef enum {
  DECAY_UNKNOWN,         ///< Not known; usually assumed that data is corrected.
  DECAY_NOTCORRECTED,    ///< Data is not corrected for physical decay.
  DECAY_CORRECTED,       ///< Data is corrected for physical decay.
  DECAY_CORRECTED_START, ///< Data is corrected for physical decay to scan start time.
  DECAY_CORRECTED_ADMIN  ///< Data is corrected for physical decay to radioligand administration time.
} decaycorrection;
/*****************************************************************************/

/*****************************************************************************/
/* elements */
extern char *elementName(unsigned short int z);
extern char *elementSymbol(unsigned short int z);
extern unsigned short int elementIdentify(const char *str);
/*****************************************************************************/

/*****************************************************************************/
/* isotope */
extern double isotopeHalflife(int isotope);
extern double isotopeBranching(int isotope);
extern char *isotopeName(int isotope);
extern int isotopeIdentifyHalflife(double halflife);
extern int isotopeIdentify(const char *isotope);
/*****************************************************************************/

/*****************************************************************************/
/* decay */
extern char *decayDescr(decaycorrection d);
extern double lambdaFromHalflife(double halflife);
extern double lambdaFromIsotope(int isotope);
extern double decayCorrectionFactorFromLambda(double lambda, double starttime, double duration);
extern double decayCorrectionFactorFromIsotope(int isotope, double starttime, double duration);
/*****************************************************************************/

/*****************************************************************************/
/* ranoise */
extern double noiseSD4Frame(double y, double t1, double dt, int isotope, double a);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCISOTOPE */
