/** @file integrate.c
 *  @brief Linear integration.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcli.h"
/*****************************************************************************/

/*****************************************************************************/
/** @brief Linear integration of TAC with trapezoidal method.
  
    @details
    Use this function for input data, or PET data when frame start and end times are not known. 
    Data must be frequently sampled to get accurate results.
    AUC calculation is started from time 0 or first input sample time, whichever is smaller.
    Very simple data extrapolation can be optionally enabled.
    Results will be wrong if data contains NaNs.
    To get the same result as with integrate() in the libtpcmodel, set se=3.
   
    @sa liInterpolate(), liIntegratePET()
    @author Vesa Oikonen
    @return 0 if successful, otherwise >0. 
 */
int liIntegrate(
  /** Array of input data x (time) values; obligatory.
      Data must be sorted by ascending x, but consecutive points may have the same x.
      Negative x is accepted, AUCs are then started from the first x, otherwise from 0. */
  double *x,
  /** Array of input data y (concentration) values; obligatory. */
  double *y,
  /** Number of samples in input data; obligatory. */
  const int nr,
  /** Array for integrals (AUCs) at each x; obligatory. */
  double *yi,
  /** Extrapolation setting for the start of data:
      - 0= assuming y=0 when x<x[0]
      - 1= assuming y=y[0] when 0<=x<x[0] and y=0 when x<x[0]<0
      - 2= assuming y=0 when x<=0 and x[0]>0, and y=x*y[0]/x[0] when 0<x<x[0] and y[0]>0
      - 3= same as '2' if 2*x[0]-x[1]<=0, otherwise same as '0'
      - 4= same as '2' if 2*x[0]-x[1]<=0, otherwise assuming 
           y=0 when x<=2*x[0]-x[1], and 
           y=(x-(2*x[0]-x[1]))*y[0]/x[0] when (2*x[0]-x[1])<x<x[0]
           and y[0]>0.
   */
  const int se,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) printf("liIntegrate(x, y, %d, yi, %d)\n", nr, se);
  /* Check for data */
  if(nr<1) return 1;
  if(x==NULL || y==NULL || yi==NULL) return 1;

  /* Initiate the first integral to 0 */
  yi[0]=0.0;
  
  /* Extrapolate the initial phase */
  if(se==0) {
    /* No extrapolation */
  } else if(se==1) {
    /* Extrapolation with a rectangle */
    if(x[0]>0.0) yi[0]+=x[0]*y[0];
  } else if(se==2) {
    /* Extrapolation with a triangle starting from zero */
    if(x[0]>0.0 && y[0]>0.0) yi[0]+=0.5*x[0]*y[0];
  } else if(se==3) {
    /* Extrapolation with a triangle, if small gap */
    if(x[0]>0.0 && y[0]>0.0) {
      if(nr<2 || (2.0*x[0]-x[1])<1.0E-10)
        yi[0]+=0.5*x[0]*y[0];
    }
  } else if(se==4) {
    /* Extrapolation with a triangle */
    if(x[0]>0.0 && y[0]>0.0) {
      if(nr<2 || (2.0*x[0]-x[1])<1.0E-10)
        yi[0]+=0.5*x[0]*y[0];
      else {
        double nx;
        nx=2.0*x[0]-x[1];
        yi[0]+=0.5*(x[0]-nx)*y[0];        
      }
    }
  }

  /* Dot-to-dot for the rest of data */
  for(int j=1; j<nr; j++) yi[j]=yi[j-1]+0.5*(y[j]+y[j-1])*(x[j]-x[j-1]);

  if(verbose>3) printf("liIntegrate() ready\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Calculate PET TAC AUC from start to each time frame, as averages during each frame
 
    @details 
    Use this function for PET data when frame start and end times are known. 
    AUC at the end of frame i is calculated as AUCe[i]=AUCe[i-1]+y[i]*(x2[i]-x1[i]),
    and the integral for frame i is calculated as an average AUC[i]=(AUCe[i]+AUCe[i-1])/2.
    AUC calculation is started from first frame start time, ignoring possible time gap before it.
    Any gaps between time frames are filled with an imaginary frame calculated as weighted average 
    of previous and next frame.
    Frames must be in ascending time order and frame overlaps removed; 
    wrong order or large overlaps will lead to substantial errors in integrals without warning.
    Results will be wrong if data contains NaNs.
   
    @sa liInterpolate(), liIntegrate(), liInterpolateForPET()
    @author Vesa Oikonen
    @return 0 if successful, otherwise >0. 
 */
int liIntegratePET(
  /** Array of input data x1 (frame start time) values; obligatory.
      Negative frame times are accepted. */
  double *x1,
  /** Array of input data x2 (frame end time) values; obligatory.
      Negative frame times are accepted, but it is required that for each frame x2>=x1. */
  double *x2,
  /** Array of frame mean y (concentration) values; obligatory. */
  double *y,
  /** Number of samples (frames); obligatory. */
  int nr,
  /** Array for integrals (AUCs) at each x (frame middle time); enter NULL if not needed. */
  double *ie,
  /** Array for 2nd integrals (AUCs) at each x (frame middle time); enter NULL if not needed. */
  double *iie,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) printf("liIntegratePET(x1, x2, y, %d, i, ii)\n", nr);

  /* Check for data */
  if(nr<1) return(1);
  if(x1==NULL || x2==NULL || y==NULL) return(1);
  /* No use to do anything if both output arrays are NULL */
  if(ie==NULL && iie==NULL) return(0);

  int i;
  double x, xs, fdur, half_fdur, gap, gap_y, midi, midii, fulli;
  double box_integral=0.0, box_integral2=0.0;
  double gap_integral=0.0, gap_integral2=0.0;
  double prev_x2=x1[0], prev_fdur=0.0;

  for(i=0; i<nr; i++) {
    /* Duration of this time frame */
    fdur=x2[i]-x1[i]; if(!(fdur>=0.0)) return(2);
    half_fdur=0.5*fdur;
    x=0.5*(x1[i]+x2[i]);
    if(verbose>3) {
      printf("  x=%g\n", x);
      printf("  fdur=%g\n", fdur);
    }
    /* Integral of a possible gap between frames */
    gap=x1[i]-prev_x2;
    if(i>0 && gap>1.0E-10) {
      /* Estimate y during gap as weighted mean of previous and this frame */
      gap_y=fdur*y[i]+prev_fdur*(y[i-1]);
      xs=fdur+prev_fdur; if(xs>0.0) gap_y/=xs;
      gap_integral=gap*gap_y;
      gap_integral2=gap*0.5*(box_integral+box_integral+gap_integral);
      if(verbose>3) {
        printf("  gap=%g\n", gap);
        printf("  gap_y=%g\n", gap_y);
        printf("  gap_integral=%g\n", gap_integral);
        printf("  gap_integral2=%g\n", gap_integral2);
      }
      /* Add gap integrals */
      box_integral+=gap_integral;
      box_integral2+=gap_integral2;
    }
    /* Calculate integrals at frame middle time */
    midi=y[i]*half_fdur;
    midii=0.5*(box_integral+box_integral+midi)*half_fdur;
    if(verbose>4) {
      printf("  midframe_integral=%g\n", midi);
      printf("  midframe_integral2=%g\n", midii);
    }
    if(ie!=NULL) ie[i]=box_integral+midi; 
    if(iie!=NULL) iie[i]=box_integral2+midii;
    /* Add full frame integrals */
    fulli=fdur*y[i];
    box_integral2+=0.5*(box_integral+box_integral+fulli)*fdur;
    box_integral+=fulli;
    if(verbose>3) {
      printf("  box_integral=%g\n", box_integral);
      printf("  box_integral2=%g\n", box_integral2);
    }
    /* Prepare for the next frame */
    prev_x2=x2[i]; prev_fdur=fdur;
  } // next time frame

  if(verbose>3) printf("liIntegratePET() ready\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** @brief Linear integration of PET TAC to frame end times.
 
    @details 
    Use this function for PET data when frame start and end times are known. 
    AUC calculation is started from first frame start time, ignoring possible time gap before it.
    Any gaps between time frames are filled with an imaginary frame calculated as weighted average 
    of previous and next frame.
    Frames must be in ascending time order and frame overlaps removed.
   
    @sa liIntegratePET, liInterpolate(), liIntegrate(), liInterpolateForPET()
    @author Vesa Oikonen
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int liIntegrateFE(
  /** Array of input data x1 (frame start time) values; obligatory.
      Negative frame times are accepted. */
  double *x1,
  /** Array of input data x2 (frame end time) values; obligatory.
      Negative frame times are accepted, but it is required that for each frame x2>=x1. */
  double *x2,
  /** Array of frame mean y (concentration) values; obligatory. */
  double *y,
  /** Number of samples (frames); obligatory. */
  int nr,
  /** Array for integrals (AUCs) at each x (frame middle time); enter NULL if not needed. */
  double *ie,
  /** Array for 2nd integrals (AUCs) at each x (frame middle time); enter NULL if not needed. */
  double *iie,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout. */
  const int verbose
) {
  if(verbose>0) printf("liIntegrateFE(x1, x2, y, %d, i, ii, %d)\n", nr, verbose);

  /* Check for data */
  if(nr<1) return(TPCERROR_FAIL);
  if(x1==NULL || x2==NULL || y==NULL) return(TPCERROR_FAIL);
  /* No use to do anything if both output arrays are NULL */
  if(ie==NULL && iie==NULL) return(TPCERROR_OK);

  double prev_fdur=0.0;
  double prev_iy=0.0, prev_iiy=0.0;
  double iy=0.0, iiy=0.0;
  for(int i=0; i<nr; i++) {
    /* Duration of this time frame */
    double fdur=x2[i]-x1[i]; 
    if(!isfinite(fdur)) return(TPCERROR_INVALID_X);
    if(fabs(fdur)<1.0E-10) { // frame length is about zero
      if(ie!=NULL) ie[i]=prev_iy;
      if(iie!=NULL) iie[i]=prev_iiy;
      continue;
    }
    if(fdur<0.0) return(TPCERROR_INVALID_X);
    /* Check for gap */
    if(i>0) {
      double gap=x1[i]-x2[i-1];
      if(gap<0.0) return(TPCERROR_INVALID_X); // overlap not accepted
      if(gap>1.0E-10) {
        /* Estimate the integral of during the gap;
           estimate y during gap as weighted mean of previous and this frame */
        double gap_y=fdur*y[i]+prev_fdur*(y[i-1]);
        double xs=fdur+prev_fdur; if(xs>0.0) gap_y/=xs;
        double gap_integral=gap*gap_y;
        double gap_integral2=gap*0.5*(iy+iy+gap_integral);
        /* Add gap integrals */
        iy+=gap_integral;
        iiy+=gap_integral2;
      }
    }
    /* Calculate integrals at frame end */
    double box_integral=fdur*y[i];
    double box_integral2=fdur*0.5*(iy+iy+box_integral);
    /* Add gap integrals */
    iy+=box_integral;
    iiy+=box_integral2;
    if(ie!=NULL) ie[i]=iy; 
    if(iie!=NULL) iie[i]=iiy;
    /* Prepare for the next frame */
    prev_fdur=fdur; prev_iy=iy; prev_iiy=iiy;
  } // next time frame

  if(verbose>3) printf("liIntegrateFE() ready\n");
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/

#if(0)
/** @brief Linear integration of TAC with trapezoidal method.
 *
 *  @details 
 *  Use this function for PET data when frame start and end
 *  times are known. Integrals are calculated to frame middle times.
 *  Frames must be in ascending time order.
 *  Previous frames are modelled with rectangles; current frames and
 *  gaps are with trapezoids, based on frame middle points.
 *  Frame overlaps are ignored, thus large overlap will lead to substantial
 *  errors in integrals.
 *  Results will be wrong if data contains NaNs.
 *  AUC calculation is started from time 0 or first frame start time,
 *  whichever is smaller.
 *  Very simple data extrapolation can be optionally enabled.
 *  To get about the same result as with petintegral() in the libtpcmodel,
 *  set se=3.
 * 
 *  @sa liInterpolate(), liIntegrate(), liInterpolateForPET()
 *  @author Vesa Oikonen
 *  @return 0 if successful, otherwise >0. 
 */
int liIntegratePET(
  /** Array of input data x1 (frame start time) values; obligatory.
   *  Negative frame times are accepted, AUCs are then started from the first x1,
   *  otherwise from 0. */
  double *x1,
  /** Array of input data x2 (frame end time) values; obligatory.
   *  Negative frame times are accepted, but it is required that for each frame
   *  x2>=x1. */
  double *x2,
  /** Array of frame mean y (concentration) values; obligatory */
  double *y,
  /** Number of samples (frames); obligatory */
  int nr,
  /** Array for integrals (AUCs) at each x (frame middle time);
      enter NULL if not needed. */
  double *ie,
  /** Array for 2nd integrals (AUCs) at each x (frame middle time);
      enter NULL if not needed. */
  double *iie,
  /** Extrapolation setting for the start of data:
  Integral at the middle of the first frame is
  - 0: triangle from frame start to frame middle, 0.5*y[0]*(x[0]-x1[0])
  - 1: rectangle from frame start to frame middle, if x1[0]<=0, or
       rectangle from 0 to frame middle, if x1[0]>0
  - 2: triangle from frame start to frame middle, if x1[0]<=0;
       triangle from 0 to frame middle, if x1[0]>0
  - 3: triangle from frame start to frame middle, if x1[0]<=0;
       triangle from frame start to frame middle, if gap is longer 
       than first frame length;
       triangle from 0 to frame middle otherwise
  - 4: triangle from frame start to frame middle, if x1[0]<=0;
       triangle from 0 to frame middle, if gap is not longer than first frame;
       triangle from (frame start - frame length) to frame middle, if gap is longer
   */
  const int se,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  const int verbose
) {

  if(verbose>0) printf("liIntegratePET(x1, x2, y, %d, i, ii, %d)\n", nr, se);
  /* Check for data */
  if(nr<1) return 1;
  if(x1==NULL || x2==NULL || y==NULL) return 1;
  /* No use to do anything if both output arrays are NULL */
  if(ie==NULL && iie==NULL) return 0;

  int i;
  double x, s, last_x, last_x1, last_x2, last_y;
  double frame_len;
  double box_integral=0.0; //, last_integral;
  double box_integral2=0.0; //, last_integral2;

  /* Set initial values, extrapolating if necessary */
  frame_len=x2[0]-x1[0]; if(frame_len<0.0) return(5);
  if(se==1) {
    /* assume that y=y[0] before the first frame */
    if(x1[0]<0.0) last_x1=x1[0]; else last_x1=0.0;
    last_x2=x1[0]; last_x=0.5*(last_x1+last_x2); last_y=y[0];
    box_integral=last_y*(last_x2-last_x1);
  } else if(x1[0]<1.0E-10 || se==0) {
    /* no extrapolation, just assume that there is all zeroes before first frame */
    last_x1=last_x2=last_x=x1[0]; last_y=0.0;
    box_integral=0.0;
  } else {
    if(se==2) {
      /* assume a triangle before the first frame */
      x=0.5*(x1[0]+x2[0]); s=y[0]/x;
      last_x1=0.0; last_x2=x1[0]; last_x=0.5*(last_x1+last_x2); last_y=s*last_x;
      box_integral=0.5*(s*x1[0])*x1[0];
    } else if(se==3) {
      if((x1[0]-frame_len)>1.0E-10) {
        last_x1=last_x2=last_x=x1[0]; last_y=0.0;
        box_integral=0.0;
      } else {
        /* assume a triangle before the first frame */
        x=0.5*(x1[0]+x2[0]); s=y[0]/x;
        last_x1=0.0; last_x2=x1[0]; last_x=0.5*(last_x1+last_x2); last_y=s*last_x;
        box_integral=0.5*(s*x1[0])*x1[0];
      }
    } else /*if(se==4)*/ {
      if((x1[0]-frame_len)>1.0E-10) {
        /* assume a triangle before the first frame, same length as the first */
        last_x1=x1[0]-frame_len; last_x2=x1[0]; last_x=0.5*(last_x1+last_x2);
        x=0.5*(x1[0]+x2[0]); s=y[0]/(x-last_x1);
        last_y=s*(last_x-last_x1);
        box_integral=0.5*(s*(x1[0]-last_x1))*(last_x2-last_x1);
      } else {
        /* assume a triangle before the first frame, starting from zero */
        x=0.5*(x1[0]+x2[0]); s=y[0]/x;
        last_x1=0.0; last_x2=x1[0]; last_x=0.5*(last_x1+last_x2); 
        last_y=s*last_x;
        box_integral=0.5*(s*last_x2)*last_x2;
      }
    }
  }
  //last_integral=box_integral;
  //printf("last_integral=%g\n", last_integral);
  box_integral2=0.5*box_integral*(last_x2-last_x1);
  //printf("integral2=%g\n", integral2);
//printf("box_integral=%g\n", box_integral);
//printf("box_integral2=%g\n", box_integral2);


  /* Frame-by-frame */
  double integral=0.0, integral2=0.0;
  double half_integral, gap, gap_y, gap_integral, xdist, box;
  double half_integral2, gap_integral2;
  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    //printf("xdist=%g\n", xdist);
    s=(y[i]-last_y)/xdist; /* slope */
    /* Integral of a possible gap between frames */
    gap=x1[i]-last_x2;
    if(i>0 && gap>1.0E-10) {
      gap_y=y[i]-s*(x-0.5*(last_x2+x1[i]));
//printf("gap=%g\n", gap);
//printf("gap_y=%g\n", gap_y);
      //gap_integral=gap*(last_y+s*((last_x2+x1[i])/2.0-last_x)); 
      gap_integral=gap*gap_y;
      gap_integral2=gap*0.5*(box_integral+box_integral+gap_integral);
//printf("gap_integral=%g\n", gap_integral);
//printf("gap_integral2=%g\n", gap_integral2);
    } else {
      gap_integral=gap_integral2=0.0; //xdist=0.5*(x2[i]-x1[i]);
    }
    box_integral+=gap_integral;
    box_integral2+=gap_integral2;
    /* Integral from the beginning of the frame to the middle of the frame */
    //half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    half_integral=0.5*frame_len*(y[i]-s*0.25*frame_len);
    half_integral2=0.5*(box_integral+box_integral+half_integral)*0.5*frame_len;
//printf("half_integral=%g\n", half_integral);
//printf("half_integral2=%g\n", half_integral2);
    integral=box_integral+half_integral;
    integral2=box_integral2+half_integral2;
    //printf("integral=%g\n", integral);
    /* half_integral is not added to box because it is more accurate to 
       increment integrals of full frames */
    box=frame_len*y[i];
    box_integral+=box;
    box_integral2+=frame_len*0.5*(box_integral+box_integral-box);
//printf("box_integral=%g\n", box_integral);
//printf("box_integral2=%g\n", box_integral2);
   //printf("integral=%g last_integral=%g xdist=%g\n", integral, last_integral, xdist);
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; //last_integral=integral; last_integral2=integral2;
  }
  
  if(verbose>3) printf("liIntegratePET() ready\n");
  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/** @brief Calculate the integrals (AUC) of the first halves of PET frames
    based on dot-to-dot linear interpolation, assuming that original y values 
    represent the mean value during the frame, i.e. 
    AUC(frame start - frame end time)/frame duration.
  
    @details
    Original data and new x values must be sorted by ascending x.
    Subsequent x values can have equal values, enabling the use of step
    functions.
    Negative x (time) values can be processed.
    Results will be wrong if data contains NaNs.
    The beginning and end of true TAC is assumed to follow a line drawn
    based on the two first/last TAC middle frame y values. 
    
    @note Not for production use, just for play.
    @author Vesa Oikonen
    @sa liInterpolate(), liIntegratePET(), liInterpolateForPET()
    @return 0 if successful, otherwise >0. 
 */
int liIntegrateHalfFrame(
  /** Array of PET frame start times; obligatory.
   *  Data must be sorted by ascending x, but consecutive points may have the
   *  same x. Negative x is accepted. */
  double *x1,
  /** Array of PET frame end times; obligatory.
   *  Data must be sorted by ascending x, but consecutive points may have the
   *  same x. Negative x is accepted, but frame duration (x2-x1) must be >=0. */
  double *x2,
  /** Array of original TAC y (concentration) values, representing the
   *  average concentration during the PET frame; obligatory */
  double *y,
  /** Number of samples in the data; obligatory */
  const size_t nr,
  /** Array for first-half frame integrals of TAC y (concentration) values 
   *  for each PET frame; obligatory */
  double *fhi,
  /** Verbose level; if zero, then nothing is printed to stderr or stdout */
  const int verbose
) {
  if(verbose>0) 
    printf("liIntegrateHalfFrame(x1, x2, y, %zu, fhi, %d)\n", nr, verbose);

  /* Check for data */
  if(nr<1) return 1;
  if(x1==NULL || x2==NULL || y==NULL || fhi==NULL) return 1;

  /* If just one frame */
  if(nr==1) {
    if(verbose>2) printf("just one frame.\n");
    fhi[0]=y[0]*(x2[0]-x1[0])/2.0;
    return 0;
  }

  /* Calculate frame middle times and durations / 2.
     Check that frames durations are >=0.
     Check that frame times are in ascending order.
  */
  double hfdur[nr], x[nr];
  for(size_t i=0; i<nr; i++) {
    x[i]=0.5*(x1[i]+x2[i]);
    hfdur[i]=0.5*(x2[i]-x1[i]); 
    if(hfdur[i]<0.0) {
      if(verbose>0) fprintf(stderr, "negative frame length.\n");
      return 2;
    }
    if(i>0 && x[i]<x[i-1]) {
      if(verbose>0) fprintf(stderr, "frame times not ascending.\n");
      return 3;
    }
  }

  /* If just two frames */
  if(nr<3) {
    if(verbose>2) printf("just two frames.\n");
    double k, dx;
    dx=x[1]-x[0]; 
    if(dx<1.0E-100) {fhi[0]=y[0]*hfdur[0]; fhi[1]=y[1]*hfdur[1]; return 0;}
    k=(y[1]-y[0])/dx;
    fhi[0]=hfdur[0]*(y[0]-k*0.5*hfdur[0]);
    fhi[1]=hfdur[1]*(y[1]-k*0.5*hfdur[1]);
    return 0; 
  }

  /* Compute data matrix for solving (very simply) the y values at frame middle
     times */
  double A[2][nr], B[nr], xdif;
  A[0][0]=1.0; A[1][0]=0.0; B[0]=y[0];
  for(size_t i=1; i<nr-1; i++) {
    xdif=x[i+1]-x[i];
    if(!(hfdur[i]>1.0E-100) || !(xdif>1.0E-100)) { // catches both zero and NaN
      A[0][i]=0.0; A[1][i]=0.0; B[i]=y[i];
    } else {
      A[0][i]=4.0*(x[i]-x[i-1])/hfdur[i] - (x[i+1]-x[i-1])/xdif;
      A[1][i]=(x[i]-x[i-1])/xdif;
      B[i]=4.0*y[i]*(x[i]-x[i-1])/hfdur[i];
    }
  }
  A[0][nr-1]=1.0; A[1][nr-1]=0.0; B[nr-1]=y[nr-1]; 
  if(verbose>4) {
    printf("A[1]\tA[2]\t\tB\n");
    for(size_t i=0; i<nr; i++) printf("%g\t%g\t\t%g\n", A[0][i], A[1][i], B[i]);
  }

  /* Solve */
  for(size_t i=1; i<nr-1; i++) {
    if(fabs(A[0][i])<1.0E-100) continue;
    A[0][i]-=A[1][i-1]; B[i]-=B[i-1];
    A[1][i]/=A[0][i]; B[i]/=A[0][i]; A[0][i]=1.0; 
  }
  if(verbose>5) {
    printf("A[1]\tA[2]\t\tB\n");
    for(size_t i=0; i<nr; i++) printf("%g\t%g\t\t%g\n", A[0][i], A[1][i], B[i]);
  }
  fhi[0]=y[0]; fhi[nr-1]=y[nr-1];
  for(size_t i=nr-2; i>0; i--) {
    fhi[i]=B[i]-fhi[i+1]*A[1][i];
    if(verbose>5) printf("X[%zu]=%g\n", i+1, fhi[i]);
  }
  /* Set any NaNs to original values (which of course may be NaN as well) */
  for(size_t i=1; i<nr-1; i++) if(isnan(fhi[i])) fhi[i]=y[i];
  if(verbose>10) {
    printf("estimated frame mid values:\n");
    for(size_t i=0; i<nr; i++) printf("  %g\n", fhi[i]);
  }

  /* Compute the first half integrals */
  double newi, lasty=fhi[0];
  double dx=x[1]-x[0];
  if(!(dx>1.0E-100)) fhi[0]*=hfdur[0];
  else fhi[0]=hfdur[0]*(fhi[0]-0.5*hfdur[0]*(fhi[1]-fhi[0])/dx);
  for(size_t i=1; i<nr; i++) {
    dx=x[i]-x[i-1];
    if(!(dx>1.0E-100)) newi=fhi[i]*hfdur[0];
    else newi=hfdur[i]*(fhi[i]-0.5*hfdur[i]*(fhi[i]-lasty)/dx);
    lasty=fhi[i]; fhi[i]=newi;    
  }

  if(verbose>1) printf("liIntegrateHalfFrame() ready\n");
  return 0;                     
}
/*****************************************************************************/

/*****************************************************************************/



#if(0)
/*****************************************************************************/

/*****************************************************************************/
/** Calculates integrals of PET data at frame end times.

    Data does not have to be continuous, but it must be increasing in time.
    For faster performance in repetitive calls, allocate memory for integral[]
    even if it is not needed.
    If x1[0] is >0 AND x1[0] is <=(x2[0]-x1[0]),
    then the beginning is interpolated from (0, 0) to (x1[0], y1[0]*x1[0]/x)
    where x is midtime of the first frame.
\return Returns 0 if ok.
*/
int petintegrate(
  /** Array of frame start times */
  double *x1,
  /** Array of frame end times */
  double *x2,
  /** Array of y values (avg during each frame) */
  double *y,
  /** Nr of frames */
  int nr,
  /** Output: integral values at frame end times, or NULL */
  double *newyi,
  /** Output: 2nd integral values at frame end times, or NULL */
  double *newyii
) {
  int i, allocated=0;
  double *ti, x, a;


  if(INTEGR_TEST) printf("in petintegrate()\n");
  /* Check that data is increasing in time and is not (much) overlapping */
  if(nr<1 || x1[0]<0) return 1;
  for(i=0; i<nr; i++) if(x2[i]<x1[i]) return 2;
  for(i=1; i<nr; i++) if(x1[i]<=x1[i-1]) return 3;

  /* Allocate memory for temp data, if necessary */
  if(newyi==NULL) {
    allocated=1;
    ti=(double*)malloc(nr*sizeof(double)); if(ti==NULL) return 4;
  } else
    ti=newyi;

  /* Calculate the integral at the end of first frame */
  ti[0]=(x2[0]-x1[0])*y[0];
  /* If frame does not start at 0 time, add the area in the beginning */
  /* But only if the gap in the beginning is less than the lenght of the 
     first frame */
  if(x1[0]>0) {
    if(x1[0]<=x2[0]-x1[0]) {
       x=(x1[0]+x2[0])/2.0; 
       a=(x1[0]*(y[0]/x)*x1[0])/2.0; 
       ti[0]+=a;
    }
  }

  /* Calculate integrals at the ends of following frames */
  for(i=1; i<nr; i++) {
    /* Add the area of this frame to the previous integral */
    a=(x2[i]-x1[i])*y[i]; ti[i]=ti[i-1]+a;
    /* Check whether frames are contiguous */
    if(x1[i]==x2[i-1]) continue;
    /* When not, calculate the area of an imaginary frame */
    x=(x1[i]+x2[i-1])/2.0;
    a=(x1[i]-x2[i-1])*
      (y[i]-(y[i]-y[i-1])*(x2[i]+x1[i]-2.0*x)/(x2[i]+x1[i]-x2[i-1]-x1[i-1]));
    ti[i]+=a;
  }

  /* Copy integrals to output if required */
  if(allocated) for(i=0; i<nr; i++) newyi[i]=ti[i];

  /* Calculate 2nd integrals if required */
  if(newyii!=NULL) {
    newyii[0]=x2[0]*ti[0]/2.0;
    for(i=1; i<nr; i++)
      newyii[i]=newyii[i-1]+(x2[i]-x2[i-1])*(ti[i-1]+ti[i])/2.0;
  }
  
  /* Free memory */
  if(allocated) free((char*)ti);

  if(INTEGR_TEST) printf("out petintegrate()\n");
  return 0;
}

/* Float version of the same function */
int fpetintegrate(
  /** Array of frame start times */
  float *x1,
  /** Array of frame end times */
  float *x2,
  /** Array of y values (avg during each frame) */
  float *y,
  /** Nr of frames */
  int nr,
  /** Output: integral values at frame end times, or NULL */
  float *newyi,
  /** Output: 2nd integral values at frame end times, or NULL */
  float *newyii
) {
  int i, allocated=0;
  float *ti, x, a;


  if(INTEGR_TEST) printf("in fpetintegrate()\n");
  /* Check that data is increasing in time and is not (much) overlapping */
  if(nr<1 || x1[0]<0) return 1;
  for(i=0; i<nr; i++) if(x2[i]<x1[i]) return 2;
  for(i=1; i<nr; i++) if(x1[i]<=x1[i-1]) return 3;

  /* Allocate memory for temp data, if necessary */
  if(newyi==NULL) {
    allocated=1;
    ti=(float*)malloc(nr*sizeof(float)); if(ti==NULL) return 4;
  } else
    ti=newyi;

  /* Calculate the integral at the end of first frame */
  ti[0]=(x2[0]-x1[0])*y[0];
  /* If frame does not start at 0 time, add the area in the beginning */
  /* But only if the gap in the beginning is less than the lenght of the 
     first frame */
  if(x1[0]>0) {
    if(x1[0]<=x2[0]-x1[0]) {
       x=(x1[0]+x2[0])/2.0; 
       a=(x1[0]*(y[0]/x)*x1[0])/2.0; 
       ti[0]+=a;
    }
  }

  /* Calculate integrals at the ends of following frames */
  for(i=1; i<nr; i++) {
    /* Add the area of this frame to the previous integral */
    a=(x2[i]-x1[i])*y[i]; ti[i]=ti[i-1]+a;
    /* Check whether frames are contiguous */
    if(x1[i]==x2[i-1]) continue;
    /* When not, calculate the area of an imaginary frame */
    x=(x1[i]+x2[i-1])/2.0;
    a=(x1[i]-x2[i-1])*
      (y[i]-(y[i]-y[i-1])*(x2[i]+x1[i]-2.0*x)/(x2[i]+x1[i]-x2[i-1]-x1[i-1]));
    ti[i]+=a;
  }

  /* Copy integrals to output if required */
  if(allocated) for(i=0; i<nr; i++) newyi[i]=ti[i];

  /* Calculate 2nd integrals if required */
  if(newyii!=NULL) {
    newyii[0]=x2[0]*ti[0]/2.0;
    for(i=1; i<nr; i++)
      newyii[i]=newyii[i-1]+(x2[i]-x2[i-1])*(ti[i-1]+ti[i])/2.0;
  }
  
  /* Free memory */
  if(allocated) free((char*)ti);

  if(INTEGR_TEST) printf("out fpetintegrate()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Interpolate and integrate TAC to PET frames.

  It is assumed, that original data is not from framed data, but that
  the values represent the actual value at specified time point, which
  allows the integration to be calculated dot-by-dot.

  If NULL is specified for *newy, *newyi and/or *newyii, their values are
  not calculated.

  Original data and new x values must be sorted by ascending x.
  Subsequent x values can have equal values, enabling the use of step functions.
  PET frames can overlap, but interpolation may then be slower.
  Negative x (time) values can be processed.
  If necessary, the data is extrapolated assuming that 1) y[inf]=y[nr-1] and
  2) if x[0]>0 and y[0]>0 and x[0]>x[1]-x[0], then an imaginary line is drawn
  from (0,0) to (x[0],y[0]).
*/
int interpolate4pet(
  /** Times of original data */
  double *x,
  /** Values of original data */
  double *y,
  /** Number of original data values */
  int nr,
  /** PET frame start times; frames may overlap */
  double *newx1,
  /** PET frame end times; frames may overlap */
  double *newx2, 
  /** Mean value during PET frame, or NULL if not needed;
   *  calculation may be faster if newyi is calculated too */
  double *newy,
  /** Integral at frame mid time, or NULL if not needed */
  double *newyi,
  /** 2nd integral at frame mid time, or NULL if not needed */
  double *newyii,
  /** Number of PET frames */
  int newnr
) {
  int ret, fi, overlap=0, zeroframe=0;
  double petx[3], pety[3], petyi[3], petyii[3], fdur;

  if(INTEGR_TEST) printf("in interpolate4pet()\n");

  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;

  /* Check frame lengths, also for overlap and zero length frames */
  for(fi=0; fi<newnr; fi++) {
    /* Calculate frame length */
    fdur=newx2[fi]-newx1[fi];
    /* If frame length is <0, that is an error */
    if(fdur<0.0) return 4;
    if(fdur==0.0) zeroframe++;
    /* Overlap? */
    if(fi>0 && newx2[fi-1]>newx1[fi]) overlap++;
  }
  if(INTEGR_TEST>1) {
    printf("overlap := %d\n", overlap);
    printf("zeroframe := %d\n", zeroframe);
  }

  /* Interpolate and integrate one frame at a time, if there is:
     -overlap in frames
     -frames of zero length
     -only few interpolated frames
     -if only integrals are needed
     -if neither of integrals is needed, then there is no temp memory to
      do otherwise
  */
  if(overlap>0 || zeroframe>0 || newnr<=3 ||
     newy==NULL || (newyi==NULL && newyii==NULL) )
  {

    if(INTEGR_TEST>1) printf("frame-by-frame interpolation/integration\n");
    for(fi=0; fi<newnr; fi++) {
      /* Set frame start, middle and end times */
      petx[0]=newx1[fi]; petx[2]=newx2[fi]; petx[1]=0.5*(petx[0]+petx[2]);
      /* Calculate frame length */
      fdur=petx[2]-petx[0];
      /* If frame length is <0, that is an error */
      if(fdur<0.0) return 4;
      /* If frame length is 0, then use direct interpolation */
      if(fdur==0.0) {
        ret=interpolate(x, y, nr, petx, pety, petyi, petyii, 1);
        if(ret) return 10+ret;
        if(newy!=NULL) newy[fi]=petyi[0];
        if(newyi!=NULL) newyi[fi]=petyi[0];
        if(newyii!=NULL) newyii[fi]=petyii[0];
        continue;
      }
      /* Calculate integrals at frame start, middle, and end */
      ret=interpolate(x, y, nr, petx, NULL, petyi, petyii, 3);
      if(ret) return 20+ret;
      /* Set output integrals, if required */
      if(newyi!=NULL) newyi[fi]=petyi[1];
      if(newyii!=NULL) newyii[fi]=petyii[1];
      /* Calculate frame mean, if required */
      if(newy!=NULL) newy[fi]=(petyi[2]-petyi[0])/fdur;
    } // next frame

  } else {

    if(INTEGR_TEST>1) printf("all-frames-at-once interpolation/integration\n");
    /* Set temp array */
    double *tp; if(newyii!=NULL) tp=newyii; else tp=newyi;
    /* Integrate at frame start times */
    ret=interpolate(x, y, nr, newx1, NULL, tp, NULL, newnr);
    if(ret) return 10+ret;
    /* Integrate at frame end times */
    ret=interpolate(x, y, nr, newx2, NULL, newy, NULL, newnr);
    if(ret) return 10+ret;
    /* Calculate average frame value */
    for(fi=0; fi<newnr; fi++)
      newy[fi]=(newy[fi]-tp[fi])/(newx2[fi]-newx1[fi]);
    /* Calculate integrals */
    if(newyi!=NULL || newyii!=NULL) {
      for(fi=0; fi<newnr; fi++) newx1[fi]+=0.5*(newx2[fi]-newx1[fi]);
      ret=interpolate(x, y, nr, newx1, NULL, newyi, newyii, newnr);
      if(ret) return 10+ret;
      for(fi=0; fi<newnr; fi++) newx1[fi]-=(newx2[fi]-newx1[fi]);
    }
  }

  if(INTEGR_TEST) printf("out interpolate4pet()\n");
  return 0;
}

/** float version of interpolate4pet() */
int finterpolate4pet(
  /** Times of original data */
  float *x,
  /** Values of original data */
  float *y,
  /** Number of original data values */
  int nr,
  /** PET frame start times; frames may overlap */
  float *newx1,
  /** PET frame end times; frames may overlap */
  float *newx2, 
  /** Mean value during PET frame, or NULL if not needed;
   *  calculation may be faster if newyi is calculated too */
  float *newy,
  /** Integral at frame mid time, or NULL if not needed */
  float *newyi,
  /** 2nd integral at frame mid time, or NULL if not needed */
  float *newyii,
  /** Number of PET frames */
  int newnr
) {
  int ret, fi, overlap=0, zeroframe=0;
  float petx[3], pety[3], petyi[3], petyii[3], fdur;

  if(INTEGR_TEST) printf("in finterpolate4pet()\n");

  /* Check for data */
  if(nr<1 || newnr<1) return 1;
  /* Check that programmer understood that outp data must've been allocated */
  if(newy==NULL && newyi==NULL && newyii==NULL) return 2;
  /* Check that input data is not totally outside the input data */
  if(newx2[newnr-1]<=x[0] || newx1[0]>=x[nr-1]) return 3;

  /* Check frame lengths, also for overlap and zero length frames */
  for(fi=0; fi<newnr; fi++) {
    /* Calculate frame length */
    fdur=newx2[fi]-newx1[fi];
    /* If frame length is <0, that is an error */
    if(fdur<0.0) return 4;
    if(fdur==0.0) zeroframe++;
    /* Overlap? */
    if(fi>0 && newx2[fi-1]>newx1[fi]) overlap++;
  }
  if(INTEGR_TEST>1) {
    printf("overlap := %d\n", overlap);
    printf("zeroframe := %d\n", zeroframe);
  }

  /* Interpolate and integrate one frame at a time, if there is:
     -overlap in frames
     -frames of zero length
     -only few interpolated frames
     -if only integrals are needed
     -if neither of integrals is needed, then there is no temp memory to
      do otherwise
  */
  if(overlap>0 || zeroframe>0 || newnr<=3 ||
     newy==NULL || (newyi==NULL && newyii==NULL) )
  {

    if(INTEGR_TEST>1) printf("frame-by-frame interpolation/integration\n");
    for(fi=0; fi<newnr; fi++) {
      /* Set frame start, middle and end times */
      petx[0]=newx1[fi]; petx[2]=newx2[fi]; petx[1]=0.5*(petx[0]+petx[2]);
      /* Calculate frame length */
      fdur=petx[2]-petx[0];
      /* If frame length is <0, that is an error */
      if(fdur<0.0) return 4;
      /* If frame length is 0, then use direct interpolation */
      if(fdur==0.0) {
        ret=finterpolate(x, y, nr, petx, pety, petyi, petyii, 1);
        if(ret) return 10+ret;
        if(newy!=NULL) newy[fi]=petyi[0];
        if(newyi!=NULL) newyi[fi]=petyi[0];
        if(newyii!=NULL) newyii[fi]=petyii[0];
        continue;
      }
      /* Calculate integrals at frame start, middle, and end */
      ret=finterpolate(x, y, nr, petx, NULL, petyi, petyii, 3);
      if(ret) return 20+ret;
      /* Set output integrals, if required */
      if(newyi!=NULL) newyi[fi]=petyi[1];
      if(newyii!=NULL) newyii[fi]=petyii[1];
      /* Calculate frame mean, if required */
      if(newy!=NULL) newy[fi]=(petyi[2]-petyi[0])/fdur;
    } // next frame

  } else {

    if(INTEGR_TEST>1) printf("all-frames-at-once interpolation/integration\n");
    /* Set temp array */
    float *tp; if(newyii!=NULL) tp=newyii; else tp=newyi;
    /* Integrate at frame start times */
    ret=finterpolate(x, y, nr, newx1, NULL, tp, NULL, newnr);
    if(ret) return 10+ret;
    /* Integrate at frame end times */
    ret=finterpolate(x, y, nr, newx2, NULL, newy, NULL, newnr);
    if(ret) return 10+ret;
    /* Calculate average frame value */
    for(fi=0; fi<newnr; fi++)
      newy[fi]=(newy[fi]-tp[fi])/(newx2[fi]-newx1[fi]);
    /* Calculate integrals */
    if(newyi!=NULL || newyii!=NULL) {
      for(fi=0; fi<newnr; fi++) newx1[fi]+=0.5*(newx2[fi]-newx1[fi]);
      ret=finterpolate(x, y, nr, newx1, NULL, newyi, newyii, newnr);
      if(ret) return 10+ret;
      for(fi=0; fi<newnr; fi++) newx1[fi]-=(newx2[fi]-newx1[fi]);
    }

  }

  if(INTEGR_TEST) printf("out finterpolate4pet()\n");
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
/** Integrate PET TAC data to frame mid times.
    Any of output arrays may be set to NULL if that is not needed.
    Frames must be in ascending time order. Gaps and small overlap are allowed.
    If x1[0]>0 and x1[0]<=x2[0]-x1[0], then an imaginary line is 
    drawn from (0,0) to (x[0],y[0]).
\return Returns 0 if ok.
*/
int petintegral(
  /** frame start times */
  double *x1,
  /** frame end times */
  double *x2,
  /** avg value during frame */
  double *y,
  /** number of frames */
  int nr,
  /** integrals at frame mid time */
  double *ie,
  /** 2nd integrals at frame mid time */
  double *iie
) {
  int i;
  double x, last_x, last_x2, last_y, last_integral, box_integral, half_integral;
  double gap_integral, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in petintegral()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=0.0;
  box_integral=integral=integral2=frame_len=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope */
    /*If there is a big gap in the beginning it is eliminated */
    if(i==0)if(x1[0]>x2[0]-x1[0]){last_x2=last_x=x1[0];}
    /*Integral of a possible gap between frames*/
    gap_integral=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); 
    /*Integral from the beginning of the frame to the middle of the frame */
    half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    integral=box_integral+gap_integral+half_integral; 
    /* half_integral is not added to box because it is more accurate to 
       increment integrals of full frames */
    box_integral+=gap_integral+frame_len*y[i];
    integral2+=xdist*(integral+last_integral)*0.5;
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; last_integral=integral;
  }

  if(INTEGR_TEST) printf("out petintegral()\n");
  return(0);
}

/** float version of petintegral() */
int fpetintegral(
  /** frame start times */
  float *x1,
  /** frame end times */
  float *x2,
  /** avg value during frame */
  float *y,
  /** number of frames */
  int nr,
  /** integrals at frame mid time */
  float *ie,
  /** 2nd integrals at frame mid time */
  float *iie
) {
  int i;
  float x, last_x, last_x2, last_y, last_integral, box_integral, half_integral;
  float gap_integral, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in fpetintegral()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=0.0;
  box_integral=gap_integral=integral=integral2=frame_len=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope */
    /*If there is a big gap in the beginning it is eliminated */
    if(i==0)if(x1[0]>x2[0]-x1[0]){last_x2=last_x=x1[0];}
    /*Integral of a possible gap between frames*/
    gap_integral=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x));
    /*Integral from the beginning of the frame i to the middle of the frame */
    half_integral=(x-x1[i])*(last_y+s*((x1[i]+x)/2.0-last_x));
    integral=box_integral+gap_integral+half_integral; 
    /* half_integral is not added to box because it is more accurate to 
       sum integrals of full frames */
    box_integral+=gap_integral+frame_len*y[i];
    integral2+=xdist*(integral+last_integral)*0.5;
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i];
    last_y=y[i]; last_integral=integral;
  }

  if(INTEGR_TEST) printf("out fpetintegral()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Integrate PET TAC data to frame end times.
    Any of output arrays may be set to NULL if that is not needed.
    Frames must be in ascending time order. Gaps and small overlap are allowed.
    If x1[0]>0 and x1[0]<=x2[0]-x1[0], then an imaginary line is 
    drawn from (0,0) to (x[0],y[0]).
\return Returns 0 if ok.
*/
int petintegrate2fe(
  /** frame start times */
  double *x1,
  /** frame end times */
  double *x2,
  /** avg value during frame */
  double *y,
  /** number of frames */
  int nr,
  /** values at frame end time */
  double *e,
  /** integrals at frame end time */
  double *ie,
  /** 2nd integrals at frame end time */
  double *iie
) {
  int i;
  double x, last_x, last_x2, last_y, last_integral;
  double value, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in petintegrate2fe()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(e==NULL && ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=value=integral=integral2=frame_len=s=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(e!=NULL) e[i]=value; if(ie!=NULL) ie[i]=integral;
      if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope between x[i-1] and x[i] */
    /* If there is a big gap in the beginning, it is eliminated */
    if(i==0 && x1[0]>x2[0]-x1[0]) {last_x2=last_x=x1[0];}
    integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    integral+=frame_len*y[i];
    integral2+=(x2[i]-last_x2)*(integral+last_integral)*0.5;
    if(e!=NULL && i>0) {
      value=last_y+s*(last_x2-last_x); /* value at previous frame end */
      e[i-1]=value;
    }
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i]; last_y=y[i]; last_integral=integral;
  }
  if(e!=NULL) {
    value=last_y+s*(last_x2-last_x); /* Value for the last frame */
    e[i-1]=value;
  }

  if(INTEGR_TEST) printf("out petintegrate2fe()\n");
  return(0);
}

/** float version of petintegrate2fe() */
int fpetintegrate2fe(
  /** frame start times */
  float *x1,
  /** frame end times */
  float *x2,
  /** avg value during frame */
  float *y,
  /** number of frames */
  int nr,
  /** values at frame end time */
  float *e,
  /** integrals at frame end time */
  float *ie,
  /** 2nd integrals at frame end time */
  float *iie
) {
  int i;
  float x, last_x, last_x2, last_y, last_integral;
  float value, integral, integral2, frame_len, xdist, s;

  if(INTEGR_TEST) printf("in fpetintegrate2fe()\n");
  /* Check for data */
  if(nr<1 || nr<1) return(1);
  /* Check that programmer understood that output must've been allocated */
  if(e==NULL && ie==NULL && iie==NULL) return(2);

  /* Initiate values to zero */
  last_x=last_x2=last_y=last_integral=value=integral=integral2=frame_len=s=0.0;

  for(i=0; i<nr; i++) {
    frame_len=x2[i]-x1[i]; if(frame_len<0.0) return(5);
    x=0.5*(x1[i]+x2[i]); xdist=x-last_x; if(last_x>0.0 && xdist<=0.0) return(6);
    if(x<0) {
      if(e!=NULL) e[i]=value; if(ie!=NULL) ie[i]=integral;
      if(iie!=NULL) iie[i]=integral2;
      continue;
    }
    s=(y[i]-last_y)/xdist; /* slope between x[i-1] and x[i] */
    /* If there is a big gap in the beginning, it is eliminated */
    if(i==0 && x1[0]>x2[0]-x1[0]) {last_x2=last_x=x1[0];}
    integral+=(x1[i]-last_x2)*(last_y+s*((last_x2+x1[i])/2.0-last_x)); /*gap*/
    integral+=frame_len*y[i];
    integral2+=(x2[i]-last_x2)*(integral+last_integral)*0.5;
    if(e!=NULL && i>0) {
      value=last_y+s*(last_x2-last_x); /* value at previous frame end */
      e[i-1]=value;
    }
    if(ie!=NULL) ie[i]=integral; if(iie!=NULL) iie[i]=integral2;
    last_x=x; last_x2=x2[i]; last_y=y[i]; last_integral=integral;
  }
  if(e!=NULL) {
    value=last_y+s*(last_x2-last_x); /* Value for the last frame */
    e[i-1]=value;
  }

  if(INTEGR_TEST) printf("out fpetintegrate2fe()\n");
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
#endif
