/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcfunc.h"
#include "test_tpcli.h"
/*****************************************************************************/

/*****************************************************************************/
int test_liFirstStepSize(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  double v;
  
  v=liFirstStepSize(NULL, 0); if(!doubleMatch(v, 0.0, 1.0E-20)) return 1;

  {
  double xin[]={0};
  v=liFirstStepSize(xin, 1); if(!doubleMatch(v, 0.0, 1.0E-20)) return 2;
  }

  {
  double xin[]={1};
  v=liFirstStepSize(xin, 1); if(!doubleMatch(v, 1.0, 1.0E-20)) return 3;
  }

  {
  double xin[]={2, 3};
  v=liFirstStepSize(xin, 2); if(!doubleMatch(v, 1.0, 1.0E-20)) return 4;
  }

  {
  double xin[]={2, 2};
  v=liFirstStepSize(xin, 2); if(!doubleMatch(v, 2.0, 1.0E-20)) return 5;
  }

  {
  double xin[]={2, 2, 5};
  v=liFirstStepSize(xin, 3); if(!doubleMatch(v, 3.0, 1.0E-20)) return 6;
  }

  {
  double xin[]={2, 2, 5, 5};
  v=liFirstStepSize(xin, 4); if(!doubleMatch(v, 3.0, 1.0E-20)) return 7;
  }

  {
  double xin[]={1, 2, 2, 5, 5};
  v=liFirstStepSize(xin, 5); if(!doubleMatch(v, 1.0, 1.0E-20)) return 8;
  }

  {
  double xin[]={-2, 2, 5, 5};
  v=liFirstStepSize(xin, 4); if(!doubleMatch(v, 4.0, 1.0E-20)) return 9;
  }
  
  fflush(stderr);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
int test_simSamples(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  unsigned int n, m;

  if(verbose>1) printf("\n trying with NULL input \n");
  n=simSamples(0.0, 0.0, 0.0, 0, NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(n!=0) return(1);

  if(verbose>1) printf("\n trying with invalid input \n");
  n=simSamples(0.0, 0.0, 0.0, 99999, NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(n!=0) return(2);
  n=simSamples(1.0, 0.0, 0.0, 1, NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(n!=0) return(3);


  double ist, mst, et;
  int mode; 

  mode=0;
  if(verbose>1) printf("\n testing mode %d \n", mode);
  {
    if(verbose>1) printf("\n get the nr of samples \n");
    ist=1.0; mst=10.0; et=100.0;

    if(verbose>3) printf("[%g %g %g, %d]\n", ist, mst, et, mode);
    n=simSamples(ist, mst, et, mode, NULL);
    if(verbose>3) printf("  -> %u\n", n);
    if(n!=101) return(11);

    if(verbose>1) printf("\n get the samples \n");
    double x[n];
    m=simSamples(ist, mst, et, mode, x);
    if(!doubleMatch(x[0], 0.0, 1.0E-20)) return(12);
    if(!doubleMatch(x[1], ist, 1.0E-20)) return(13);
    if(!doubleMatch(x[m-1], et, 1.0E-20)) return(14);
    if(!doubleMatch(x[m-2], et-ist, 1.0E-20)) return(15);
  }  

  mode=1;
  if(verbose>1) printf("\n testing mode %d \n", mode);
  {
    if(verbose>1) printf("\n get the nr of samples \n");
    ist=1.0; mst=10.0; et=100.0;

    if(verbose>3) printf("[%g %g %g, %d]\n", ist, mst, et, mode);
    n=simSamples(ist, mst, et, mode, NULL);
    if(verbose>3) printf("  -> %u\n", n);
    if(n!=25) return(21);

    if(verbose>1) printf("\n get the samples \n");
    double x[n];
    m=simSamples(ist, mst, et, mode, x);
    if(verbose>3) {
      printf("  -> %g", x[0]);
      for(unsigned int i=1; i<m; i++) printf(", %g", x[i]);
      printf("\n"); fflush(stdout);
    } 
    if(!doubleMatch(x[0], 0.0, 1.0E-20)) return(22);
    if(!doubleMatch(x[1], ist, 1.0E-10)) return(23);
    if(x[m-1]<et) return(24);
    if(!doubleMatch(x[m-1]-x[m-2], mst, 1.0E-02)) return(25);
  }  

  mode=2;
  if(verbose>1) printf("\n testing mode %d \n", mode);
  {
    if(verbose>1) printf("\n get the nr of samples \n");
    ist=1.0; mst=10.0; et=100.0;

    if(verbose>3) printf("[%g %g %g, %d]\n", ist, mst, et, mode);
    n=simSamples(ist, mst, et, mode, NULL);
    if(verbose>3) printf("  -> %u\n", n);
    if(n!=40) return(31);

    if(verbose>1) printf("\n get the samples \n");
    double x[n];
    m=simSamples(ist, mst, et, mode, x);
    if(verbose>3) {
      printf("  -> %g", x[0]);
      for(unsigned int i=1; i<m; i++) printf(", %g", x[i]);
      printf("\n"); fflush(stdout);
    } 
    if(!doubleMatch(x[0], 0.0, 1.0E-20)) return(32);
    if(!doubleMatch(x[1], ist, 1.0E-10)) return(33);
    if(x[m-1]<et) return(34);
    if(!doubleMatch(x[m-1]-x[m-2], mst, 1.0E-02)) return(35);
  }  

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_liInterpolate(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\n trying with NULL input \n");
  liInterpolate(NULL, NULL, 0, NULL, NULL, NULL, NULL, 0, 0, 0, 1);
  if(verbose>1) printf("did not segfault, that is good.\n");


  if(verbose>1) printf("\n\n tests against math functions \n\n");
  /* Linear interpolation cannot give absolutely correct results in these
     test, but only approximations; therefore quite high tolerances are used.
  */
  {
    int ret;
    /* function parameters */
    double p1[]={2.218984E+002, -6.676388E-001, 6.596488E+001, -4.376002E+000,
                 3.618718E+001, -3.969950E-002, 2.000000E+000};
    /* calculate function values with frequent intervals */
    int i, n=500;
    double fx[n], fy[n], fi[n];
    for(i=0; i<n; i++) fx[i]=0.1*(double)i;
    ret=mfEvalY("FENGM2", 7, p1, n, fx, fy, verbose-3);
    if(ret!=0) return(101);
    ret=mfEvalInt("FENGM2", 7, p1, n, fx, fi, verbose-3);
    if(ret!=0) return(102);

    if(verbose>1) printf("\n interpolate and integrate at original x \n");
    double iy[n], ii[n];
    ret=liInterpolate(fx, fy, n, fx, iy, ii, NULL, n, 0, 0, 
                    verbose-10);
    if(ret!=0) return 103;
    for(i=0, ret=0; i<n; i++) {
      if(!doubleMatch(fy[i], iy[i], 1.0E-08)) ret++;
      if(!doubleMatch(fi[i], ii[i], 5.0E-01)) ret+=2;
      if(ret>0) {
        if(verbose>1) printf("%g: %g vs %g ; %g vs %g\n", 
                             fx[i], fy[i], iy[i], fi[i], ii[i]);
        return 104;
      }
    }


    if(verbose>1) printf("\n interpolate and integrate at different x \n\n");
    double nx[n];
    for(i=0; i<n; i++) nx[i]=1.0+0.088888889*(double)i;
    double cy[n], ci[n];
    ret=mfEvalY("FENGM2", 7, p1, n, nx, cy, verbose-3);
    if(ret!=0) return(111);
    ret=mfEvalInt("FENGM2", 7, p1, n, nx, ci, verbose-3);
    if(ret!=0) return(112);

    if(verbose>1) printf("\n interpolate and integrate at different x \n\n");

    if(verbose>1) printf("\n se=0 \n");
    ret=liInterpolate(fx, fy, n, nx, iy, ii, NULL, n, 0, 0, 
                    verbose-10);
    if(ret!=0) return 113;
    for(i=0, ret=0; i<n; i++) {
      if(!doubleMatch(cy[i], iy[i], 1.0)) ret++;
      if(!doubleMatch(ci[i], ii[i], 5.0E-01)) ret+=2;
      if(ret>0) {
        if(verbose>1) printf("%g: %g vs %g ; %g vs %g\n", 
                             nx[i], cy[i], iy[i], ci[i], ii[i]);
        return 114;
      }
    }

    if(verbose>1) printf("\n se=1 \n");
    ret=liInterpolate(fx, fy, n, nx, iy, ii, NULL, n, 1, 0, 
                    verbose-10);
    if(ret!=0) return 121;
    for(i=0, ret=0; i<n; i++) {
      if(!doubleMatch(cy[i], iy[i], 1.0)) ret++;
      if(!doubleMatch(ci[i], ii[i], 5.0E-01)) ret+=2;
      if(ret>0) {
        if(verbose>1) printf("%g: %g vs %g ; %g vs %g\n", 
                             nx[i], cy[i], iy[i], ci[i], ii[i]);
        return 122;
      }
    }

    if(verbose>1) printf("\n se=2 \n");
    ret=liInterpolate(fx, fy, n, nx, iy, ii, NULL, n, 2, 0, 
                    verbose-10);
    if(ret!=0) return 131;
    for(i=0, ret=0; i<n; i++) {
      if(!doubleMatch(cy[i], iy[i], 1.0)) ret++;
      if(!doubleMatch(ci[i], ii[i], 5.0E-01)) ret+=2;
      if(ret>0) {
        if(verbose>1) printf("%g: %g vs %g ; %g vs %g\n", 
                             nx[i], cy[i], iy[i], ci[i], ii[i]);
        return 132;
      }
    }

    if(verbose>1) printf("\n se=3 \n");
    ret=liInterpolate(fx, fy, n, nx, iy, ii, NULL, n, 3, 1, 
                    verbose-10);
    if(ret!=0) return 141;
    for(i=0, ret=0; i<n; i++) {
      if(!doubleMatch(cy[i], iy[i], 1.0)) ret++;
      if(!doubleMatch(ci[i], ii[i], 5.0E-01)) ret+=2;
      if(ret>0) {
        if(verbose>1) printf("%g: %g vs %g ; %g vs %g\n", 
                             nx[i], cy[i], iy[i], ci[i], ii[i]);
        return 142;
      }
    }

    if(verbose>1) printf("\n se=4 \n");
    ret=liInterpolate(fx, fy, n, nx, iy, ii, NULL, n, 4, 1, 
                    verbose-10);
    if(ret!=0) return 151;
    for(i=0, ret=0; i<n; i++) {
      if(!doubleMatch(cy[i], iy[i], 1.0)) ret++;
      if(!doubleMatch(ci[i], ii[i], 5.0E-01)) ret+=2;
      if(ret>0) {
        if(verbose>1) printf("%g: %g vs %g ; %g vs %g\n", 
                             nx[i], cy[i], iy[i], ci[i], ii[i]);
        return 152;
      }
    }

  }




  if(verbose>1) printf("\n\n exact tests \n\n");

  {
  if(verbose>1) printf("\n no extrapolation #1:\n");
  int ret, i, nrin=2, nrout=1;
  double xin[]={0, 100};
  double yin[]={1, 1};
  double xout[]={1}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1}, ci[]={1}, cii[]={0.5};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1011;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1012;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1013;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1014;
  }
  }

  {
  if(verbose>1) printf("\n no extrapolation #2:\n");
  int ret, i, nrin=2, nrout=2;
  double xin[]={0, 100};
  double yin[]={1, 1};
  double xout[]={1, 2}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1}, ci[]={1, 2}, cii[]={0.5, 2};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1021;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1022;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1023;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1024;
  }
  }

  {
  if(verbose>1) printf("\n no extrapolation #3:\n");
  int ret, i, nrin=2, nrout=2;
  double xin[]={-1, 100};
  double yin[]={1, 1};
  double xout[]={0, 1}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1}, ci[]={1, 2}, cii[]={0.5, 2};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1031;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1032;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1033;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1034;
  }
  }


  {
  if(verbose>1) printf("\n step function #1: input\n");
  int ret, i, nrin=6, nrout=4;
  double xin[]={0, 1, 1, 2, 2, 100};
  double yin[]={0, 0, 1, 1, 0, 0};
  double xout[]={0, 1, 2, 3}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0, 1, 0}, ci[]={0, 0, 1, 1}, cii[]={0, 0, 0.5, 1.5};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1041;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1042;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1043;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1044;
  }
  }

  {
  if(verbose>1) printf("\n step function #2: input and output\n");
  int ret, i, nrin=6, nrout=4;
  double xin[]={0, 1, 1, 2, 2, 100};
  double yin[]={0, 0, 1, 1, 0, 0};
  double xout[]={1, 2, 2, 3}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 1, 1, 0}, ci[]={0, 1, 1, 1}, cii[]={0, 0.5, 0.5, 1.5};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1051;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1052;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1053;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1054;
  }
  }

  
  {
  if(verbose>1) printf("\n end extrapolated #1: assume zero\n");
  int ret, i, nrin=2, nrout=3;
  double xin[]={0, 1};
  double yin[]={1, 1};
  double xout[]={0, 1, 2}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1, 0}, ci[]={0, 1, 1}, cii[]={0, 0.5, 1.5};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1101;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1102;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1103;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1104;
  }
  }
  
  {
  if(verbose>1) printf("\n end extrapolated #2: assume the same as last sample\n");
  int ret, i, nrin=2, nrout=3;
  double xin[]={0, 1};
  double yin[]={1, 1};
  double xout[]={0, 1, 2}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1, 1}, ci[]={0, 1, 2}, cii[]={0, 0.5, 2};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 1, 
                    verbose-10);
  if(ret!=0) return 1111;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1112;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1113;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1114;
  }
  }
  
  /* Start extrapolated */
  
  {
  if(verbose>1) printf("\n start extrapolated #1: assume zero\n");
  int ret, i, nrin=2, nrout=3;
  double xin[]={1, 2};
  double yin[]={1, 1};
  double xout[]={-0.5, 0.5, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0, 1}, ci[]={0, 0, 0.5}, cii[]={0, 0, 0.125};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 0, 0, 
                    verbose-10);
  if(ret!=0) return 1201;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1202;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1203;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1204;
  }
  }
  
  {
  if(verbose>1) printf("\n start extrapolated #2: assume first sample\n");
  int ret, i, nrin=2, nrout=3;
  double xin[]={1, 2};
  double yin[]={1, 1};
  double xout[]={-0.5, 0.5, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 1, 1}, ci[]={0, 0.5, 1.5}, cii[]={0, 0.125, 1.125};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 1, 1, 
                    verbose-10);
  if(ret!=0) return 1211;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1212;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1213;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1214;
  }
  }
    
  {
  if(verbose>1) printf("\n start extrapolated #2b: assume first sample\n");
  int ret, i, nrin=2, nrout=3;
  double xin[]={-1, 2};
  double yin[]={1, 1};
  double xout[]={-1.5, -0.5, 0.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 1, 1}, ci[]={0, 0.5, 1.5}, cii[]={0, 0.125, 1.125};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 1, 1, 
                    verbose-10);
  if(ret!=0) return 1221;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1222;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1223;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1224;
  }
  }

  {
  if(verbose>1) printf("\n start extrapolated #3a: always assume triangle\n");
  int ret, i, nrin=2, nrout=4;
  double xin[]={1, 3};
  double yin[]={1, 1};
  double xout[]={-0.5, 0.5, 1.0, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0.5, 1, 1}, ci[]={0, 0.125, 0.5, 1}, 
         cii[]={0, 0.03125, 0.25, 0.625};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 2, 1, 
                    verbose-10);
  if(ret!=0) return 1231;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1232;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1233;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1234;
  }
  }
  
  {
  if(verbose>1) printf("\n start extrapolated #3b: always assume triangle\n");
  int ret, i, nrin=1, nrout=2;
  double xin[]={1};
  double yin[]={1};
  double xout[]={0.5, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0.5, 1}, ci[]={0.125, 1}, cii[]={0.03125, 0.625};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 2, 1, 
                    verbose-10);
  if(ret!=0) return 1241;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1242;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1243;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1244;
  }
  }
  
  {
  if(verbose>1) printf("\n start extrapolated #4a: assume triangle if allowed\n");
  int ret, i, nrin=1, nrout=2;
  double xin[]={1}; // triangle not allowed because n=1
  double yin[]={1};
  double xout[]={0.5, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 1}, ci[]={0, 0.5}, cii[]={0, 0.125};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 3, 1, 
                    verbose-10);
  if(ret!=0) return 1251;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1252;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1253;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1254;
  }
  }
  
  {
  if(verbose>1) printf("\n start extrapolated #4b: assume triangle if allowed\n");
  int ret, i, nrin=3, nrout=2;
  double xin[]={1, 1.5, 5.0}; // triangle not allowed because of gap
  double yin[]={1, 1, 1};
  double xout[]={0.5, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 1}, ci[]={0, 0.5}, cii[]={0, 0.125};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 3, 1, 
                    verbose-10);
  if(ret!=0) return 1261;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1262;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1263;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1264;
  }
  }

  {
  if(verbose>1) printf("\n start extrapolated #4c: assume triangle if allowed\n");
  int ret, i, nrin=2, nrout=4;
  double xin[]={1, 5}; // triangle is allowed because of small gap
  double yin[]={1, 1};
  double xout[]={-0.5, 0.5, 1.0, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0.5, 1, 1}, ci[]={0, 0.125, 0.5, 1}, 
         cii[]={0, 0.03125, 0.25, 0.625};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 3, 1, 
                    verbose-10);
  if(ret!=0) return 1271;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1272;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1273;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1274;
  }
  }

  /* Old tests */
  
  {
  double xin[10], yin[10], xnew1[10], yout[10], cy[10];
  int ret, i, nrin=5, nrout=5;
  /* Five timepoints (1, 5), values 1.0, 2.0,...5.0
          new frames (1.5, 2.5,.., 5.5)*/
  for(i=0; i<5; i++){
     xin[i]=i+1;
     yin[i]=i+1.0;
     xnew1[i]=i+1.5;
  }
  for(i=0; i<4; i++) cy[i]=i+1.5; 
  cy[4]=5.0;
  
  ret=liInterpolate(xin, yin, nrin, xnew1, yout, NULL, NULL, nrout, 3, 1, 
                    verbose-10);
  if(ret!=0) return 1301;
  /*  printf("%f %f %f %f %f\n", yout[0], yout[1], yout[2], yout[3], yout[4]);*/
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1302;
  }
  }
  
  {
  double xin[10], yin[10], xnew1[10], yout[10], cy[10];
  int ret, i, nrin=3, nrout=2;
  /* DATA 1.2:Three timepoints (3,5), values 3.0, 4.0, 5.0
             New frames (1.5,3.5)*/
  for(i=0; i<3; i++){
     xin[i]=i+3;
     yin[i]=i+3;
 }
  xnew1[0]=1.5; cy[0]=0.0;
  xnew1[1]=3.5; cy[1]=3.5;

  ret=liInterpolate(xin, yin, nrin, xnew1, yout, NULL, NULL, nrout, 3, 1, 
                    verbose-10);
  if(ret!=0) return 1311;
  /*  printf("%f %f %f %f %f %f\n",
      f_xin[0], f_xin[1], f_xin[2], f_yin[0], f_yin[1], f_yin[2]);*/
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1312;
  }
  }
  

  {
  if(verbose>1) printf("\n start extrapolated #5a: assume zero, then triangle\n");
  int ret, i, nrin=2, nrout=4;
  double xin[]={2, 3}; // gap is large
  double yin[]={1, 1};
  double xout[]={-0.5, 1.0, 1.5, 2.0}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0, 0.5, 1}, ci[]={0, 0, 0.125, 0.5}, 
         cii[]={0, 0, 0.03125, 0.25};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 4, 1, 
                    verbose-10);
  if(ret!=0) return 1801;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1802;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1803;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1804;
  }
  /* Again with larger gap */
  for(i=0; i<nrin; i++) xin[i]+=1.0;
  for(i=0; i<nrout; i++) xout[i]+=1.0;
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 4, 1, 
                    verbose-10);
  if(ret!=0) return 1805;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1806;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1807;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1808;
  }
  }

  {
  if(verbose>1) printf("\n start extrapolated #5b: assume zero, then triangle\n");
  int ret, i, nrin=2, nrout=4;
  double xin[]={1, 5}; // gap is small
  double yin[]={1, 1};
  double xout[]={-0.5, 0.5, 1.0, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0.5, 1, 1}, ci[]={0, 0.125, 0.5, 1}, 
         cii[]={0, 0.03125, 0.25, 0.625};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 4, 1, 
                    verbose-10);
  if(ret!=0) return 1811;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1812;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1813;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1814;
  }
  }

  {
  if(verbose>1) printf("\n start extrapolated #5c: assume zero, then triangle\n");
  int ret, i, nrin=1, nrout=4;
  double xin[]={1}; // only one sample
  double yin[]={1};
  double xout[]={-0.5, 0.5, 1.0, 1.5}, yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 0.5, 1, 1}, ci[]={0, 0.125, 0.5, 1}, 
         cii[]={0, 0.03125, 0.25, 0.625};
  ret=liInterpolate(xin, yin, nrin, xout, yout, iout, iiout, nrout, 4, 1, 
                    verbose-10);
  if(ret!=0) return 1821;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 1822;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 1823;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 1824;
  }
  }
  

  
  fflush(stderr);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_liInterpolateForPET(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("\n trying with NULL input \n");
  liInterpolateForPET(NULL, NULL, 0, NULL, NULL, NULL, NULL, NULL, 0, 0, 0, 1);
  if(verbose>1) printf("did not segfault, that is good.\n");

  {
  if(verbose>1) printf("\n easy, no extrapolation #1:\n");
  int ret, i, nrin=2, nrout=2;
  double xin[]={0, 100};
  double yin[]={1, 1};
  double x1out[]={0, 1}, x2out[]={1,3};
  double yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1}, ci[]={0.5, 2.0}, cii[]={0.125, 2};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2101;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2102;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2103;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2104;
  }

  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          4, 2, verbose-10);
  if(ret!=0) return 2105;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2106;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2107;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2108;
  }
  }


  {
  if(verbose>1) printf("no extrapolation, negative input  #1:\n");
  int ret, i, nrin=2, nrout=2;
  double xin[]={-1, 100};
  double yin[]={1, 1};
  double x1out[]={0, 1}, x2out[]={1,3};
  double yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1}, ci[]={1.5, 3.0}, cii[]={1.125, 4.5};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2121;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2122;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2123;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2124;
  }

  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          3, 1, verbose-10);
  if(ret!=0) return 2125;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2126;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2127;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2128;
  }
  }


  {
  if(verbose>1) printf("step function  #1:\n");
  int ret, i, nrin=6, nrout=3;
  double xin[]={0, 1, 1, 2, 2, 100};
  double yin[]={0, 0, 1, 1, 2, 2};
  double x1out[]={0, 1, 2}, x2out[]={1, 2, 3};
  double yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0, 1, 2}, ci[]={0, 0.5, 2}, cii[]={0, 0.125, 1.25};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2201;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2202;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2203;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2204;
  }
  }
  
  {
  if(verbose>1) printf("step function  #2:\n");
  int ret, i, nrin=6, nrout=3;
  double xin[]={0, 0.5, 0.5, 1.5, 1.5, 100};
  double yin[]={0, 0,   1,   1,   2,   2};
  double x1out[]={0, 1, 2}, x2out[]={1, 2, 3};
  double yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={0.5, 1.5, 2.0}, ci[]={0, 1, 3}, cii[]={0, 0.5, 2.5};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2211;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2212;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2213;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2214;
  }
  }
  
  {
  if(verbose>1) printf("step function  #3:\n");
  int ret, i, nrin=4, nrout=3;
  double xin[]={0,  0.1, 0.1, 100};
  double yin[]={10, 10,  0,   0};
  double x1out[]={0, 1, 2}, x2out[]={1, 2, 3};
  double yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 0, 0}, ci[]={1, 1, 1}, cii[]={0.45, 1.45, 2.45};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2221;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2222;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2223;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2224;
  }
  }
  
  {
  if(verbose>1) printf("step function with extrapolation options #4:\n");
  int ret, i, nrin=3, nrout=3;
  double xin[]={0.1, 0.1, 100};
  double yin[]={10,  0,   0};
  double x1out[]={0, 1, 2}, x2out[]={1, 2, 3};
  double yout[nrout], iout[nrout], iiout[nrout];

  {
  double cy[]={0, 0, 0}, ci[]={0, 0, 0}, cii[]={0, 0, 0};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2231;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2232;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2233;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2234;
  }
  }

  {
  double cy[]={1, 0, 0}, ci[]={1, 1, 1}, cii[]={0.45, 1.45, 2.45};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          1, 0, verbose-10);
  if(ret!=0) return 2241;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2242;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2243;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2244;
  }
  }

  {
  double cy[]={0.5, 0, 0}, ci[]={0.5, 0.5, 0.5}, cii[]={0.225, 0.725, 1.225};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          2, 0, verbose-10);
  if(ret!=0) return 2251;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2252;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2253;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2254;
  }
  }

  {
  double cy[]={0.5, 0, 0}, ci[]={0.5, 0.5, 0.5}, cii[]={0.225, 0.725, 1.225};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          3, 0, verbose-10);
  if(ret!=0) return 2261;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
    
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2262;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2263;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2264;
  }
  }

  {
  double cy[]={0.5, 0, 0}, ci[]={0.5, 0.5, 0.5}, cii[]={0.225, 0.725, 1.225};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          4, 0, verbose-10);
  if(ret!=0) return 2271;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: %g %g %g\n", i, yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2272;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2273;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2274;
  }
  
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, NULL, NULL, nrout,
                          4, 0, verbose-10);
  if(ret!=0) return 2275;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2276;
  }
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, NULL, iout, NULL, nrout,
                          4, 0, verbose-10);
  if(ret!=0) return 2277;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2278;
  }
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, NULL, NULL, iiout, nrout,
                          4, 0, verbose-10);
  if(ret!=0) return 2279;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2280;
  }
  }
  
  }
  

  {
  if(verbose>1) printf("gaps in output #1:\n");
  int ret, i, nrin=2, nrout=4;
  double xin[]={0, 100};
  double yin[]={1, 1};
  double x1out[]={0, 10, 20, 50}, x2out[]={1, 11, 30, 52};
  double yout[nrout], iout[nrout], iiout[nrout];
  double cy[]={1, 1, 1, 1}, ci[]={0.5, 10.5, 25.0, 51.0}, 
         cii[]={0.125, 55.125, 312.5, 1300.5};
  ret=liInterpolateForPET(xin, yin, nrin, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 2301;
  if(verbose>3) for(i=0; i<nrout; i++)
    printf("  i=%d: x=%g %g %g %g\n", i, 0.5*(x1out[i]+x2out[i]), 
                                      yout[i], iout[i], iiout[i]);
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cy[i], yout[i], 1.0E-10)) return 2302;
    if(!doubleMatch(ci[i], iout[i], 1.0E-10)) return 2303;
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2304;
  }
  
  nrout=1;
  double x1out2[]={50}, x2out2[]={52};
  cii[0]=1300.5;
  ret=liInterpolateForPET(xin, yin, nrin, x1out2, x2out2, NULL, NULL, iiout, 
                          nrout, 0, 0, verbose-10);
  if(ret!=0) return 2305;
  for(i=0; i<nrout; i++) {
    if(!doubleMatch(cii[i], iiout[i], 1.0E-10)) return 2305;
  }
  
  }

  
  
  if(verbose>1) printf("\ntesting with polynomial function\n");
  double pp[3]={6000, -150, 1};
  int fnr=1202;
  double px[fnr], py[fnr];
  for(int i=0; i<fnr; i++) {
    px[i]=0.05*(double)i;
    pol3eval(pp, px[i], px[i], py+i, NULL, NULL);
    //py[i]=pp[0]*px[i]+pp[1]*pow(px[i], 2.0)+pp[2]*pow(px[i], 3.0);
  }

  {
  if(verbose>1) printf("One late frame:\n");
  int ret, nrout=1; 
  double x1out[]={50}, x2out[]={60};
  double yout[nrout], iout[nrout], iiout[nrout];
  /* Compute correct results */
  double cy[nrout], ci[nrout], cii[nrout];
  pol3eval(pp, x1out[0], x2out[0], cy, NULL, NULL);
  pol3eval(pp, 0.0, 0.5*(x1out[0]+x2out[0]), NULL, ci, cii);
  ret=liInterpolateForPET(px, py, fnr, x1out, x2out, yout, iout, iiout, nrout,
                          0, 0, verbose-10);
  if(ret!=0) return 3001;
  for(int i=0; i<nrout; i++) {
    if(verbose>3) printf("  y: %g vs %g\n", cy[i], yout[i]);
    if(!doubleMatchRel(cy[i], yout[i], 1.0E-05)) return 3002;
    if(verbose>3) printf("  i: %g vs %g\n", ci[i], iout[i]);
    if(!doubleMatchRel(ci[i], iout[i], 1.0E-05)) return 3003;
    if(verbose>3) printf("  ii: %g vs %g\n", cii[i], iiout[i]);
    if(!doubleMatchRel(cii[i], iiout[i], 1.0E-05)) return 3004;
  }
  }
  

  {
  if(verbose>1) printf("Overlapping frames:\n");
  int ret, nrout=10; 
  double x1out[]={0,   1, 2, 3, 4, 7,  10, 15, 20, 40}, 
         x2out[]={2.5, 3, 4, 5, 8, 12, 20, 30, 45, 60};
  double yout[nrout], iout[nrout], iiout[nrout];
  /* Compute correct results */
  double cy[nrout], ci[nrout], cii[nrout];
  for(int i=0; i<nrout; i++) {
    pol3eval(pp, x1out[i], x2out[i], cy+i, NULL, NULL);
    pol3eval(pp, 0.0, 0.5*(x1out[i]+x2out[i]), NULL, ci+i, cii+i);
  }
  ret=liInterpolateForPET(px, py, fnr, x1out, x2out, yout, iout, iiout, nrout,
                          0, 2, verbose-10);
  if(ret!=0) return 3011;
  for(int i=0; i<nrout; i++) {
    if(verbose>3) printf("  y: %g vs %g\n", cy[i], yout[i]);
    if(!doubleMatchRel(cy[i], yout[i], 1.0E-04)) return 3012;
    if(verbose>3) printf("  i: %g vs %g\n", ci[i], iout[i]);
    if(!doubleMatchRel(ci[i], iout[i], 1.0E-04)) return 3013;
    if(verbose>3) printf("  ii: %g vs %g\n", cii[i], iiout[i]);
    if(!doubleMatchRel(cii[i], iiout[i], 1.0E-03)) return 3014;
  }
  }
  
  
  fflush(stderr);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
#if(0)
int test_liFrameMidValue(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;

  {
  if(verbose>1) printf("testing with empty data\n");
  ret=liFrameMidValue(NULL, NULL, NULL, 0, NULL, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 1;
  ret=liFrameMidValue(NULL, NULL, NULL, 100, NULL, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 2;

  double a[10], b[10], c[10], d[10];
  ret=liFrameMidValue(a, b, c, 10, NULL, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 3;
  ret=liFrameMidValue(a, b, c, 0, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 4;
  ret=liFrameMidValue(a, b, NULL, 10, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 5;
  ret=liFrameMidValue(a, NULL, c, 0, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 6;
  ret=liFrameMidValue(NULL, b, c, 0, d, verbose-1);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return 7;
  }

  {
  if(verbose>1) printf("testing with one sample\n");
  double x1[1], x2[1], y[1], y2[1];
  x1[0]=0.0; x2[0]=2.0; y[0]=100.0;
  ret=liFrameMidValue(x1, x2, y, 1, y2, verbose-1);
  if(ret!=0 || !doubleMatch(y[0], y2[0], 1.0E-20)) return 11;

  x1[0]=0.0; x2[0]=0.0; y[0]=100.0;
  ret=liFrameMidValue(x1, x2, y, 1, y2, verbose-1);
  if(ret!=0 || !doubleMatch(y[0], y2[0], 1.0E-20)) return 12;
  }

  {
  if(verbose>1) printf("testing with two samples\n");
  size_t n=2;
  double x1[n], x2[n], y[n], y2[n];
  x1[0]=0.0; x2[0]=2.0;  y[0]=100.0;
  x1[1]=4.0; x2[1]=10.0; y[1]=200.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-1);
  if(ret!=0) return 21;
  for(size_t i=0; i<n; i++) if(!doubleMatch(y[i], y2[i], 1.0E-20)) return 22;

  x1[0]=0.0; x2[0]=6.0;  y[0]=100.0;
  x1[1]=5.0; x2[1]=10.0; y[1]=200.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-1);
  if(ret!=0) return 23;
  for(size_t i=0; i<n; i++) if(!doubleMatch(y[i], y2[i], 1.0E-20)) return 24;
  }

  {
  if(verbose>1) printf("testing with 6 samples against known results\n");
  size_t i, n=6;
  double x1[n], x2[n], y[n], y2[n], ycorr[n];
  i=0; x1[i]=0.0;   x2[i]=10.0;
  i++; x1[i]=10.0;  x2[i]=30.0;
  i++; x1[i]=30.0;  x2[i]=60.0;
  i++; x1[i]=60.0;  x2[i]=100.0;
  i++; x1[i]=100.0; x2[i]=150.0;
  i++; x1[i]=150.0; x2[i]=250.0;
  i=0; y[i]=5.0;          ycorr[i]=5.0;
  i++; y[i]=25.5;         ycorr[i]=20.0;
  i++; y[i]=85.85714286;  ycorr[i]=100.0;
  i++; y[i]=76.19047619;  ycorr[i]=80.0;
  i++; y[i]=27.5;         ycorr[i]=20.0;
  i++; y[i]=10.0;         ycorr[i]=10.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 31;
  for(size_t i=0; i<n; i++) if(!doubleMatch(ycorr[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, ycorr[i], y2[i]);
    return 32;
  }

  if(verbose>1) printf("negative frame duration should lead to error\n");
  i=n-1; x1[i]=350.0; x2[i]=250.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-2);
  if(ret==0) return 41;

  if(verbose>1) printf("non-ascending x values should lead to error\n");
  i=n-1; x1[i]=50.0; x2[i]=60.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-2);
  if(ret==0) return 51;

  }


  {
  if(verbose>1) printf("testing with 4 samples against known results\n");
  size_t i, n=4;
  double x1[n], x2[n], y[n], y2[n], ycorr[n];
  i=0; x1[i]=0.0;   x2[i]=10.0;
  i++; x1[i]=10.0;  x2[i]=30.0;
  i++; x1[i]=30.0;  x2[i]=50.0;
  i++; x1[i]=50.0;  x2[i]=100.0;
  i=0; y[i]=5.0;          ycorr[i]=5.0;
  i++; y[i]=16.25;        ycorr[i]=20.0;
  i++; y[i]=10.89285714;  ycorr[i]=10.0;
  i++; y[i]=5.0;          ycorr[i]=5.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 61;
  for(size_t i=0; i<n; i++) if(!doubleMatch(ycorr[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, ycorr[i], y2[i]);
    return 62;
  }

  if(verbose>1) printf("testing with negative frame times\n");
  for(size_t i=0; i<n; i++) {x1[i]-=80.0; x2[i]-=80.0;}
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 71;
  for(size_t i=0; i<n; i++) if(!doubleMatch(ycorr[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, ycorr[i], y2[i]);
    return 72;
  }

  if(verbose>1) printf("testing with all-zero frame times\n");
  for(size_t i=0; i<n; i++) {x1[i]=0.0; x2[i]=0.0;}
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 81;
  for(size_t i=0; i<n; i++) if(!doubleMatch(y[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, y[i], y2[i]);
    return 82;
  }

  if(verbose>1) printf("testing with missing (NaN) frame times\n");
  for(size_t i=0; i<n; i++) {x1[i]=x2[i]=nan("");}
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 91;
  for(size_t i=0; i<n; i++) if(!doubleMatch(y[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, y[i], y2[i]);
    return 92;
  }
  }


  {
  if(verbose>1) printf("testing with pseudoframe TAC (all frame lengths 0)\n");
  size_t i, n=6;
  double x1[n], x2[n], y[n], y2[n];
  i=0; x1[i]=x2[i]=10.0;  y[i]=2.0;
  i++; x1[i]=x2[i]=30.0;  y[i]=200.0;
  i++; x1[i]=x2[i]=50.0;  y[i]=180.0;
  i++; x1[i]=x2[i]=100.0; y[i]=130.0;
  i++; x1[i]=x2[i]=200.0; y[i]=90.0;
  i++; x1[i]=x2[i]=500.0; y[i]=60.0;
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 101;
  /* y values should not change */
  for(size_t i=0; i<n; i++) if(!doubleMatch(y[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, y[i], y2[i]);
    return 102;
  }

  if(verbose>1) printf("testing with one NaN y value\n");
  i=3; y[i]=nan("");
  ret=liFrameMidValue(x1, x2, y, n, y2, verbose-10);
  if(ret!=0) return 103;
  /* y values should not change */
  for(size_t i=0; i<n; i++) if(!doubleMatch(y[i], y2[i], 1.0E-08)) {
    printf("y[%zu]: %E vs %E\n", i+1, y[i], y2[i]);
    return 104;
  }
  }

  
  fflush(stderr);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
#endif
/*****************************************************************************/

/*****************************************************************************/
/** Evaluate the 3rd degree polynomial for testing */
int pol3eval(
  /* Array of length three, containing the constants of the 1st, 2nd, and 3rd
     degree polynomial component */
  double *pp,
  /* Evaluation start x */
  double x1,
  /* Evaluation end x */
  double x2,
  /* Pointer to mean polynomial value between x1 and x2 */
  double *y,
  /* Pointer to definite integral between x1 and x2 */
  double *yi,
  /* Pointer to definite 2nd integral between x1 and x2 */
  double *yii
) {
  double d;
  if(pp==NULL) return 1;
  d=x2-x1; if(d<0.0) return 2;

  /* Calculate y */
  if(y!=NULL) {
    if(d<1.0E-20) {
      /* If x range is zero, then calculate polynomial at point x1 */
      *y=pp[0]*x1 + pp[1]*pow(x1, 2.0) + pp[2]*pow(x1, 3.0);
    } else {
      /* Otherwise calculate definite integral between x1 and x2 
         and divide by range */
      *y=( pp[0]*((pow(x2, 2.0) - pow(x1, 2.0))/2.0)
          +pp[1]*((pow(x2, 3.0) - pow(x1, 3.0))/3.0)
          +pp[2]*((pow(x2, 4.0) - pow(x1, 4.0))/4.0) ) / d;
    }
  }

  /* Calculate integral */
  if(yi!=NULL) {
    if(d<1.0E-20) {
      /* If x range is zero, then definite integral is zero */
      *yi=0.0;
    } else {
      /* Otherwise calculate definite integral between x1 and x2 */
      *yi= pp[0]*((pow(x2, 2.0) - pow(x1, 2.0))/2.0)
          +pp[1]*((pow(x2, 3.0) - pow(x1, 3.0))/3.0)
          +pp[2]*((pow(x2, 4.0) - pow(x1, 4.0))/4.0);
    }
  }

  /* Calculate 2nd integral */
  if(yii!=NULL) {
    if(d<1.0E-20) {
      /* If x range is zero, then definite integral is zero */
      *yii=0.0;
    } else {
      /* Otherwise calculate definite 2nd integral between x1 and x2 */
      *yii= (pp[0]/2.0)*((pow(x2, 3.0) - pow(x1, 3.0))/3.0)
           +(pp[1]/3.0)*((pow(x2, 4.0) - pow(x1, 4.0))/4.0)
           +(pp[2]/4.0)*((pow(x2, 5.0) - pow(x1, 5.0))/5.0);
    }
  }
    
  return 0;
}
/*****************************************************************************/

/*****************************************************************************/
