/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpclinopt.h"
/*****************************************************************************/

/*****************************************************************************/
int test_qr(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret, i, n, m;
  double *buf=NULL, **A=NULL, *B=NULL, *X=NULL, *C=NULL, R;

  ret=qr(NULL, 0, 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);

  
  if(verbose>1) printf("\n 2-parameter estimation \n");
  m=4; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(11);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;
  A[2][0]=3.0;  A[2][1]=9.0;
  A[3][0]=4.0;  A[3][1]=16.0;
  B[0]=0.6;
  B[1]=2.2;
  B[2]=4.8;
  B[3]=8.4;
  C[0]=0.1;
  C[1]=0.5;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(12);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(13);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation \n");
  m=4; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(21);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;  A[0][2]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;  A[1][2]=8.0;
  A[2][0]=3.0;  A[2][1]=9.0;  A[2][2]=27.0;
  A[3][0]=4.0;  A[3][1]=16.0; A[3][2]=64.0;
  B[0]=0.73;
  B[1]=3.24;
  B[2]=8.31;
  B[3]=16.72;
  C[0]=0.1;
  C[1]=0.5;
  C[2]=0.13;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(22);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(23);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 4-parameter estimation, where 4th parameter is zero \n");
  m=4; n=4;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(31);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;  A[0][2]=1.0;  A[0][3]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;  A[1][2]=8.0;  A[1][3]=16.0;
  A[2][0]=3.0;  A[2][1]=9.0;  A[2][2]=27.0; A[2][3]=81.0;
  A[3][0]=4.0;  A[3][1]=16.0; A[3][2]=64.0; A[3][3]=256.0;
  B[0]=0.73;
  B[1]=3.24;
  B[2]=8.31;
  B[3]=16.72;
  C[0]=0.1;
  C[1]=0.5;
  C[2]=0.13;
  C[3]=0.0;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(32);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(33);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, where 2nd parameter is zero \n");
  m=4; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(41);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;  A[0][2]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;  A[1][2]=8.0;
  A[2][0]=3.0;  A[2][1]=9.0;  A[2][2]=27.0;
  A[3][0]=4.0;  A[3][1]=16.0; A[3][2]=64.0;
  B[0]=0.23;
  B[1]=1.24;
  B[2]=3.81;
  B[3]=8.72;
  C[0]=0.1;
  C[1]=0.0;
  C[2]=0.13;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(42);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(43);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 2-parameter estimation, with negative parameter \n");
  m=4; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(51);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;
  A[2][0]=3.0;  A[2][1]=9.0;
  A[3][0]=4.0;  A[3][1]=16.0;
  B[0]=1.0;
  B[1]=6.0;
  B[2]=15.0;
  B[3]=28.0;
  C[0]=-1.0;
  C[1]=2.0;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(52);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); 
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(53);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 2-parameter estimation with errors \n");
  m=6; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(61);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=7.0;  A[0][1]=0.0;
  A[1][0]=3.0;  A[1][1]=1.0;
  A[2][0]=2.0;  A[2][1]=4.0;
  A[3][0]=1.0;  A[3][1]=6.0;
  A[4][0]=2.0;  A[4][1]=3.0;
  A[5][0]=6.0;  A[5][1]=2.0;
  B[0]=8.0;
  B[1]=3.0;
  B[2]=7.0;
  B[3]=6.0;
  B[4]=6.0;
  B[5]=7.0;
  C[0]=1.0;
  C[1]=1.0;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(62);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatchRel(X[i], C[i], 0.1)) {free(buf); free(A); return(63);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 2-parameter estimation with smaller errors \n");
  m=6; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(71);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=7.0;  A[0][1]=0.0;
  A[1][0]=3.0;  A[1][1]=1.0;
  A[2][0]=2.0;  A[2][1]=4.0;
  A[3][0]=1.0;  A[3][1]=6.0;
  A[4][0]=2.0;  A[4][1]=3.0;
  A[5][0]=6.0;  A[5][1]=2.0;
  B[0]=3.51;
  B[1]=2.69;
  B[2]=5.81;
  B[3]=7.69;
  B[4]=4.61;
  B[5]=5.39;
  C[0]=0.5;
  C[1]=1.2;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(72);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatchRel(X[i], C[i], 0.002)) {free(buf); free(A); return(73);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, 6 samples \n");
  m=6; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(81);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=2.0;  A[0][2]=3.0;    B[0]=3.0; 
  A[1][0]=4.0;  A[1][1]=5.0;  A[1][2]=6.0;    B[1]=9.0; 
  A[2][0]=7.0;  A[2][1]=8.0;  A[2][2]=9.0;    B[2]=15.0; 
  A[3][0]=10.0; A[3][1]=11.0; A[3][2]=12.0;   B[3]=22.0; 
  A[4][0]=13.0; A[4][1]=14.0; A[4][2]=15.0;   B[4]=27.0; 
  A[5][0]=16.0; A[5][1]=17.0; A[5][2]=-5.0;   B[5]=33.0;
  C[0]=1.08841;
  C[1]=0.923188;
  C[2]=0.0217391;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(82);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(83);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3x3 matrix \n");
  // See http://www.excel-easy.com/examples/system-of-linear-equations.html
  m=3; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(91);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=5.0;  A[0][1]=1.0;  A[0][2]=8.0;    B[0]=46.0; 
  A[1][0]=4.0;  A[1][1]=-2.0; A[1][2]=0.0;    B[1]=12.0; 
  A[2][0]=6.0;  A[2][1]=7.0;  A[2][2]=4.0;    B[2]=50.0; 
  C[0]=4.0;
  C[1]=2.0;
  C[2]=3.0;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(92);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-08)) {free(buf); free(A); return(93);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, data all zeroes \n");
  m=6; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(101);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=0.0;  A[0][1]=0.0;  A[0][2]=0.0;   B[0]=0.0; 
  A[1][0]=0.0;  A[1][1]=0.0;  A[1][2]=0.0;   B[1]=0.0; 
  A[2][0]=0.0;  A[2][1]=0.0;  A[2][2]=0.0;   B[2]=0.0; 
  A[3][0]=0.0;  A[3][1]=0.0;  A[3][2]=0.0;   B[3]=0.0; 
  A[4][0]=0.0;  A[4][1]=0.0;  A[4][2]=0.0;   B[4]=0.0; 
  A[5][0]=0.0;  A[5][1]=0.0;  A[5][2]=0.0;   B[5]=0.0;
  C[0]=0.0;
  C[1]=0.0;
  C[2]=0.0;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret==0) {free(buf); free(A); return(102);}
  if(verbose>2) {
    printf("solution not possible\n");
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, B[] all zeroes \n");
  m=6; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(111);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=6.0;  A[0][2]=3.0;   B[0]=0.0; 
  A[1][0]=2.0;  A[1][1]=5.0;  A[1][2]=3.0;   B[1]=0.0; 
  A[2][0]=3.0;  A[2][1]=4.0;  A[2][2]=3.0;   B[2]=0.0; 
  A[3][0]=4.0;  A[3][1]=3.0;  A[3][2]=3.0;   B[3]=0.0; 
  A[4][0]=5.0;  A[4][1]=2.0;  A[4][2]=3.0;   B[4]=0.0; 
  A[5][0]=6.0;  A[5][1]=1.0;  A[5][2]=3.0;   B[5]=0.0;
  C[0]=0.0;
  C[1]=0.0;
  C[2]=0.0;
  ret=qr(A, m, n, B, X, &R, NULL, NULL, NULL, NULL);
  if(ret!=0) {free(buf); free(A); return(112);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(113);} 
  }
  free(buf); free(A);


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_qrLSQ(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret, i, n, m;
  double *buf=NULL, **A=NULL, *B=NULL, *X=NULL, *C=NULL, R;

  ret=qrLSQ(NULL, NULL, NULL, 0, 0, NULL);
  if(verbose>1) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);

  
  if(verbose>1) printf("\n 2-parameter estimation \n");
  m=4; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(11);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;
  A[2][0]=3.0;  A[2][1]=9.0;
  A[3][0]=4.0;  A[3][1]=16.0;
  B[0]=0.6;
  B[1]=2.2;
  B[2]=4.8;
  B[3]=8.4;
  C[0]=0.1;
  C[1]=0.5;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(12);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(13);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation \n");
  m=4; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(21);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;  A[0][2]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;  A[1][2]=8.0;
  A[2][0]=3.0;  A[2][1]=9.0;  A[2][2]=27.0;
  A[3][0]=4.0;  A[3][1]=16.0; A[3][2]=64.0;
  B[0]=0.73;
  B[1]=3.24;
  B[2]=8.31;
  B[3]=16.72;
  C[0]=0.1;
  C[1]=0.5;
  C[2]=0.13;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(22);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(23);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 4-parameter estimation, where 4th parameter is zero \n");
  m=4; n=4;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(31);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;  A[0][2]=1.0;  A[0][3]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;  A[1][2]=8.0;  A[1][3]=16.0;
  A[2][0]=3.0;  A[2][1]=9.0;  A[2][2]=27.0; A[2][3]=81.0;
  A[3][0]=4.0;  A[3][1]=16.0; A[3][2]=64.0; A[3][3]=256.0;
  B[0]=0.73;
  B[1]=3.24;
  B[2]=8.31;
  B[3]=16.72;
  C[0]=0.1;
  C[1]=0.5;
  C[2]=0.13;
  C[3]=0.0;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(32);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(33);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, where 2nd parameter is zero \n");
  m=4; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(41);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;  A[0][2]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;  A[1][2]=8.0;
  A[2][0]=3.0;  A[2][1]=9.0;  A[2][2]=27.0;
  A[3][0]=4.0;  A[3][1]=16.0; A[3][2]=64.0;
  B[0]=0.23;
  B[1]=1.24;
  B[2]=3.81;
  B[3]=8.72;
  C[0]=0.1;
  C[1]=0.0;
  C[2]=0.13;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(42);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(43);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 2-parameter estimation, with negative parameter \n");
  m=4; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(51);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=1.0;
  A[1][0]=2.0;  A[1][1]=4.0;
  A[2][0]=3.0;  A[2][1]=9.0;
  A[3][0]=4.0;  A[3][1]=16.0;
  B[0]=1.0;
  B[1]=6.0;
  B[2]=15.0;
  B[3]=28.0;
  C[0]=-1.0;
  C[1]=2.0;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(52);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); 
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-12)) {free(buf); free(A); return(53);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 2-parameter estimation with errors \n");
  m=6; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(61);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=7.0;  A[0][1]=0.0;
  A[1][0]=3.0;  A[1][1]=1.0;
  A[2][0]=2.0;  A[2][1]=4.0;
  A[3][0]=1.0;  A[3][1]=6.0;
  A[4][0]=2.0;  A[4][1]=3.0;
  A[5][0]=6.0;  A[5][1]=2.0;
  B[0]=8.0;
  B[1]=3.0;
  B[2]=7.0;
  B[3]=6.0;
  B[4]=6.0;
  B[5]=7.0;
  C[0]=1.0;
  C[1]=1.0;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(62);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatchRel(X[i], C[i], 0.1)) {free(buf); free(A); return(63);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 2-parameter estimation with smaller errors \n");
  m=6; n=2;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(71);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=7.0;  A[0][1]=0.0;
  A[1][0]=3.0;  A[1][1]=1.0;
  A[2][0]=2.0;  A[2][1]=4.0;
  A[3][0]=1.0;  A[3][1]=6.0;
  A[4][0]=2.0;  A[4][1]=3.0;
  A[5][0]=6.0;  A[5][1]=2.0;
  B[0]=3.51;
  B[1]=2.69;
  B[2]=5.81;
  B[3]=7.69;
  B[4]=4.61;
  B[5]=5.39;
  C[0]=0.5;
  C[1]=1.2;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(72);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatchRel(X[i], C[i], 0.002)) {free(buf); free(A); return(73);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, 6 samples \n");
  m=6; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(81);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=2.0;  A[0][2]=3.0;    B[0]=3.0; 
  A[1][0]=4.0;  A[1][1]=5.0;  A[1][2]=6.0;    B[1]=9.0; 
  A[2][0]=7.0;  A[2][1]=8.0;  A[2][2]=9.0;    B[2]=15.0; 
  A[3][0]=10.0; A[3][1]=11.0; A[3][2]=12.0;   B[3]=22.0; 
  A[4][0]=13.0; A[4][1]=14.0; A[4][2]=15.0;   B[4]=27.0; 
  A[5][0]=16.0; A[5][1]=17.0; A[5][2]=-5.0;   B[5]=33.0;
  C[0]=1.08841;
  C[1]=0.923188;
  C[2]=0.0217391;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(82);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(83);} 
  }
  free(buf); free(A);


  if(verbose>1) printf("\n 3x3 matrix \n");
  // See http://www.excel-easy.com/examples/system-of-linear-equations.html
  m=3; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(91);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=5.0;  A[0][1]=1.0;  A[0][2]=8.0;    B[0]=46.0; 
  A[1][0]=4.0;  A[1][1]=-2.0; A[1][2]=0.0;    B[1]=12.0; 
  A[2][0]=6.0;  A[2][1]=7.0;  A[2][2]=4.0;    B[2]=50.0; 
  C[0]=4.0;
  C[1]=2.0;
  C[2]=3.0;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(92);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-08)) {free(buf); free(A); return(93);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, data all zeroes \n");
  m=6; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(101);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=0.0;  A[0][1]=0.0;  A[0][2]=0.0;   B[0]=0.0; 
  A[1][0]=0.0;  A[1][1]=0.0;  A[1][2]=0.0;   B[1]=0.0; 
  A[2][0]=0.0;  A[2][1]=0.0;  A[2][2]=0.0;   B[2]=0.0; 
  A[3][0]=0.0;  A[3][1]=0.0;  A[3][2]=0.0;   B[3]=0.0; 
  A[4][0]=0.0;  A[4][1]=0.0;  A[4][2]=0.0;   B[4]=0.0; 
  A[5][0]=0.0;  A[5][1]=0.0;  A[5][2]=0.0;   B[5]=0.0;
  C[0]=0.0;
  C[1]=0.0;
  C[2]=0.0;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(102);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(103);} 
  }
  free(buf); free(A);

  
  if(verbose>1) printf("\n 3-parameter estimation, B[] all zeroes \n");
  m=6; n=3;
  buf=(double*)calloc(n*m+m+n+n, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(111);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  B=buf+(m*n); X=buf+(m*n+m); C=buf+(m*n+m+n);
  A[0][0]=1.0;  A[0][1]=6.0;  A[0][2]=3.0;   B[0]=0.0; 
  A[1][0]=2.0;  A[1][1]=5.0;  A[1][2]=3.0;   B[1]=0.0; 
  A[2][0]=3.0;  A[2][1]=4.0;  A[2][2]=3.0;   B[2]=0.0; 
  A[3][0]=4.0;  A[3][1]=3.0;  A[3][2]=3.0;   B[3]=0.0; 
  A[4][0]=5.0;  A[4][1]=2.0;  A[4][2]=3.0;   B[4]=0.0; 
  A[5][0]=6.0;  A[5][1]=1.0;  A[5][2]=3.0;   B[5]=0.0;
  C[0]=0.0;
  C[1]=0.0;
  C[2]=0.0;
  ret=qrLSQ(A, B, X, m, n, &R);
  if(ret!=0) {free(buf); free(A); return(112);}
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", R, sqrt(R));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(113);} 
  }
  free(buf); free(A);


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_qr_decomp(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  int ret, i, n, m;
  double *buf=NULL, **A=NULL, **wm=NULL;


  if(verbose>1) printf("\n 3 columns (parameters), 6 rows (samples) \n");
  m=6; n=3;
  buf=(double*)calloc(2*n*m, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  wm=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL || wm==NULL) return(1);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  for(; i<2*m; i++) wm[i-m]=buf+(i*n);
  A[0][0]=1.0;  A[0][1]=2.0;  A[0][2]=3.0;
  A[1][0]=4.0;  A[1][1]=5.0;  A[1][2]=6.0; 
  A[2][0]=7.0;  A[2][1]=8.0;  A[2][2]=9.0; 
  A[3][0]=10.0; A[3][1]=11.0; A[3][2]=12.0; 
  A[4][0]=13.0; A[4][1]=14.0; A[4][2]=15.0; 
  A[5][0]=16.0; A[5][1]=17.0; A[5][2]=-5.0;
  if(verbose>2) {
    printf("\tMatrix A:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[mi][ni]);
      printf("\n");
    }
  }
  double tau[n]; 
  double chain[m];
  ret=qr_decomp(A, m, n, tau, wm, chain);
  if(ret!=0) {free(buf); free(A); free(wm); return(2);}
  if(verbose>2) {
    printf("->\tMatrix A:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[mi][ni]);
      printf("\n");
    }
    printf("tau:");
    for(int ni=0; ni<n; ni++) printf(" %g", tau[ni]);
    printf("\n");
  }
  if(!doubleMatch(A[0][0], -24.310, 0.001)) {free(buf); free(A); free(wm); return(3);} 
  if(!doubleMatch(A[0][1], -26.408, 0.001)) {free(buf); free(A); free(wm); return(3);} 
  if(!doubleMatch(A[0][2], -13.369, 0.001)) {free(buf); free(A); free(wm); return(3);} 
  if(!doubleMatch(A[1][1], -1.265, 0.001)) {free(buf); free(A); free(wm); return(3);} 
  if(!doubleMatch(A[1][2], -9.454, 0.001)) {free(buf); free(A); free(wm); return(3);} 
  if(!doubleMatch(A[2][2], -15.872, 0.001)) {free(buf); free(A); free(wm); return(3);} 

  free(buf); free(A); free(wm); 


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_qrWeight(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("Not yet tested.\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_qrWeightRm(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(verbose>1) printf("Not yet tested.\n");

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_qrSimpleLSQ(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  int ret, i, n, m;
  double *buf=NULL, **A=NULL;
  double B[20], X[10], C[10], r2;

  if(verbose>1) printf("\n 3-parameter estimation, 6 samples \n");
  m=6; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(m, sizeof(double*));
  if(buf==NULL || A==NULL) return(1);
  for(i=0; i<m; i++) A[i]=buf+(i*n);
  A[0][0]=1.0;  A[0][1]=2.0;  A[0][2]=3.0;    B[0]=3.0; 
  A[1][0]=4.0;  A[1][1]=5.0;  A[1][2]=6.0;    B[1]=9.0; 
  A[2][0]=7.0;  A[2][1]=8.0;  A[2][2]=9.0;    B[2]=15.0; 
  A[3][0]=10.0; A[3][1]=11.0; A[3][2]=12.0;   B[3]=22.0; 
  A[4][0]=13.0; A[4][1]=14.0; A[4][2]=15.0;   B[4]=27.0; 
  A[5][0]=16.0; A[5][1]=17.0; A[5][2]=-5.0;   B[5]=33.0;
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[mi][ni]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=1.08841;
  C[1]=0.923188;
  C[2]=0.0217391;
  ret=qrSimpleLSQ(A, B, m, n, X, &r2);
  if(ret!=0) {free(buf); free(A); return(2);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[mi][ni]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(3);} 
  }
  free(buf); free(A);


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_qrLH(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }
  int ret, i, n, m;
  double *buf=NULL, **A=NULL;
  double B[20], X[10], C[10], r2;

  {
  if(verbose>1) printf("\n test with invalid inputs \n");

  ret=qrLH(0, 0, NULL, NULL, NULL, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(1);

  ret=qrLH(10, 5, NULL, NULL, NULL, NULL);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(2);

  ret=qrLH(0, 0, buf, B, X, &r2);
  if(verbose>2) printf("did not segfault, that is good.\n");
  if(ret==0) return(3);
  }


  {
  if(verbose>1) printf("\n 2-parameter estimation, 4 samples \n");
  m=4; n=2;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(101);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=1.0;  B[0]=0.6; 
  A[0][1]=2.0;  A[1][1]=4.0;  B[1]=2.2; 
  A[0][2]=3.0;  A[1][2]=9.0;  B[2]=4.8; 
  A[0][3]=4.0;  A[1][3]=16.0; B[3]=8.4; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.1;
  C[1]=0.5;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(102);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(103);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-04)) {free(buf); free(A); return(104);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 3-parameter estimation, 4 samples \n");
  m=4; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(111);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=1.0;  A[2][0]=1.0;  B[0]=0.73; 
  A[0][1]=2.0;  A[1][1]=4.0;  A[2][1]=8.0;  B[1]=3.24; 
  A[0][2]=3.0;  A[1][2]=9.0;  A[2][2]=27.0; B[2]=8.31; 
  A[0][3]=4.0;  A[1][3]=16.0; A[2][3]=64.0; B[3]=16.72; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.1;
  C[1]=0.5;
  C[2]=0.13;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(112);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(113);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-04)) {free(buf); free(A); return(114);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 4-parameter estimation, 4 samples - 4th parameter is useless\n");
  m=4; n=4;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(121);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=1.0;  A[2][0]=1.0;  A[3][0]=1.0;   B[0]=0.73; 
  A[0][1]=2.0;  A[1][1]=4.0;  A[2][1]=8.0;  A[3][1]=16.0;  B[1]=3.24; 
  A[0][2]=3.0;  A[1][2]=9.0;  A[2][2]=27.0; A[3][2]=81.0;  B[2]=8.31; 
  A[0][3]=4.0;  A[1][3]=16.0; A[2][3]=64.0; A[3][3]=256.0; B[3]=16.72; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.1;
  C[1]=0.5;
  C[2]=0.13;
  C[3]=0.0;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(122);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(123);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-04)) {free(buf); free(A); return(124);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 3-parameter estimation, 4 samples - 2nd parameter is useless\n");
  m=4; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(131);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=1.0;  A[2][0]=1.0;  B[0]=0.23; 
  A[0][1]=2.0;  A[1][1]=4.0;  A[2][1]=8.0;  B[1]=1.24; 
  A[0][2]=3.0;  A[1][2]=9.0;  A[2][2]=27.0; B[2]=3.81; 
  A[0][3]=4.0;  A[1][3]=16.0; A[2][3]=64.0; B[3]=8.72; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.1;
  C[1]=0.0;
  C[2]=0.13;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(132);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(133);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-04)) {free(buf); free(A); return(134);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 2-parameter estimation, 4 samples - negative parameter\n");
  m=4; n=2;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(141);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=1.0;  B[0]=1.0; 
  A[0][1]=2.0;  A[1][1]=4.0;  B[1]=6.0; 
  A[0][2]=3.0;  A[1][2]=9.0;  B[2]=15.0; 
  A[0][3]=4.0;  A[1][3]=16.0; B[3]=28.0; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=-1.0;
  C[1]=+2.0;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(142);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(143);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-04)) {free(buf); free(A); return(144);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 3-parameter estimation, 3 samples - negative sample value\n");
  // See http://www.excel-easy.com/examples/system-of-linear-equations.html
  m=3; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(151);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=5.0;  A[1][0]= 1.0;  A[2][0]=8.0;  B[0]=46.0; 
  A[0][1]=4.0;  A[1][1]=-2.0;  A[2][1]=0.0;  B[1]=12.0; 
  A[0][2]=6.0;  A[1][2]= 7.0;  A[2][2]=4.0;  B[2]=50.0; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=4.0;
  C[1]=2.0;
  C[2]=3.0;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(152);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(153);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-04)) {free(buf); free(A); return(154);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 2-parameter estimation, 6 samples - with error\n");
  m=6; n=2;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(201);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=7.0;  A[1][0]=0.0;  B[0]=8.0; 
  A[0][1]=3.0;  A[1][1]=1.0;  B[1]=3.0; 
  A[0][2]=2.0;  A[1][2]=4.0;  B[2]=7.0; 
  A[0][3]=1.0;  A[1][3]=6.0;  B[3]=6.0; 
  A[0][4]=2.0;  A[1][4]=3.0;  B[4]=6.0; 
  A[0][5]=6.0;  A[1][5]=2.0;  B[5]=7.0; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=1.0;
  C[1]=1.0;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(202);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 0.1)) {free(buf); free(A); return(203);} 
  }
  if(!doubleMatch(correct_r2, r2, 10.0)) {free(buf); free(A); return(204);} 
  free(buf); free(A);
  }


  {
  if(verbose>1) printf("\n 2-parameter estimation, 6 samples - with smaller error\n");
  m=6; n=2;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(211);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=7.0;  A[1][0]=0.0;  B[0]=3.51; 
  A[0][1]=3.0;  A[1][1]=1.0;  B[1]=2.69; 
  A[0][2]=2.0;  A[1][2]=4.0;  B[2]=5.81; 
  A[0][3]=1.0;  A[1][3]=6.0;  B[3]=7.69; 
  A[0][4]=2.0;  A[1][4]=3.0;  B[4]=4.61; 
  A[0][5]=6.0;  A[1][5]=2.0;  B[5]=5.39; 
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.5;
  C[1]=1.2;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(212);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 0.002)) {free(buf); free(A); return(213);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0)) {free(buf); free(A); return(214);} 
  free(buf); free(A);
  }



  {
  if(verbose>1) printf("\n 3-parameter estimation, 6 samples \n");
  m=6; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(221);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=2.0;  A[2][0]=3.0;    B[0]=3.0; 
  A[0][1]=4.0;  A[1][1]=5.0;  A[2][1]=6.0;    B[1]=9.0; 
  A[0][2]=7.0;  A[1][2]=8.0;  A[2][2]=9.0;    B[2]=15.0; 
  A[0][3]=10.0; A[1][3]=11.0; A[2][3]=12.0;   B[3]=22.0; 
  A[0][4]=13.0; A[1][4]=14.0; A[2][4]=15.0;   B[4]=27.0; 
  A[0][5]=16.0; A[1][5]=17.0; A[2][5]=-5.0;   B[5]=33.0;
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=1.08841;
  C[1]=0.923188;
  C[2]=0.0217391;
  double correct_r2=0.7;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(222);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(223);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-05)) {free(buf); free(A); return(224);} 
  free(buf); free(A);
  }



  {
  if(verbose>1) printf("\n 3-parameter estimation, 6 samples - All zeroes\n");
  m=6; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(301);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=0.0;  A[1][0]=0.0;  A[2][0]=0.0;    B[0]=0.0; 
  A[0][1]=0.0;  A[1][1]=0.0;  A[2][1]=0.0;    B[1]=0.0; 
  A[0][2]=0.0;  A[1][2]=0.0;  A[2][2]=0.0;    B[2]=0.0; 
  A[0][3]=0.0;  A[1][3]=0.0;  A[2][3]=0.0;   B[3]=0.0; 
  A[0][4]=0.0;  A[1][4]=0.0;  A[2][4]=0.0;   B[4]=0.0; 
  A[0][5]=0.0;  A[1][5]=0.0;  A[2][5]=0.0;   B[5]=0.0;
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.0;
  C[1]=0.0;
  C[2]=0.0;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=1) {free(buf); free(A); return(302);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(303);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-05)) {free(buf); free(A); return(304);} 
  free(buf); free(A);
  }



  {
  if(verbose>1) printf("\n 3-parameter estimation, 6 samples - All b[] zeroes\n");
  m=6; n=3;
  buf=(double*)calloc(n*m, sizeof(double));
  A=(double**)calloc(n, sizeof(double*));
  if(buf==NULL || A==NULL) return(311);
  for(i=0; i<n; i++) A[i]=buf+i*m;
  A[0][0]=1.0;  A[1][0]=6.0;  A[2][0]=3.0;   B[0]=0.0; 
  A[0][1]=2.0;  A[1][1]=5.0;  A[2][1]=3.0;   B[1]=0.0; 
  A[0][2]=3.0;  A[1][2]=4.0;  A[2][2]=3.0;   B[2]=0.0; 
  A[0][3]=4.0;  A[1][3]=3.0;  A[2][3]=3.0;   B[3]=0.0; 
  A[0][4]=5.0;  A[1][4]=2.0;  A[2][4]=3.0;   B[4]=0.0; 
  A[0][5]=6.0;  A[1][5]=1.0;  A[2][5]=3.0;   B[5]=0.0;
  if(verbose>2) {
    printf("\tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  C[0]=0.0;
  C[1]=0.0;
  C[2]=0.0;
  double correct_r2=0.0;
  ret=qrLH(m, n, buf, B, X, &r2);
  if(ret!=0) {free(buf); free(A); return(312);}
  if(verbose>2) {
    printf("-> \tMatrix A and vector B:\n");
    for(int mi=0; mi<m; mi++) {
      for(int ni=0; ni<n; ni++) printf("\t%g", A[ni][mi]);
      printf("\t%g\n", B[mi]);
    }
    printf("\n");
  }
  if(verbose>2) {
    printf("solution:"); for(i=0; i<n; i++) printf(" %g", X[i]); printf("\n");
    printf("\nR^2= %g  R=%g\n", r2, sqrt(r2));
  }
  for(i=0; i<n; i++) {
    if(!doubleMatch(X[i], C[i], 1.0E-05)) {free(buf); free(A); return(313);} 
  }
  if(!doubleMatch(correct_r2, r2, 1.0E-05)) {free(buf); free(A); return(314);} 
  free(buf); free(A);
  }


  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
