/// @file micropetio.c
/// @brief Procedures for reading Siemens Inveon micropET images.
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcmicropet.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read microPET header and verify that it is a valid Concorde/microPET file header.
    @sa micropetExists
    @return enum tpcerror (TPCERROR_OK when successful).
 */
int micropetHeaderRead(
  /** Concorde/microPET header file name, with correct extension and path. */ 
  const char *hdrfile,
  /** Pointer to microPET header contents, which is filled in this function; 
      any previous contents are removed; enter NULL, if not needed. */
  IFT *header,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s('%s', ...)\n", __func__, hdrfile); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);

  /* Empty file name or shorter than 'x.hdr' means not a microPET header file */
  if(hdrfile==NULL || strnlen(hdrfile, 6)<5) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
    return(TPCERROR_INVALID_FILENAME);
  }

  /* Read header */
  IFT ift, *local_hdr;
  iftInit(&ift);
  if(header==NULL) local_hdr=&ift; else local_hdr=header;
  FILE *fp;
  if((fp=fopen(hdrfile, "r"))==NULL) {
    if(verbose>1) printf("  cannot open header file.\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_OPEN);
    return(TPCERROR_CANNOT_OPEN);
  }
  if(iftRead(local_hdr, fp, 2, 0, NULL)!=TPCERROR_OK) {
    if(verbose>1) printf("  cannot read header file.\n");
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    iftFree(&ift); fclose(fp);
    return(TPCERROR_INVALID_HEADER);
  }
  fclose(fp);

  /* Check that certain header parameters do exist */
  if(verbose>1) printf("  checking header contents...\n");
  if(iftFindKey(local_hdr, "version", 0)<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    iftFree(&ift); return(TPCERROR_INVALID_HEADER);
  }
  if(iftFindKey(local_hdr, "model", 0)<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    iftFree(&ift); return(TPCERROR_INVALID_HEADER);
  }
  if(iftFindKey(local_hdr, "institution", 0)<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    iftFree(&ift); return(TPCERROR_INVALID_HEADER);
  }
  if(iftFindKey(local_hdr, "number_of_dimensions", 0)<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    iftFree(&ift); return(TPCERROR_INVALID_HEADER);
  }
  iftFree(&ift);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/******************************************************************************/

/******************************************************************************/
/** @brief Verify if specified file name refers to an existing microPET file.
    @details Other image formats may have similar names (*.img and *img.hdr), thus image header 
     contents are verified here. Pixel data (*.img) is only verified to exist.
    @sa niftiExists, anaExists, micropetHeaderRead
    @return Returns 0 if it is not microPET, otherwise the major version of microPET format.
    @todo Finalize and test.
 */
int micropetExists(
  /** File name, either header file (*.img.hdr), image file (*.img), or base name without extensions
      .img, .img.hdr, .pet.img, .pet.img.hdr, .ct.img, or .ct.img.hdr */
  const char *filename,
  /** If file name refers to a microPET file, then header file name will be written in this 
      char pointer (space needs to allocated by caller); enter NULL if not needed. */
  char *hdrfile,
  /** If file name refers to a microPET file, then image file name will be written in this 
      char pointer (space needs to allocated by caller); enter NULL if not needed. */
  char *imgfile,
  /** Pointer to microPET header contents, which is filled in this function; 
      any previous contents are removed; enter NULL, if not needed. */
  IFT *header,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s('%s', ...)\n", __func__, filename); fflush(stdout);}
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);

  /* Initiate output */
  if(hdrfile!=NULL) hdrfile[0]=(char)0;
  if(imgfile!=NULL) imgfile[0]=(char)0;
  if(header!=NULL) iftFree(header);

  /* Empty file name means not a microPET file */
  if(filename==NULL || strnlen(filename, 2)<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FILENAME);
    return(0);
  }

  char local_imgfile[FILENAME_MAX], local_hdrfile[FILENAME_MAX+4];

  /* If there are extensions, what is the last extension? */
  /* Based on that, try to make image and header file names */
  char *last_extension=filenameGetExtension(filename);
  if(verbose>1) {printf("  last extension := '%s'\n", last_extension); fflush(stdout);}
  if(last_extension==NULL) { // Base name without any extension was given
    strcpy(local_imgfile, filename); strlcat(local_imgfile, ".img", FILENAME_MAX);
    strcpy(local_hdrfile, local_imgfile); strlcat(local_hdrfile, ".hdr", FILENAME_MAX);
  } else if(strcasecmp(last_extension, ".IMG")==0) { // img file name was given
    strcpy(local_imgfile, filename);
    strcpy(local_hdrfile, local_imgfile); strlcat(local_hdrfile, ".hdr", FILENAME_MAX);
  } else if(strcasecmp(last_extension, ".HDR")==0) { // header file name was given
    strcpy(local_imgfile, filename); filenameRmExtension(local_imgfile);
    strcpy(local_hdrfile, filename);
  } else { // Base name, including extra extensions, was given
    strcpy(local_imgfile, filename); strlcat(local_imgfile, ".img", FILENAME_MAX);
    strcpy(local_hdrfile, local_imgfile); strlcat(local_hdrfile, ".hdr", FILENAME_MAX);
  }
  if(verbose>1) {
    printf("  tentative image file name := '%s'\n", local_imgfile);
    printf("  tentative header file name := '%s'\n", local_hdrfile);
    fflush(stdout);
  }

  /* Check that files do exist */
  {
    int bothExist=1;
    if(!fileExist(local_hdrfile)) {
      bothExist=0;
      if(verbose>1) printf("  '%s' does not exist.\n", local_hdrfile);
    }
    if(!fileExist(local_imgfile)) {
      bothExist=0;
      if(verbose>1) printf("  '%s' does not exist.\n", local_imgfile);
    }
    if(!bothExist) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_FILE);
      return(0);
    }
  }

  /* Read header to check that this indeed is a microPET file */
  IFT ift, *local_hdr;
  iftInit(&ift);
  if(header==NULL) local_hdr=&ift; else local_hdr=header;
  if(micropetHeaderRead(local_hdrfile, local_hdr, status)!=TPCERROR_OK) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    return(0);
  }

  /* Read file version from the header */
  double version=0.0;
  if(iftGetDoubleValue(local_hdr, "version", 0, &version)<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_HEADER);
    iftFree(&ift); return(0);
  }
  iftFree(&ift);
  if(verbose>1) {printf("  version := %g\n", version); fflush(stdout);}

  /* Set output */
  if(hdrfile!=NULL) strlcpy(hdrfile, local_hdrfile, FILENAME_MAX);
  if(imgfile!=NULL) strlcpy(imgfile, local_imgfile, FILENAME_MAX);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return((int)version);
}
/*****************************************************************************/

/*****************************************************************************/
