/** @file tpcnifti.h
    @brief Header file for libtpcnifti.
    @details 
     Function are not intended to support all NIfTI and Analyze files or file properties, 
     but only those that have been found necessary in Turku PET Centre. 
     For full NIfTI support, use other libraries, e.g. niftilib <http://niftilib.sourceforge.net/>

     NIfTI-1 and NIfTI-2 documentation and source codes in <http://nifti.nimh.nih.gov/>

     Procedures in this file are not dependent on IMG struct. 

    @author Vesa Oikonen
 */
#ifndef _TPCNIFTI_H_
#define _TPCNIFTI_H_
/*****************************************************************************/

/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpcfileutil.h"
/*****************************************************************************/

/*****************************************************************************/
/** NIFTI1 header size */
#define NIFTI1_HEADER_SIZE 348
/** NIFTI1 header extender size */
#define NIFTI1_HEADER_EXTENDER_SIZE 4
/** NIFTI2 header size */
#define NIFTI2_HEADER_SIZE 540
/** NIFTI2 header extender size */
#define NIFTI2_HEADER_EXTENDER_SIZE 4
/** Analyze header size */
#define ANALYZE_HEADER_SIZE 348
/*****************************************************************************/
/** NIFTI1 units: unknown */
#define NIFTI_UNITS_UNKNOWN 0
/** NIFTI1 units: meter */
#define NIFTI_UNITS_METER   1
/** NIFTI1 units: millimetre */
#define NIFTI_UNITS_MM      2
/** NIFTI1 units: micrometer */
#define NIFTI_UNITS_MICRON  4
/** NIFTI1 units: seconds */
#define NIFTI_UNITS_SEC     8
/** NIFTI1 units: milliseconds */
#define NIFTI_UNITS_MSEC   16
/** NIFTI1 units: microseconds */
#define NIFTI_UNITS_USEC   24
/** NIFTI1 units: Hertz */
#define NIFTI_UNITS_HERTZ   32
/** NIFTI1 units: parts per million */
#define NIFTI_UNITS_PPM     40
/** NIFTI1 units: radians per second */
#define NIFTI_UNITS_RADS    48
/*****************************************************************************/
/** Analyze 7.5 data type */
#define ANALYZE_DT_NONE 0
/** Analyze 7.5 data type */
#define ANALYZE_DT_UNKNOWN 0
/** Analyze 7.5 data type, 1 bit */
#define ANALYZE_DT_BINARY 1
/** Analyze 7.5 data type, 8 bits */
#define ANALYZE_DT_UNSIGNED_CHAR 2
/** Analyze 7.5 data type, 16 bits */
#define ANALYZE_DT_SIGNED_SHORT 4
/** Analyze 7.5 data type, 32 bits */
#define ANALYZE_DT_SIGNED_INT 8
/** Analyze 7.5 data type, 32 bits */
#define ANALYZE_DT_FLOAT 16
/** Analyze 7.5 data type, 64 bits (two floats) */
#define ANALYZE_DT_COMPLEX 32
/** Analyze 7.5 data type, 64 bits */
#define ANALYZE_DT_DOUBLE 64
/** Analyze 7.5 data type */
#define ANALYZE_DT_RGB 128
/** Analyze 7.5 data type */
#define ANALYZE_DT_ALL 255

/** NIFTI1 data type (compatible with Analyze) */
#define NIFTI_DT_NONE ANALYZE_DT_NONE
/** NIFTI1 data type (compatible with Analyze) */
#define NIFTI_DT_UNKNOWN ANALYZE_DT_UNKNOWN
/** NIFTI1 data type 1 bit (compatible with Analyze) */
#define NIFTI_DT_BINARY ANALYZE_DT_BINARY
/** NIFTI1 data type 8 bits (compatible with Analyze) */
#define NIFTI_DT_UNSIGNED_CHAR ANALYZE_DT_UNSIGNED_CHAR
/** NIFTI1 data type 16 bits (compatible with Analyze) */
#define NIFTI_DT_SIGNED_SHORT ANALYZE_DT_SIGNED_SHORT
/** NIFTI1 data type 32 bits (compatible with Analyze) */
#define NIFTI_DT_SIGNED_INT ANALYZE_DT_SIGNED_INT
/** NIFTI1 data type 32 bits (compatible with Analyze) */
#define NIFTI_DT_FLOAT ANALYZE_DT_FLOAT
/** NIFTI1 data type 64 bits (compatible with Analyze) */
#define NIFTI_DT_COMPLEX ANALYZE_DT_COMPLEX
/** NIFTI1 data type 64 bits (compatible with Analyze) */
#define NIFTI_DT_DOUBLE ANALYZE_DT_DOUBLE
/** NIFTI1 data type 24 bits (compatible with Analyze) */
#define NIFTI_DT_RGB ANALYZE_DT_RGB
/** NIFTI1 data type (compatible with Analyze) */
#define NIFTI_DT_ALL ANALYZE_DT_ALL
/** NIFTI1 data type 8 bits*/
#define NIFTI_DT_SIGNED_CHAR 256
/** NIFTI1 data type 16 bits*/
#define NIFTI_DT_UNSIGNED_SHORT 512
/** NIFTI1 data type 32 bits*/
#define NIFTI_DT_UNSIGNED_INT 768
/** NIFTI1 data type 64 bits*/
#define NIFTI_DT_LONG_LONG 1024
/** NIFTI1 data type 64 bits*/
#define NIFTI_DT_UNSIGNED_LONG_LONG 1280
/** NIFTI1 data type 128 bits*/
#define NIFTI_DT_LONG_DOUBLE 1536
/** NIFTI1 data type 128 bits, complex double */
#define NIFTI_DT_DOUBLE_PAIR 1792
/** NIFTI1 data type 256 bits, complex long double */
#define NIFTI_DT_LONG_DOUBLE_PAIR 2048
/** NIFTI1 data type 32 bits, 4-byte RGBA */
#define NIFTI_DT_RGBA 2304
/*****************************************************************************/
/** NIFTI1 intent dataset */
#define NIFTI_INTENT_NONE 0
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_CORREL 2
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_TTEST 3
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_FTEST 4 
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_ZSCORE 5
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_CHISQ 6
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_BETA 7
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_BINOM 8
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_GAMMA 9
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_POISSON 10
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_NORMAL 11
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_FTEST_NONC 12
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_CHISQ_NONC 13
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_LOGISTIC 14
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_LAPLACE 15
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_UNIFORM 16
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_TTEST_NONC 17
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_WEIBULL 18
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_CHI 19
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_INVGAUSS 20
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_EXTVAL 21
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_PVAL 22
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_LOGPVAL 23 
/** NIFTI1 intent statistics */
#define NIFTI_INTENT_LOG10PVAL 24
/** NIFTI1 intent other */
#define NIFTI_INTENT_ESTIMATE 1001
/** NIFTI1 intent other */
#define NIFTI_INTENT_LABEL 1002
/** NIFTI1 intent other */
#define NIFTI_INTENT_NEURONAME 1003
/** NIFTI1 intent other */
#define NIFTI_INTENT_GENMATRIX 1004
/** NIFTI1 intent other */
#define NIFTI_INTENT_SYMMATRIX 1005
/** NIFTI1 intent other */
#define NIFTI_INTENT_DISPVECT 1006
/** NIFTI1 intent other */
#define NIFTI_INTENT_VECTOR 1007
/** NIFTI1 intent other */
#define NIFTI_INTENT_POINTSET 1008
/** NIFTI1 intent other */
#define NIFTI_INTENT_TRIANGLE 1009
/** NIFTI1 intent other */
#define NIFTI_INTENT_QUATERNION 1010
/** NIFTI1 intent other */
#define NIFTI_INTENT_DIMLESS 1011
/*****************************************************************************/
/** NIFTI1 Coordinate System: Arbitrary coordinates. */
#define NIFTI_XFORM_UNKNOWN 0
/** NIFTI1 Coordinate System: Scanner-based anatomical coordinates. */
#define NIFTI_XFORM_SCANNER_ANAT 1
/** NIFTI1 Coordinate System: Coordinates aligned to another file or "truth". */
#define NIFTI_XFORM_ALIGNED_ANAT 2
/** NIFTI1 Coordinate System: Coordinates aligned to the Talairach-Tournoux atlas. */
#define NIFTI_XFORM_TALAIRACH 3
/** NIFTI1 Coordinate System: Coordinates aligned to the MNI 152 space. */
#define NIFTI_XFORM_MNI_152 4
/*****************************************************************************/

/*****************************************************************************/
/** Nifti-1 header, 348 bytes */
typedef struct {
  /** Size of the header. Must be 348 for NIFTI-1 and Analyze 7.5, and 
      540 for NIFTI-2 (byte offset 0) */
  int sizeof_hdr;
  /** Unused. Needed for compatibility with Analyze (byte offset 4) */
  char data_type[10];
  /** Unused. Needed for compatibility with Analyze (byte offset 14) */
  char db_name[18];
  /** Unused. Value 16384 needed for compatibility with Analyze (byte offset 32) */
  int extents;
  /** Unused. Needed for compatibility with Analyze (byte offset 36) */
  short int session_error;
  /** Unused. Value 'r' needed for compatibility with Analyze (byte offset 38) */
  char regular;
  /** MRI slice ordering, encoding directions(phase, frequency, slice). (byte offset 39) */
  char dim_info;

  /** Data array dimensions; dim[0] is for the nr of dimensions,
      1,2,3 are for space (x,y,z), 4 is for time, 5 is for storing multiple values
      at each spatiotemporal voxel. (byte offset 40) */
  short int dim[8];
  /** 1st intent parameter, dependent on intent_code (byte offset 56) */
  float intent_p1;
  /** 2nd intent parameter, dependent on intent_code (byte offset 60) */
  float intent_p2;
  /** 3rd intent parameter, dependent on intent_code (byte offset 64) */
  float intent_p3;
  /** NIFTI_INTENT_* (byte offset 68). */
  short int intent_code;
  /** Data type (byte offset 70) */
  short int datatype;
  /** Nr of bits per voxel (byte offset 72) */
  short int bitpix;
  /** First slice index (byte offset 74) */
  short int slice_start;
  /** Grid spacings starting from pixdim[1]; pixdim[0] contains orientation (byte offset 76) */
  float pixdim[8];
  /** Offset into .nii file (byte offset 108) */
  float vox_offset;
  /** Data scaling: slope (byte offset 112); pixel values should be scaled as scl_slope*x + scl_inter */
  float scl_slope;
  /** Data scaling: offset (byte offset 116); pixel values should be scaled as scl_slope*x + scl_inter */
  float scl_inter;
  /** Last slice index (byte offset 120) */
  short int slice_end;
  /** Slice timing order (byte offset 122) */
  char slice_code;
  /** Units of pixdim[1..4], combination of NIFTI_UNITS_* (byte offset 123). */
  char xyzt_units;
  /** Max display intensity (byte offset 124) */
  float cal_max;
  /** Min display intensity (byte offset 128) */
  float cal_min;
  /** Time for 1 slice (byte offset 132) */
  float slice_duration;
  /** Time axis shift (byte offset 136) */
  float toffset;
  /** Unused. Needed for compatibility with Analyze. (byte offset 140) */
  int glmax;
  /** Unused. Needed for compatibility with Analyze. (byte offset 144) */
  int glmin;

  /** Free text field for study description (byte offset 148) */
  char descrip[80];
  /** Auxiliary file name (byte offset 228) */
  char aux_file[24];
  /** Use the quaternion fields, NIFTI_XFORM_UNKNOWN, NIFTI_XFORM_SCANNER_ANAT, or
      NIFTI_XFORM_ALIGNED_ANAT. (byte offset 252) */
  short int qform_code;
  /** Use of the affine fields, NIFTI_XFORM_* code (byte offset 254) */
  short int sform_code;
  /** Quaternion b parameter (byte offset 256) */
  float quatern_b;
  /** Quaternion c parameter (byte offset 260) */
  float quatern_c;
  /** Quaternion d parameter (byte offset 264) */
  float quatern_d;
  /** Quaternion x shift (byte offset 268) */
  float qoffset_x;
  /** Quaternion y shift (byte offset 272) */
  float qoffset_y;
  /** Quaternion z shift (byte offset 276) */
  float qoffset_z;
  /** 1st row affine transformation (byte offset 280) */
  float srow_x[4];
  /** 2nd row affine transformation (byte offset 296) */
  float srow_y[4];
  /** 3rd row affine transformation (byte offset 312) */
  float srow_z[4];
  /** Name or Meaning of data (Offset 0) (byte offset 328) */
  char intent_name[16];
  /** Magic string, "ni1\0" (dual file) or "n+1\0" (single file) (byte offset 344). 
      If zero, file should be treated as Analyze. */
  char magic[4];
} NIFTI_1_HEADER;

/** Nifti-2 header, 540 bytes */
typedef struct {
  /** Size of the header. Must be 540 for NIFTI-2 (byte offset 0) */
  int sizeof_hdr;
  /** Magic string, "ni2\0" (dual file) or "n+2\0" (single file), followed by 
      "\r\n\032\n" (hex 0D 0A 1A 0A). (byte offset 4). */
  char magic[8];
  /** Data type (byte offset 12) */
  int16_t datatype;
  /** Nr of bits per voxel (byte offset 14) */
  int16_t bitpix;
  /** Data array dimensions; dim[0] is for the nr of dimensions,
      1,2,3 are for space (x,y,z), 4 is for time, 5 is for storing multiple values
      at each spatiotemporal voxel. (byte offset 16) */
  int64_t dim[8];
  /** 1st intent parameter, dependent on intent_code (byte offset 80) */
  double intent_p1;
  /** 2nd intent parameter, dependent on intent_code (byte offset 88) */
  double intent_p2;
  /** 3rd intent parameter, dependent on intent_code (byte offset 96) */
  double intent_p3;
  /** Grid spacings starting from pixdim[1]; pixdim[0] contains orientation (byte offset 104) */
  double pixdim[8];
  /** Offset into .nii file (byte offset 168) */
  int64_t vox_offset;
  /** Data scaling: slope (byte offset 176); pixel values should be scaled as scl_slope*x + scl_inter */
  double scl_slope;
  /** Data scaling: offset (byte offset 184); pixel values should be scaled as scl_slope*x + scl_inter */
  double scl_inter;
  /** Max display intensity (byte offset 192) */
  double cal_max;
  /** Min display intensity (byte offset 200) */
  double cal_min;
  /** Time for 1 slice (byte offset 208) */
  double slice_duration;
  /** Time axis shift (byte offset 216) */
  double toffset;
  /** First slice index (byte offset 224) */
  int64_t slice_start;
  /** Last slice index (byte offset 232) */
  int64_t slice_end;
  /** Free text field for study description (byte offset 240) */
  char descrip[80];
  /** Auxiliary file name (byte offset 320) */
  char aux_file[24];
  /** Use the quaternion fields, NIFTI_XFORM_UNKNOWN, NIFTI_XFORM_SCANNER_ANAT, or
      NIFTI_XFORM_ALIGNED_ANAT. (byte offset 344) */
  int qform_code;
  /** Use of the affine fields, NIFTI_XFORM_* code (byte offset 348) */
  int sform_code;
  /** Quaternion b parameter (byte offset 352) */
  double quatern_b;
  /** Quaternion c parameter (byte offset 360) */
  double quatern_c;
  /** Quaternion d parameter (byte offset 368) */
  double quatern_d;
  /** Quaternion x shift (byte offset 376) */
  double qoffset_x;
  /** Quaternion y shift (byte offset 384) */
  double qoffset_y;
  /** Quaternion z shift (byte offset 392) */
  double qoffset_z;
  /** 1st row affine transformation (byte offset 400) */
  double srow_x[4];
  /** 2nd row affine transformation (byte offset 432) */
  double srow_y[4];
  /** 3rd row affine transformation (byte offset 464) */
  double srow_z[4];
  /** Slice timing order (byte offset 496) */
  int slice_code;
  /** Units of pixdim[1..4], combination of NIFTI_UNITS_* (byte offset 500). */
  int xyzt_units;
  /** NIFTI_INTENT_* (byte offset 504). */
  int intent_code;
  /** Name or Meaning of data (byte offset 508) */
  char intent_name[16];
  /** MRI slice ordering, encoding directions(phase, frequency, slice). (byte offset 524) */
  char dim_info;
  /** unused, filled with \0 (byte offset 525). */
  char unused_str[15];
} NIFTI_2_HEADER;

/** This structure represents a 4-byte string that should follow the
    binary nifti_1_header and nifti_2_header data in a NIfTI header file. 
 */
typedef struct {
 /** If the char values are {1,0,0,0}, the file is expected to contain extensions, 
     values of {0,0,0,0} imply the file does not contain extensions.
     Other sequences of values are not currently defined. */
  char extension[4];
} NIFTI_EXTENDER;

/** Combination of all NIFTI headers */
typedef struct {
  /** NIfTI-1 header */
  NIFTI_1_HEADER h1;
  /** NIfTI-2 header */
  NIFTI_2_HEADER h2;
  /** Extension; obligatory in .nii file */ 
  NIFTI_EXTENDER e;
  /** Specifies which of the NIfTI headers above is in use (1/2). */
  int n;
  /** Specifies whether data is stored on disk as little endian (1), or big endian (0). */
  int byte_order;
} NIFTI_DSR;
/*****************************************************************************/

/*****************************************************************************/
/** Analyze header, 348 bytes */
typedef struct {
  /** Size of the header. Must be 348 for NIFTI-1 and Analyze 7.5, and 
      540 for NIFTI-2 (byte offset 0) */
  int sizeof_hdr;
  /** data_type */
  char data_type[10];
  /** db_name */
  char db_name[18];
  /** Required field, should be 16384 if image file is created as contiguous;
      a minimum extent size. */
  int extents;
  /** session_error */
  short int session_error;
  /** Required field; 'r' to indicate that all images and volumes are the same size. */
  char regular;
  /** hkey_un0 */
  char hkey_un0;
  /** Array of the image dimensions; nr of dimensions (usually 4), x dimension,
      y dimension, z dimension, and time dimension (nr of image volumes). */
  short int dim[8];
  /** unused */
  short int unused8;
  /** unused */
  short int unused9;
  /** unused */
  short int unused10;
  /** unused */
  short int unused11;
  /** unused */
  short int unused12;
  /** unused */
  short int unused13;
  /** unused */
  short int unused14;
  /** Datatype: ANALYZE_DT_NONE, ANALYZE_DT_UNKNOWN, ...
      @remark Not all SW supports all data types. */
  short int datatype;
  /** Nr of bits per pixel; 1, 8, 16, 32, or 64. */
  short int bitpix;
  /** Unused */
  short int dim_un0;
  /** Pixel dimensions in mm or ms, corresponding to dim[]. */
  float pixdim[8];
  /** Byte offset in the .img file at which voxels start; negative value
      specifies that the absolute value is applied for every image in the file. */
  float vox_offset;
  /** funused1 */
  float funused1;
  /** funused2 */
  float funused2;
  /** funused3 */
  float funused3;
  /** Maximum of calibrated values. */
  float cal_max;
  /** Minimum of calibrated values. */
  float cal_min;
  /** compressed */
  float compressed;
  /** verified */
  float verified;
  /** Maximum pixel value of the whole database. */
  int glmax;
  /** Minimum pixel value of the whole database. */
  int glmin;
  /** descrip */
  char descrip[80];
  /** aux_file */
  char aux_file[24];
  /** Slice orientation for the dataset; 0=transverse unflipped, 1=coronal unflipped, 
      2=sagittal unflipped, 3=transverse flipped, 4=coronal flipped, 5=sagittal flipped. */
  char orient;
  /** originator */
  char originator[10];
  /** generated */
  char generated[10];
  /** scannum */
  char scannum[10];
  /** patient_id */
  char patient_id[10];
  /** exp_date */
  char exp_date[10];
  /** exp_time */
  char exp_time[10];
  /** hist_un0 */
  char hist_un0[3];
  /** views */
  int views;
  /** vols_added */
  int vols_added;
  /** start_field */
  int start_field;
  /** field_skip */
  int field_skip;
  /** omax */
  int omax;
  /** omin */
  int omin;
  /** smax */
  int smax;
  /** smin */
  int smin;
} ANALYZE_HEADER;

/** Combination of all Analyze header data */
typedef struct {
  /** Analyze header */
  ANALYZE_HEADER h;
  /** Specifies whether data is stored on disk as little endian (1), or big endian (0). */
  int byte_order;
} ANALYZE_DSR;
/*****************************************************************************/

/*****************************************************************************/
/* niftiname */
extern void niftiBasename(char *filename);
/*****************************************************************************/

/*****************************************************************************/
/* niftiio */
extern int niftiExists(
  const char *filename, char *hdrfile, char *imgfile, char *siffile, NIFTI_DSR *header,
  TPCSTATUS *status
);
extern int niftiReadHeader(const char *filename, NIFTI_DSR *dsr, int verbose);
extern int niftiWriteHeader(const char *filename, NIFTI_DSR *dsr, int verbose);
/*****************************************************************************/

/*****************************************************************************/
/* analyzeio */
extern int anaExists(
  const char *filename, char *hdrfile, char *imgfile, char *siffile, ANALYZE_DSR *header,
  TPCSTATUS *status
);
extern int anaReadHeader(const char *filename, ANALYZE_DSR *dsr, int verbose);
/*****************************************************************************/

/*****************************************************************************/
#endif /* TPCNIFTI */
