/** @file parres.c
 *  @brief TPC result (*.res) file format I/O functions.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpccsv.h"
#include "tpcift.h"
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write PAR data into specified file pointer in result (RES) file format.
    @sa parWrite
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int parWriteRES(
  /** Pointer to source PAR struct, contents of which are to be written. */
  PAR *par,
  /** Output file pointer. */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(verbose>0) {printf("%s():\n", __func__); fflush(stdout);}
  if(par==NULL || par->tacNr<1 || par->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  int i, j, n;
  char tmp[256];

  /* RES format has quite strict rules on the header, therefore 
     it is written here directly and not through IFT struct etc. */

  /* Analysis program name (obligatory, with (c)) */
  i=iftFindKey(&par->h, "program", 0);
  if(i<0 || !strcasestr(par->h.item[i].value, "(c)")) {
    //statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    //return TPCERROR_INVALID_VALUE;
    n=fprintf(fp, "results (c) none\n\n");
  } else {
    n=fprintf(fp, "%s\n\n", par->h.item[i].value);
  }
  if(n<6) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write calculation date and time (obligatory) */
  i=iftFindKey(&par->h, "analysis_time", 0);
  if(i<0) {
    //statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_VALUE);
    //return TPCERROR_INVALID_VALUE;
    time_t t=time(NULL);
    fprintf(fp, "Date:\t%s\n", ctime_r_int(&t, tmp));
  } else {
    fprintf(fp, "Date:\t%s\n", par->h.item[i].value);
  }

  /* Write the studynr (optional) */
  i=iftFindKey(&par->h, "studynr", 0);
  if(i<0) i=iftFindKey(&par->h, "study_number", 0);
  if(i>=0) fprintf(fp, "Study:\t%s\n", par->h.item[i].value);

  /* Write the names of the original datafiles (optional) */
  i=iftFindKey(&par->h, "datafile", 0);
  if(i>=0) fprintf(fp, "Data file:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "plasmafile", 0);
  if(i>=0) fprintf(fp, "Plasma file:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "plasmafile2", 0);
  if(i>=0) fprintf(fp, "2nd Plasma file:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "bloodfile", 0);
  if(i>=0) fprintf(fp, "Blood file:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "reffile", 0);
  if(i>=0) fprintf(fp, "Reference file:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "refname", 0);
  if(i>=0) fprintf(fp, "Reference region:\t%s\n", par->h.item[i].value);

  /* Write data range, nr, and fit method (optional) */
  if(!isnan(par->r[0].start) && !isnan(par->r[0].end))
    fprintf(fp, "Data range:\t%g - %g min\n", par->r[0].start, par->r[0].end);
  if(par->r[0].dataNr>0)
    fprintf(fp, "Data nr:\t%d\n", par->r[0].dataNr);
  i=iftFindKey(&par->h, "fitmethod", 0);
  if(i>=0) fprintf(fp, "Fit method:\t%s\n", par->h.item[i].value);

  /* Write constants (optional) */
  i=iftFindKey(&par->h, "density", 0);
  if(i>=0) fprintf(fp, "Tissue density:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "LC", 0);
  if(i>=0) fprintf(fp, "Lumped constant:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "concentration", 0);
  if(i>=0) fprintf(fp, "Concentration:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "beta", 0);
  if(i>=0) fprintf(fp, "Beta:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "Vb", 0);
  if(i>=0) fprintf(fp, "Vb:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "fA", 0);
  if(i>=0) fprintf(fp, "fA:\t%s\n", par->h.item[i].value);
  i=iftFindKey(&par->h, "E", 0);
  if(i<0) i=iftFindKey(&par->h, "extraction", 0);
  if(i>=0) fprintf(fp, "Extraction:\t%s\n", par->h.item[i].value);

  /* Weighting (obligatory) */
  i=iftFindKey(&par->h, "weighting", 0);
  if(i>=0) fprintf(fp, "Weighting:\t%s\n", par->h.item[i].value);
  else fprintf(fp, "Weighting:\t%s\n", "unknown");

  /* Should column be printed as integers (0), floats (1) or exponentials (2)? */
  int partype[par->parNr]; /* 0=int, 1=double, 2=exp */
  for(i=0; i<par->parNr; i++) partype[i]=parPrintType(par, i);

  /* Title line (obligatory) */
  fprintf(fp, "\n%s", "Region");
  for(i=0; i<par->parNr; i++) {
    if(strlen(par->n[i].name)<1) strcpy(tmp, ".");
    else strcpy(tmp, par->n[i].name);
    fprintf(fp, "\t%s", tmp);
  }
  if(parIsWSS(par)) fprintf(fp, "\tWSS");
  fprintf(fp, "\n");

  /* Write units, if they exist, as comment line */
  for(i=j=0; i<par->parNr; i++) if(par->n[i].unit!=UNIT_UNKNOWN) j++;
  if(j>0) {
    fprintf(fp, "%s", "# Units:");
    for(i=0; i<par->parNr; i++) {
      fprintf(fp, "\t%s", unitName(par->n[i].unit));
    }
    if(parIsWSS(par)) fprintf(fp, "\t.");
    fprintf(fp, "\n");
  }


  /* Write regional results */
  for(i=0; i<par->tacNr; i++) {
    if(strlen(par->r[i].name)>0) {
      if(!roinameSubpart(par->r[i].name, "_- ", 0, tmp, 7)) strcpy(tmp, ".");
      fprintf(fp, "%s ", tmp);
      if(!roinameSubpart(par->r[i].name, "_- ", 1, tmp, 7)) strcpy(tmp, ".");
      fprintf(fp, "%s ", tmp);
      if(!roinameSubpart(par->r[i].name, "_- ", 2, tmp, 7)) strcpy(tmp, ".");
      fprintf(fp, "%s", tmp);
    } else {
      fprintf(fp, "tac%d . .", 1+i);
    }
    for(j=0; j<par->parNr; j++) {
      if(isnan(par->r[i].p[j])) {fprintf(fp, "\t."); continue;}
      switch(partype[j]) {
        case 0: fprintf(fp, "\t%.0f", par->r[i].p[j]); break;
        case 1:
          if(par->r[i].p[j]>=0) n=4; else n=3;
          fprintf(fp, "\t%.*f", n, par->r[i].p[j]);
          break;
        default:
          if(par->r[i].p[j]>=0) n=4; else n=3;
          fprintf(fp, "\t%.*e", n, par->r[i].p[j]);
        break;
      }
    }
    if(parIsWSS(par)) {
      if(isnan(par->r[i].wss)) fprintf(fp, "\t."); 
      else fprintf(fp, "\t%g", par->r[i].wss);
    }
    fprintf(fp, "\n");
    /* Write SD's, if they exist */
    for(j=n=0; j<par->parNr; j++) if(!isnan(par->r[i].sd[j])) n++;
    if(n>0) {
      fprintf(fp, "SD . .");
      for(j=0; j<par->parNr; j++) {
        if(!isnan(par->r[i].sd[j])) {
          switch(partype[j]) {
            case 0: fprintf(fp, "\t%.0f", par->r[i].sd[j]); break;
            case 1:
              if(par->r[i].sd[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*f", n, par->r[i].sd[j]);
              break;
            default:
              if(par->r[i].sd[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*e", n, par->r[i].sd[j]);
              break;
          }
        } else {
          fprintf(fp, "\t.");
        }
      }
      if(parIsWSS(par)) fprintf(fp, "\t.");
      fprintf(fp, "\n");
    }
    /* Write lower confidence limits, if they exist */
    for(j=n=0; j<par->parNr; j++) if(!isnan(par->r[i].cl1[j])) n++;
    if(n>0) {
      fprintf(fp, "CL 95%% Lower");
      for(j=0; j<par->parNr; j++) {
        if(!isnan(par->r[i].cl1[j])) {
          switch(partype[j]) {
            case 0: fprintf(fp, "\t%.0f", par->r[i].cl1[j]); break;
            case 1:
              if(par->r[i].cl1[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*f", n, par->r[i].cl1[j]);
              break;
            default:
              if(par->r[i].cl1[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*e", n, par->r[i].cl1[j]);
              break;
          }
        } else {
          fprintf(fp, "\t.");
        }
      }
      if(parIsWSS(par)) fprintf(fp, "\t.");
      fprintf(fp, "\n");
    }
    /* Write upper confidence limits, if they exist */
    for(j=n=0; j<par->parNr; j++) if(!isnan(par->r[i].cl2[j])) n++;
    if(n>0) {
      fprintf(fp, "CL 95%% Upper");
      for(j=0; j<par->parNr; j++) {
        if(!isnan(par->r[i].cl2[j])) {
          switch(partype[j]) {
            case 0: fprintf(fp, "\t%.0f", par->r[i].cl2[j]); break;
            case 1:
              if(par->r[i].cl2[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*f", n, par->r[i].cl2[j]);
              break;
            default:
              if(par->r[i].cl2[j]>=0) n=4; else n=3;
              fprintf(fp, "\t%.*e", n, par->r[i].cl2[j]);
              break;
          }
        } else {
          fprintf(fp, "\t.");
        }
      }
      if(parIsWSS(par)) fprintf(fp, "\t.");
      fprintf(fp, "\n");
    }
  } /* next region */

  
  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read result (RES) data into PAR struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
 */
int parReadRES(
  /** Pointer to target PAR struct. */
  PAR *par,
  /** Pointer to CSV from which data is read. */
  CSV *csv,
  /** Pointer to IFT struct which contains whole result file data, in case the data could not be
      read into CSV (which happens when spaces were used as field separator). */
  IFT *ift,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}
  if(par==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  parFree(par);
  
  if(csv==NULL || ift==NULL || csv->row_nr<1 || csv->col_nr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  

  if(verbose>10) {
    printf("\n CSV contents shown with separator ; \n\n");
    char sep=csv->separator;
    csv->separator=';'; csvWrite(csv, 0, stdout, NULL); csv->separator=sep;
    printf("\n IFT contents \n\n");
    iftWrite(ift, stdout, NULL);
  }

  int i, ret, titlerow;
  int ri, ci, parNr=0, tacNr=0;
  char *cptr;

  /* Find the result title line; get initial parameter and tac nr */
  i=0; if(verbose>2) printf("estimating parNr and tacNr\n"); 
  while(i<csv->nr) {
    i=csvFindField(csv, "Region", i); if(i<0) break; 
    if(i>=0 && csv->c[i].col==0) break;
    i++;
  }
  if(i<0 || i==csv->nr) {
    parFree(par);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  titlerow=csv->c[i].row;
  /* Get parameter nr */
  parNr=csvRowLength(csv, titlerow)-1;
  // sometimes empty cells in the end
  ci=parNr; while(ci>0 && strlen(csv->c[ci].content)<1) {ci--; parNr--;}
  // sometimes an empty cell before parameter names
  if(strlen(csv->c[i+1].content)<1) parNr--;
  /* Get region nr */
  ri=1+titlerow; ci=0;
  cptr=csvCell(csv, ri, ci);
  while(cptr!=NULL) {
    if(csv->separator!=' ') {
      if(strncasecmp(cptr, "SD . .", 3) && strncasecmp(cptr, "CL 95% ", 7)) tacNr++;
    } else {
      if(strcasecmp(cptr, "SD") && strcasecmp(cptr, "CL")) tacNr++;
    }
    ri++; cptr=csvCell(csv, ri, ci);
  }
  if(verbose>2) {
    printf("  parNr := %d\n", parNr);
    printf("  tacNr := %d\n", tacNr);
  }
  if(parNr<1 || tacNr<1) {
    parFree(par);
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Allocate space for parameters */
  ret=parAllocate(par, parNr, tacNr); if(ret!=TPCERROR_OK) {
    parFree(par);
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }
  par->parNr=parNr; par->tacNr=tacNr;

  /* Get program name from the first non-comment IFT field */
  if(verbose>2) printf("searching obligatory program name\n");
  i=0; 
  while(ift->item[i].comment && i<ift->keyNr && ift->item[i].key!=NULL) i++;
  if(verbose>3) {
    printf("ift->item[%d].value := '%s'\n", i, ift->item[i].value);
  }
  if(i==ift->keyNr || !strcasestr(ift->item[i].value, "(c)")) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_UNSUPPORTED);
    return TPCERROR_UNSUPPORTED;
  }
  iftPut(&par->h, "program", ift->item[i].value, 0, NULL);

  /* Get analysis time from IFT struct */
  i=iftFindKey(ift, "Date", 0);
  if(i>=0) iftPut(&par->h, "analysis_time", ift->item[i].value, 0, NULL);

  /* Study number */
  i=iftFindKey(ift, "Study", 0);
  if(i>=0) iftPut(&par->h, "studynr", ift->item[i].value, 0, NULL);

  /* Optional filenames etc */
  i=iftFindKey(ift, "Data file", 0);
  if(i>=0) iftPut(&par->h, "datafile", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Plasma file", 0);
  if(i>=0) iftPut(&par->h, "plasmafile", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "2nd plasma file", 0);
  if(i>=0) iftPut(&par->h, "plasmafile2", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Blood file", 0);
  if(i>=0) iftPut(&par->h, "bloodfile", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Reference file", 0);
  if(i>=0) iftPut(&par->h, "reffile", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Reference region", 0);
  if(i>=0) iftPut(&par->h, "refname", ift->item[i].value, 0, NULL);

  /* Fit method (optional) */
  i=iftFindKey(ift, "Fit method", 0);
  if(i>=0) iftPut(&par->h, "fitmethod", ift->item[i].value, 0, NULL);

  /* Constants (optional) */
  i=iftFindKey(ift, "Tissue density", 0);
  if(i>=0) iftPut(&par->h, "density", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Lumped constant", 0);
  if(i>=0) iftPut(&par->h, "LC", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Concentration", 0);
  if(i>=0) iftPut(&par->h, "concentration", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Beta", 0);
  if(i>=0) iftPut(&par->h, "beta", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Vb", 0);
  if(i>=0) iftPut(&par->h, "Vb", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "fA", 0);
  if(i>=0) iftPut(&par->h, "fA", ift->item[i].value, 0, NULL);
  i=iftFindKey(ift, "Extraction", 0);
  if(i>=0) iftPut(&par->h, "E", ift->item[i].value, 0, NULL);

  /* Weighting (obligatory) */
  if(verbose>2) printf("searching obligatory information on weighting\n");
  if(iftSearchValue(ift, "not weighted", 0)>=0) {
    iftPut(&par->h, "weighting", "no", 0, NULL);
  } else if(iftSearchValue(ift, "was weighted", 0)>=0) {
    iftPut(&par->h, "weighting", "yes", 0, NULL);
  } else {
    i=iftFindKey(ift, "Weighting", 0);
    if(i>=0) { 
      iftPut(&par->h, "weighting", ift->item[i].value, 0, NULL);
    } else {
      parFree(par);
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
      return TPCERROR_INVALID_FORMAT;
    }
  }

  /* Read parameter names (possibly reorganized later) */
  ri=titlerow; ci=1; cptr=csvCell(csv, ri, ci);
  if(strlen(cptr)<1) {ci++; cptr=csvCell(csv, ri, ci);}  
  for(i=0; i<parNr; i++) {
    strlcpy(par->n[i].name, cptr, MAX_PARNAME_LEN);
    ci++; cptr=csvCell(csv, ri, ci);
  }

  /* Read parameter values for each region */
  int tac=0;
  double *dptr;
  ri=1+titlerow; ret=TPCERROR_OK;
  do {
    ci=0; cptr=csvCell(csv, ri, ci); 
    if(cptr==NULL) {if(tac<tacNr) ret=TPCERROR_INVALID_FORMAT; break;}
    //printf("  cell='%s'\n", cptr);
    //printf("  ri=%d ci=%d tac=%d\n", ri, ci, tac);
    /* check if this is SD line */
    dptr=NULL;
    if(csv->separator==' ') {
      if(!strcasecmp(cptr, "SD")) {
        if(tac<1) {ret=TPCERROR_INVALID_FORMAT; break;}
        ci=3; dptr=par->r[tac-1].sd;
      } else if(!strcasecmp(cptr, "CL")) {
        if(tac<1) {ret=TPCERROR_INVALID_FORMAT; break;}
        if(!strcasecmp(csvCell(csv, ri, 2), "Lower")) {
          ci=3; dptr=par->r[tac-1].cl1;
        } else if(!strcasecmp(csvCell(csv, ri, 2), "Upper")) {
          ci=3; dptr=par->r[tac-1].cl2;
        }
      }
    } else {
      if(!strncasecmp(cptr, "SD . .", 3)) {
        if(tac<1) {ret=TPCERROR_INVALID_FORMAT; break;}
        ci=1; dptr=par->r[tac-1].sd;
      } else if(!strncasecmp(cptr, "CL 95% Lower", 8)) {
        if(tac<1) {ret=TPCERROR_INVALID_FORMAT; break;}
        ci=1; dptr=par->r[tac-1].cl1;
      } else if(!strncasecmp(cptr, "CL 95% Upper", 8)) {
        if(tac<1) {ret=TPCERROR_INVALID_FORMAT; break;}
        ci=1; dptr=par->r[tac-1].cl2;
      }
    }
    if(dptr!=NULL) {
      //printf("  sd or cl\n");
      for(i=0; i<parNr; i++) {
        cptr=csvCell(csv, ri, ci);
        if(cptr==NULL) {ret=TPCERROR_INVALID_FORMAT; break;}
        dptr[i]=atofVerified(cptr);
        ci++;
      }
      ri++; continue;
    }
    if(tac>=tacNr) break;
    //printf("  param\n");
    /* copy tac name */
    strlcpy(par->r[tac].name, cptr, MAX_TACNAME_LEN);
    if(csv->separator==' ') {
      ci++; cptr=csvCell(csv, ri, ci);
      strlcat(par->r[tac].name, " ", MAX_TACNAME_LEN);
      strlcat(par->r[tac].name, cptr, MAX_TACNAME_LEN);
      ci++; cptr=csvCell(csv, ri, ci);
      strlcat(par->r[tac].name, " ", MAX_TACNAME_LEN);
      strlcat(par->r[tac].name, cptr, MAX_TACNAME_LEN);
    }
    /* copy parameter values */
    ci++; //cptr=csvCell(csv, ri, ci); //if(strlen(cptr)<1) ci++;
    for(i=0; i<parNr; i++) {
      cptr=csvCell(csv, ri, ci);
      if(cptr==NULL) {ret=TPCERROR_INVALID_FORMAT; break;}
      par->r[tac].p[i]=atofVerified(cptr);
      ci++;
    }
    if(ret!=TPCERROR_OK) break;
    ri++; tac++;
  } while(1);
  if(ret!=TPCERROR_OK) {
    parFree(par);
    statusSet(status, __func__, __FILE__, __LINE__, ret);
    return ret;
  }

  /* Read parameter units, if available */
  i=iftFindKey(ift, "Units", 0);
  if(i>0) {
    int n=strTokenNr(ift->item[i].value, " \t");
    if(n>parNr) n=parNr;
    char tmp[256];
    for(int j=0; j<n; j++) {
      strTokenNCpy(ift->item[i].value, " \t", 1+j, tmp, 256);
      par->n[j].unit=unitIdentify(tmp);
    }
  }

  /* Fix region names (mainly, remove dots) */
  {
    int i, tac;
    char fnsp[3][MAX_TACNAME_LEN+1];
    for(tac=0; tac<par->tacNr; tac++) {
      for(i=0; i<3; i++) {
        if(roinameSubpart(par->r[tac].name, "_ ", i, fnsp[i], MAX_TACNAME_LEN)==NULL ||
           strcmp(fnsp[i], ".")==0)
          strcpy(fnsp[i], "");
      }
      strcpy(par->r[tac].name, fnsp[0]);
      if(strlen(fnsp[1]) || strlen(fnsp[2])) strcat(par->r[tac].name, "_");
      if(strlen(fnsp[1])) strcat(par->r[tac].name, fnsp[1]);
      if(strlen(fnsp[2])) {
        strcat(par->r[tac].name, "_"); strcat(par->r[tac].name, fnsp[2]);}
    }
  }

  /* Data range (optional) */
  i=iftFindKey(ift, "Data range", 0);
  if(i>=0) {
    double t1, t2;
    char tmp[256];
    int n;
    n=sscanf(ift->item[i].value, "%lf - %lf %s", &t1, &t2, tmp);
    if(n==3 && unitIdentify(tmp)==UNIT_SEC) {
      t1/=60.; t2/=60.;
    }
    if(n>=2) for(i=0; i<par->tacNr; i++) {par->r[i].start=t1; par->r[i].end=t2;}
  }


  /* Data nr (optional) */
  i=iftFindKey(ift, "Data nr", 0);
  if(i>=0) {
    int n; n=atoi(ift->item[i].value);
    for(i=0; i<par->tacNr; i++) par->r[i].dataNr=n;
  }

  /* If one 'parameter' is named WSS or SS, then move its contents to
     correct place inside the struct */
  ci=-1;
  for(i=0; i<par->parNr; i++) {
    if(!strcasecmp(par->n[i].name, "WSS") || !strcasecmp(par->n[i].name, "SS"))
      ci=i;
  }
  if(ci>=0 && ci<par->parNr && par->parNr>1) {
    if(verbose>4) printf("column %d contains WSS, moving to correct place\n", 1+ci);
    for(i=0; i<par->tacNr; i++) par->r[i].wss=par->r[i].p[ci];
    /* delete the parameter column */
    parDeletePar(par, ci);
  }

  /* Set format in struct */
  par->format=PAR_FORMAT_RES;
  
  /* Quit */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Determine whether the parameter should be printed as
    integer (0), float (1), or exponential (2).
    @return Returns the code, or <0 in case of an error.
 */
int parPrintType(
  /** Pointer to the PAR structure. */
  PAR *par,
  /** Index of the parameter to test. */
  int parIndex
) {
  int vi, partype=0;
  double x, m=0.0, pint;

  if(par==NULL || par->tacNr<1 || parIndex<0 || parIndex>=par->tacNr)
    return(-1);
  for(vi=0; vi<par->tacNr; vi++) {
    x=par->r[vi].p[parIndex];
    if(modf(x, &pint)!=0.0) partype=1;
    x=fabs(x); if(x>m) m=x;
  }
  if(partype==1 && (m>=10.0 || m<0.1)) partype=2;
  return(partype);
}
/*****************************************************************************/

/*****************************************************************************/
