/** @file parxmlio.c
 *  @brief I/O functions for model parameter XML format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
/** Write PAR data into specified file pointer in Excel compatible XML format.
 *  @sa parWrite, parRead, parWriteCSV
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int parWriteXML(
  /** Pointer to source PAR struct, contents of which are to be written */
  PAR *par,
  /** File pointer */
  FILE *fp,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>0) printf("%s()\n", __func__);
  if(fp==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  if(par==NULL || par->tacNr<1 || par->parNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }

  /* Calculate column number */
  int colNr=par->parNr+1; // including tac name
  if(parIsModel(par)>0) colNr++;
  for(int pi=0; pi<par->parNr; pi++) {
    if(parSDWithPar(par, pi)) colNr++;
    if(parCLWithPar(par, pi)) colNr+=2;
  }
  if(parIsWSS(par)) colNr++;
  if(parIsFitRange(par)>0) colNr+=2;
  if(parIsDataNr(par)>0) colNr++;
  if(parIsFitNr(par)>0) colNr++;
  if(verbose>1) printf("  colNr := %d\n", colNr);

  /* Write XML header */
  if(verbose>2) printf("writing XML header\n");
  int n=fprintf(fp, "<?xml version=\"1.0\"?>\n");
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  fprintf(fp, "<ss:Workbook xmlns:ss=\"urn:schemas-microsoft-com:office:spreadsheet\">\n");
  fprintf(fp, "  <ss:Styles>\n");
  fprintf(fp, "    <ss:Style ss:ID=\"1\">\n");
  fprintf(fp, "      <ss:Font ss:Bold=\"1\"/>\n");
  fprintf(fp, "    </ss:Style>\n");
  fprintf(fp, "  </ss:Styles>\n");
  fprintf(fp, "  <ss:Worksheet ss:Name=\"Sheet1\">\n");
  fprintf(fp, "    <ss:Table>\n");

  /* Set column widths */
  for(int i=0; i<colNr; i++) fprintf(fp, "      <ss:Column ss:Width=\"80\"/>\n");

  /* Write the title line */
  if(verbose>2) printf("writing title line\n");
  fprintf(fp, "      <ss:Row ss:StyleID=\"1\">\n");
  fprintf(fp, "        <ss:Cell>\n");
  fprintf(fp, "          <ss:Data ss:Type=\"String\">Parameters</ss:Data>\n");
  fprintf(fp, "        </ss:Cell>\n");
  /* model id */
  if(parIsModel(par)>0) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">Model</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  } 
  /* parameter names with units, and SD and/or CLs, if available */
  for(int pi=0; pi<par->parNr; pi++) {
    if(par->n[pi].unit==UNIT_UNKNOWN) {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s</ss:Data>\n", par->n[pi].name);
      fprintf(fp, "        </ss:Cell>\n");
    } else {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s[%s]</ss:Data>\n",
              par->n[pi].name, unitName(par->n[pi].unit));
      fprintf(fp, "        </ss:Cell>\n");
    }
    if(parSDWithPar(par, pi)) {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"String\">SD</ss:Data>\n");
      fprintf(fp, "        </ss:Cell>\n");
    }
    if(parCLWithPar(par, pi)) {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"String\">95%%CL1</ss:Data>\n");
      fprintf(fp, "        </ss:Cell>\n");
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"String\">95%%CL2</ss:Data>\n");
      fprintf(fp, "        </ss:Cell>\n");
    }
  }
  if(parIsWSS(par)) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">WSS</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  }
  if(parIsFitRange(par)>0) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">Start[min]</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">End[min]</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  }
  if(parIsDataNr(par)>0) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">DataNr</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  }
  if(parIsFitNr(par)>0) {
    fprintf(fp, "        <ss:Cell>\n");
    fprintf(fp, "          <ss:Data ss:Type=\"String\">FitNr</ss:Data>\n");
    fprintf(fp, "        </ss:Cell>\n");
  }
  /* End the title line */
  n=fprintf(fp, "      </ss:Row>\n");
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }

  /* Write parameters */
  if(verbose>2) printf("writing data table\n");
  for(int ri=0; ri<par->tacNr; ri++) {
    if(verbose>3) printf("    for region %d\n", 1+ri);
    /* Start the line */
    fprintf(fp, "      <ss:Row>\n");
    /* Region names and parameter values */
    fprintf(fp, "        <ss:Cell>\n");
    char *senc=strEncodeForXML(par->r[ri].name);
    if(senc==NULL) fprintf(fp, "          <ss:Data ss:Type=\"String\">%s</ss:Data>\n", par->r[ri].name);
    else {fprintf(fp, "          <ss:Data ss:Type=\"String\">%s</ss:Data>\n", senc); free(senc);}
    fprintf(fp, "        </ss:Cell>\n");
    if(parIsModel(par)>0) {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"String\">%s</ss:Data>\n", modelCode(par->r[ri].model));
      fprintf(fp, "        </ss:Cell>\n");
    }
    for(int pi=0; pi<par->parNr; pi++) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(par->r[ri].p[pi])) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].p[pi]);
      fprintf(fp, "        </ss:Cell>\n");
      if(parSDWithPar(par, pi)) {
        fprintf(fp, "        <ss:Cell>\n");
        if(isnan(par->r[ri].sd[pi])) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
        else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].sd[pi]);
        fprintf(fp, "        </ss:Cell>\n");
      }
      if(parCLWithPar(par, pi)) {
        fprintf(fp, "        <ss:Cell>\n");
        if(isnan(par->r[ri].cl1[pi])) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
        else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].cl1[pi]);
        fprintf(fp, "        </ss:Cell>\n");
        fprintf(fp, "        <ss:Cell>\n");
        if(isnan(par->r[ri].cl2[pi])) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
        else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].cl2[pi]);
        fprintf(fp, "        </ss:Cell>\n");
      }
    }
    /* WSS etc */
    if(parIsWSS(par)) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(par->r[ri].wss)) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].wss);
      fprintf(fp, "        </ss:Cell>\n");
    }
    if(parIsFitRange(par)>0) {
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(par->r[ri].start)) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].start);
      fprintf(fp, "        </ss:Cell>\n");
      fprintf(fp, "        <ss:Cell>\n");
      if(isnan(par->r[ri].end)) fprintf(fp, "          <ss:Data ss:Type=\"Number\"></ss:Data>\n");
      else fprintf(fp, "          <ss:Data ss:Type=\"Number\">%g</ss:Data>\n", par->r[ri].end);
      fprintf(fp, "        </ss:Cell>\n");
    }
    if(parIsDataNr(par)>0) {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"Number\">%d</ss:Data>\n", par->r[ri].dataNr);
      fprintf(fp, "        </ss:Cell>\n");
    }
    if(parIsFitNr(par)>0) {
      fprintf(fp, "        <ss:Cell>\n");
      fprintf(fp, "          <ss:Data ss:Type=\"Number\">%d</ss:Data>\n", par->r[ri].fitNr);
      fprintf(fp, "        </ss:Cell>\n");
    }
    /* End the line */
    n=fprintf(fp, "      </ss:Row>\n");
    if(n<1) {
      statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
      return TPCERROR_CANNOT_WRITE;
    }
  }

  /* Write XML end part */
  fprintf(fp, "    </ss:Table>\n");
  fprintf(fp, "  </ss:Worksheet>\n");
  n=fprintf(fp, "</ss:Workbook>\n");
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_CANNOT_WRITE);
    return TPCERROR_CANNOT_WRITE;
  }
  
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
