/** @file mean.c
 *  @brief Functions for calculating mean and stdev.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
/*****************************************************************************/
#include "tpcextensions.h"
/*****************************************************************************/
#include "tpcstatist.h"
/*****************************************************************************/

/*****************************************************************************/
/** Calculate mean and sample standard deviation in an array of given data.
    NaNs in the data are left out.
    @author Vesa Oikonen
    @sa statMedian, doubleMean, doubleWMean, floatMean, fitLine, statSortDouble, 
     doubleMatch, doubleRange, fstatMeanSD
    @return 0 if successful, otherwise <>0.
 */
int statMeanSD(
  /** Pointer to data array of size n; data is not changed in any way. */
  double *data,
  /** Length of data array. */
  unsigned int n,
  /** Pointer to variable where mean will be written; enter NULL if not needed. */
  double *mean,
  /** Pointer to variable where SD will be written; enter NULL if not needed. */
  double *sd,
  /** Pointer to variable where the number of valid (not NaN) samples will be written;
      enter NULL if not needed. */
  unsigned int *vn
) {
  unsigned int i, m;
  double sumsqr=0.0, sqrsum=0.0;

  if(mean!=NULL) *mean=nan(""); 
  if(sd!=NULL) *sd=nan(""); 
  if(vn!=NULL) *vn=0;
  if(n<1 || data==NULL) return(1);
  for(i=m=0; i<n; i++) if(!isnan(data[i])) m++;
  if(m<1) return(2);
  if(vn!=NULL) *vn=m;  

  /* Calculate mean, and sum for possible SD calculation */
  for(i=0; i<n; i++) if(!isnan(data[i])) sqrsum+=data[i];
  if(mean!=NULL) *mean=sqrsum/(double)m;
  
  /* Calculate SD, if required */
  if(sd==NULL) return(0);
  if(m==1) {*sd=0.0; return(0);}
  sqrsum*=sqrsum;

  for(i=0; i<n; i++) if(!isnan(data[i])) sumsqr+=data[i]*data[i];
  double ff=sumsqr - sqrsum/(double)m; if(!(ff>0.0)) {*sd=0.0; return(0);}
  *sd=sqrt( ff/(double)(m-1) );

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Calculate mean and sample standard deviation in an array of given float data.
    NaNs in the data are left out.
    @author Vesa Oikonen
    @sa statMeanSD, floatMean, floatSum, floatMatch
    @return 0 if successful, otherwise <>0.
 */
int fstatMeanSD(
  /** Pointer to data array of size n; data is not changed in any way. */
  float *data,
  /** Length of data array. */
  unsigned int n,
  /** Pointer to variable where mean will be written; enter NULL if not needed. */
  float *mean,
  /** Pointer to variable where SD will be written; enter NULL if not needed. */
  float *sd,
  /** Pointer to variable where the number of valid (not NaN) samples will be written;
      enter NULL if not needed. */
  unsigned int *vn
) {
  unsigned int i, m;
  float sumsqr=0.0, sqrsum=0.0;

  if(mean!=NULL) *mean=nanf(""); 
  if(sd!=NULL) *sd=nanf(""); 
  if(vn!=NULL) *vn=0;
  if(n<1 || data==NULL) return(1);
  for(i=m=0; i<n; i++) if(!isnan(data[i])) m++;
  if(m<1) return(2);
  if(vn!=NULL) *vn=m;

  /* Calculate mean, and sum for possible SD calculation */
  for(i=0; i<n; i++) if(!isnan(data[i])) sqrsum+=data[i];
  if(mean!=NULL) *mean=sqrsum/(float)m;
  
  /* Calculate SD, if required */
  if(sd==NULL) return(0);
  if(m==1) {*sd=0.0; return(0);}
  sqrsum*=sqrsum;

  for(i=0; i<n; i++) if(!isnan(data[i])) sumsqr+=data[i]*data[i];
  float ff=sumsqr - sqrsum/(float)m; if(!(ff>0.0)) {*sd=0.0; return(0);}
  *sd=sqrtf( ff/(float)(m-1) );

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
