/** @file matio.c
 *  @brief I/O functions for Mat TAC file format.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read Mat format from CSV struct into TAC struct.
 *  @return enum tpcerror (TPCERROR_OK when successful).
 *  @author Vesa Oikonen
 */
int tacReadMat(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to CSV from which data is read. */
  CSV *csv,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(csv==NULL || csv->row_nr<1 || csv->col_nr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) printf("%s()\n", __func__);

  int ret, n;

  /* Check that CSV is of regular size */
  if(!csvIsRegular(csv)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Column and row numbers must be at least 2 */
  if(csv->col_nr<2 || csv->row_nr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Allocate memory for TAC data */
  ret=tacAllocate(tac, csv->col_nr-1, csv->row_nr-1);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return ret;
  tac->tacNr=csv->row_nr-1; tac->sampleNr=csv->col_nr-1;
  tac->isframe=1;

  /* First cell may contain study number */
  char *cptr=csvCell(csv, 0, 0);
  if(cptr!=NULL && cptr[0] && strnlen(cptr, 20)<20)
    tacSetHeaderStudynr(&tac->h, cptr);

  /* Get sample times from the first line */
  for(int j=0; j<tac->sampleNr; j++) {
    cptr=csvCell(csv, 0, j+1); if(cptr==NULL) {ret++; break;}
    n=strlen(cptr); if(n<3) {ret++; break;}
    char *cptr2=strchr(cptr+1, '-'); if(cptr2!=NULL) {*cptr2=(char)0; cptr2++;}
    tac->x1[j]=atofVerified(cptr);
    tac->x2[j]=atofVerified(cptr2);
    if(isnan(tac->x1[j]) || isnan(tac->x2[j])) {ret++; break;}
    tac->x[j]=0.5*(tac->x1[j]+tac->x2[j]);
  }
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }


  /* Get region names from the first column */
  ret=0;
  for(int i=0; i<tac->tacNr; i++) {
    cptr=csvCell(csv, i+1, 0); if(cptr==NULL) {ret++; break;}
    n=strlen(cptr); if(n>2 && cptr[n-1]=='\'') {cptr[n-1]=(char)0; n--;}
    if(n>1 && cptr[0]=='\'') {cptr++; n--;}
    if(n>0) strlcpy(tac->c[i].name, cptr, MAX_TACNAME_LEN); 
  }
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Get concentrations */
  n=0; ret=0;
  for(int i=0; i<tac->tacNr && ret==0; i++) {
    for(int j=0; j<tac->sampleNr; j++) {
      cptr=csvCell(csv, i+1, j+1); if(cptr==NULL) {ret++; break;}
      tac->c[i].y[j]=atofVerified(cptr); if(!isnan(tac->c[i].y[j])) n++;
    }
  }
  if(ret) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  if(n<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  


  tac->format=TAC_FORMAT_MAT;
  tac->weighting=WEIGHTING_OFF;

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
