/** @file mtac.c
 *  @brief Processing list of TAC structures.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Initiate the MTAC structure before any use.
    @sa mtacFree, MTAC, TAC
 */
void mtacInit(
  /** Pointer to MTAC. */
  MTAC *mtac
) {
  if(mtac==NULL) return;
  mtac->nr=mtac->_nr=0;
  mtac->tac=NULL;
}
/*****************************************************************************/

/*****************************************************************************/
/** Free memory allocated for MTAC data. All contents are destroyed. 
    @pre Before first use initialize the MTAC struct with mtacInit().
    @sa mtacInit, tacFree
 */
void mtacFree(
  /** Pointer to MTAC. */
  MTAC *mtac
) {
  if(mtac==NULL) return;
  /* Free allocated TACs. */ 
  for(int i=0; i<mtac->_nr; i++) tacFree(&mtac->tac[i]); 
  free(mtac->tac);
  // then set everything to zero or NULL again
  mtacInit(mtac);
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for TACs in MTAC data structure.
    @sa mtacInit, mtacFree, mtacAddTAC
    @return Returns TPCERROR status.
 */
int mtacAllocate(
  /** Pointer to initiated MTAC structure; any old contents are deleted.
      Number of TACs is kept at zero, but TACs in the list are initiated. */
  MTAC *mtac,
  /** Nr of TACs to allocate */
  int nr
) {
  if(mtac==NULL) return TPCERROR_FAIL;
  /* Delete any previous contents */
  mtacFree(mtac);
  /* If no memory is requested, then just return */
  if(nr<1) return TPCERROR_OK;
  /* Allocate the list of TACs */
  mtac->tac=(TAC*)malloc(nr*sizeof(TAC));
  if(mtac->tac==NULL) return TPCERROR_OUT_OF_MEMORY;
  for(int i=0; i<nr; i++) tacInit(&mtac->tac[i]);
  mtac->_nr=nr;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Allocate memory for more TACs in MTAC structure.
    Previous contents are not changed.
    @sa mtacAllocate, mtacInit, mtacFree, mtacAddTAC, MTAC, TAC
    @return Returns TPCERROR status.
 */
int mtacAllocateMore(
  /** Pointer to initiated and previously allocated MTAC structure; any old contents are 
      preserved, including nr of TACs. */
  MTAC *mtac,
  /** Nr of additional TAC structures to allocate; if structure contains 
      unused space for requested TACs already, then nothing is done. */
  int nr
) {
  if(mtac==NULL) return TPCERROR_FAIL;

  /* If not allocated before, then use mtacAllocate */
  if(mtac->_nr==0) return(mtacAllocate(mtac, nr));

  /* Check if there is enough space already */
  int newNr, addNr;
  newNr=mtac->nr + nr;
  addNr=newNr - mtac->_nr;
  if(addNr<=0) return TPCERROR_OK;
  
  /* Reallocate memory for TACC data */
  TAC *tacPtr;
  tacPtr=realloc(mtac->tac, sizeof(TAC)*newNr);
  if(tacPtr==NULL) return TPCERROR_OUT_OF_MEMORY;
  mtac->tac=tacPtr;

  /* Update the nr of allocated TACs and initiate new TACs */ 
  for(int i=mtac->_nr; i<newNr; i++) tacInit(&mtac->tac[i]);
  mtac->_nr=newNr;

  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
/** Copy given TAC structure into MTAC list.
    @sa mtacInit, mtacFree, mtacAllocate, MTAC, TAC
    @return Returns TPCERROR status.
 */
int mtacAddTAC(
  /** Pointer to initiated MTAC structure; any old contents are preserved. */
  MTAC *mtac,
  /** Pointer to TAC structure to add. */
  TAC *tac
) {
  if(mtac==NULL || tac==NULL) return TPCERROR_FAIL;
  /* Make room for the TAC */
  int ret=mtacAllocateMore(mtac, 1); if(ret!=TPCERROR_OK) return(ret);
  /* Copy TAC into the end of the list */
  ret=tacDuplicate(tac, &mtac->tac[mtac->nr]); if(ret!=TPCERROR_OK) return(ret);
  mtac->nr++;
  return TPCERROR_OK;
}
/*****************************************************************************/

/*****************************************************************************/
