/** @file tacabss.c
 *  @brief I/O functions for automatic blood sampling system (ABSS) files.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
#include "tpcisotope.h"
#include "tpccsv.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Read Allogg #2 ABSS data from IFT struct into TAC struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacInit, tacRead, tacReadOldAllogg
 */
int tacReadAllogg(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to IFT to read data from. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(hdr==NULL || hdr->keyNr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  /* Check whether IFT even can contain Allogg data */
  int ii;
  /* Find the line '//Heading' */
  ii=iftSearchValue(hdr, "//Heading", 0);
  if(ii<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Find the line '//Data' */
  ii=iftSearchValue(hdr, "//Data", 0);
  if(ii<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  /* Sample data should start after this index (right after title) */
  if(verbose>3) {
    printf("found '%s' at ii=%d\n", hdr->item[ii].value, ii);
    fflush(stdout);
  }
  /* Allocate memory for TAC data */
  int ret;
  ret=tacAllocate(tac, hdr->keyNr-ii, 8);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->tacNr=8;
  /* Set basic header information */
  tac->format=TAC_FORMAT_ABSS_ALLOGG;
  tac->isframe=1; // Allogg data always contains frame start and end times
  tac->tunit=UNIT_SEC;
  tac->cunit=UNIT_COUNTS;
  /* Set TAC names into TAC struct */
  strcpy(tac->c[0].name, "Singles");
  strcpy(tac->c[1].name, "Coincidents");
  strcpy(tac->c[2].name, "Singles-count-rate");
  strcpy(tac->c[3].name, "Coincidents-count-rate");
  strcpy(tac->c[4].name, "Singles-count-rate-DTC");
  strcpy(tac->c[5].name, "Coincidents-count-rate-DTC");
  strcpy(tac->c[6].name, "Singles-DTC-decay");
  strcpy(tac->c[7].name, "Coincidents-DTC-decay");

  /* Find the first line starting with valid date and time */
  struct tm start_time;
  ii++;
  for(; ii<hdr->keyNr; ii++) {
    if(strDateTimeRead(hdr->item[ii].value, &start_time)==0) break;
  }
  if(ii==hdr->keyNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_MISSING_DATA);
    return(TPCERROR_MISSING_DATA);
  }
  /* Save the start time of the first sample to be saved later in TAC header */
  char scan_start_time[32];
  if(strftime(scan_start_time, 20, "%Y-%m-%d %H:%M:%S", &start_time)==0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  /* Go through the lines, extracting sample data, and deleting data lines
     from the header; stop storing sample data after first invalid line */
  if(verbose>2) {printf("  reading data lines...\n"); fflush(stdout);}
  int i=0, n, stopped=0;
  while(ii<hdr->keyNr && i<tac->_sampleNr) {
    if(stopped) {iftDelete(hdr, ii); continue;}
    if(hdr->item[ii].value[0]=='#') {ii++; continue;}
    if(hdr->item[ii].value==NULL) {iftDelete(hdr, ii); continue;}
    if(strnlen(hdr->item[ii].key, 3)>0) {
      iftDelete(hdr, ii); stopped=1; continue;
    }
    if(strDateTimeRead(hdr->item[ii].value, &start_time)!=0) {
      iftDelete(hdr, ii); stopped=1; continue;
    }
    //printf("reading sample '%s'\n", hdr->item[ii].value+20); fflush(stdout);
    n=sscanf(hdr->item[ii].value+20, "%lf %lf %lf %lf %lf %lf %lf %lf %lf\n", 
         &tac->x1[i],
         &tac->c[0].y[i], &tac->c[1].y[i], &tac->c[2].y[i], &tac->c[3].y[i], 
         &tac->c[4].y[i], &tac->c[5].y[i], &tac->c[6].y[i], &tac->c[7].y[i]
        ); 
    if(n==9) i++; else stopped=1;
    iftDelete(hdr, ii);
  }
  tac->sampleNr=i;
  if(verbose>2) printf("  %d data line(s) read.\n", i);
  if(i<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }

  /* Calculate frame end and mid times */
  for(i=0; i<tac->sampleNr-1; i++) {
    tac->x2[i]=tac->x1[i+1];
    tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);
  }
  /* last sample */
  if(tac->sampleNr>1) {
    tac->x2[i]=tac->x1[i]+(tac->x2[i-1]-tac->x1[i-1]);
  } else {
    tac->x2[i]=tac->x1[i]+1.0;
  }
  tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);

  /* Delete more lines from the header */
  ii=iftSearchValue(hdr, "//Heading", 0); if(ii>=0) iftDelete(hdr, ii);
  ii=iftSearchValue(hdr, "//Data", 0); if(ii>=0) iftDelete(hdr, ii);
  ii=iftSearchValue(hdr, "Absolute time", 0); if(ii>=0) iftDelete(hdr, ii);
  /* Set all header lines to comments */
  for(ii=0; ii<hdr->keyNr; ii++) hdr->item[ii].comment=1;
  /* Copy the header contents to TAC (deletes previous contents) */
  (void)iftDuplicate(hdr, &tac->h);
  tac->h.type=3;
  /* Copy study number to the TAC header */
  ii=iftFindKey(hdr, "Run number", 0); 
  if(ii>=0 && strnlen(hdr->item[ii].value, 5)>0)
   (void)tacSetHeaderStudynr(&tac->h, hdr->item[ii].value);
  /* Copy the start time of the first sample to the TAC header */
  (void)tacSetHeaderScanstarttime(&tac->h, scan_start_time);


  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read Allogg #1 ABSS data from IFT struct into TAC struct.
    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacInit, tacRead, tacReadAllogg
 */
int tacReadOldAllogg(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to IFT to read data from. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(hdr==NULL || hdr->keyNr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  /* Check whether IFT even can contain Allogg 1 data */
  int ii;
  /* Find a specific key */
  ii=iftFindKey(hdr, "Discriminators", 0);
  if(ii<0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }

  /* Find the measurement start date and time, and store for later use */
  char scan_start_time[32];
  int n, year, month, day, hour, min, sec;
  char buf[128];
  for(ii=0; ii<hdr->keyNr; ii++) {
    if(strnlen(hdr->item[ii].key, 3)>0 || hdr->item[ii].value==NULL) continue;
    if(strDateTimeValid(hdr->item[ii].value, scan_start_time)==0) {
      n=sscanf(scan_start_time, "%d-%d-%d %d:%d:%d", 
               &year, &month, &day, &hour, &min, &sec);
    } else {
      n=strnlen(hdr->item[ii].value, 15);
      if(n<8 || n>10) continue;
      //printf("ii=%d '%s' '%s'\n", ii, hdr->item[ii].key, hdr->item[ii].value);
      n=sscanf(hdr->item[ii].value, "%d-%d-%d", &year, &month, &day);
    }
    if(n==3 || n==6) break;
  }
  if(ii==hdr->keyNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  //printf("found '%s' at ii=%d\n", hdr->item[ii].value, ii);

  /* Find the measurement start time HHMMSS */
  for(ii=0; ii<hdr->keyNr; ii++) {
    if(strnlen(hdr->item[ii].key, 3)>0 || hdr->item[ii].value==NULL) continue;
    //printf("ii=%d '%s' '%s'\n", ii, hdr->item[ii].key, hdr->item[ii].value);
    if(strnlen(hdr->item[ii].value, 10)==6) break;
  }
  if(ii==hdr->keyNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  //printf("found '%s' at ii=%d\n", hdr->item[ii].value, ii);
  strlcpy(buf, hdr->item[ii].value, 128);   buf[2]=(char)0; hour=atoi(buf);
  strlcpy(buf, hdr->item[ii].value+2, 128); buf[2]=(char)0; min=atoi(buf);
  strlcpy(buf, hdr->item[ii].value+4, 128); buf[2]=(char)0; sec=atoi(buf);
    
  /* Make date and time string and set abss->startTime */
  sprintf(scan_start_time, "%04d-%02d-%02d %02d:%02d:%02d", 
          year, month, day, hour, min, sec);
  if(verbose>11) printf("  scan_start_time := %s\n", buf);

  /* Sample data should start after this index */
  /* Allocate memory for TAC data */
  int ret;
  ret=tacAllocate(tac, hdr->keyNr-ii, 2);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->tacNr=2;
  /* Set basic header information */
  tac->format=TAC_FORMAT_ABSS_ALLOGG_OLD;
  tac->isframe=1; // Allogg data always contains frame start and end times
  tac->tunit=UNIT_SEC;
  tac->cunit=UNIT_COUNTS;
  /* Set TAC names into TAC struct */
  strcpy(tac->c[0].name, "Singles");
  strcpy(tac->c[1].name, "Coincidents");

  /* Go through the lines, extracting sample data, and deleting data lines
     from the header */
  if(verbose>2) printf("  reading data lines...\n");
  ii++;
  int i=0;
  while(ii<hdr->keyNr && i<tac->_sampleNr) {
    if(strnlen(hdr->item[ii].key, 3)>0 || hdr->item[ii].value==NULL) {
      iftDelete(hdr, ii); continue;}
    if(hdr->item[ii].value[0]=='#') {iftDelete(hdr, ii); continue;}
    n=sscanf(hdr->item[ii].value, "%lf %lf %lf", 
         &tac->x2[i], &tac->c[0].y[i], &tac->c[1].y[i]);
    if(n==3) i++;
    iftDelete(hdr, ii);
  }
  tac->sampleNr=i;
  if(verbose>2) printf("  %d data line(s) read.\n", i);
  if(i<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }
  
  /* Calculate frame start and mid times */
  for(i=1; i<tac->sampleNr; i++) {
    tac->x1[i]=tac->x2[i-1];
    tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);
  }
  /* first sample */
  i=0;
  if(tac->sampleNr>1) {
    tac->x1[i]=tac->x2[i]-(tac->x2[i+1]-tac->x1[i+1]);
  } else {
    tac->x1[i]=0.0;
  }
  tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);

  /* Set all header lines to comments */
  for(ii=0; ii<hdr->keyNr; ii++) hdr->item[ii].comment=1;
  /* Copy the header contents to TAC (deletes previous contents) */
  (void)iftDuplicate(hdr, &tac->h);
  tac->h.type=3;
  /* Copy the start time to the TAC header */
  (void)tacSetHeaderScanstarttime(&tac->h, scan_start_time);


  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read Scanditronics ABSS data from IFT struct into TAC struct.

    Two date-time strings will be written in header, scan_start_time containing
    the time of the first sample, and sampler_start_time containing the time
    given in ABSS file header.

    Scanditronics ABSS format is almost identical to GEMS, but the headers
    and times are processed differently, hence separate functions.
    If function finds that provided data is actually from GEMS, it calls
    tacReadGEMS.

    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacInit, tacRead, tacReadGEMS, tacReadAllogg
 */
int tacReadScanditronics(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to IFT to read data from. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(hdr==NULL || hdr->keyNr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  /* Check whether IFT contains Scanditronics (or GEMS) data */
  int format;
  if(iftSearchValue(hdr, "AUX", 0)<0 || 
     iftSearchValue(hdr, "1st detector pair", 0)<0)
  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  format=TAC_FORMAT_ABSS_SCANDITRONICS;

  /* Find the measurement start date, possibly with time; Scanditronics
     header is not required to contain the date and time, because those are
     stored as the first column in the sample data. */
  struct tm start_time; start_time.tm_mday=0;
  int ii, n;
  for(ii=0; ii<hdr->keyNr; ii++) {
    if(strnlen(hdr->item[ii].key, 3)>0) continue;
    n=strnlen(hdr->item[ii].value, 22);
    if(n<10 || n>19) continue;
    //printf("ii=%d '%s' '%s'\n", ii, hdr->item[ii].key, hdr->item[ii].value);
    if(strDateTimeRead(hdr->item[ii].value, &start_time)!=0) {
      /* Date and time not found, try only date */
      if(strDateRead(hdr->item[ii].value, &start_time)!=0) continue; 
    }
    break;
  }
  if(verbose>1) {
    if(ii==hdr->keyNr) printf("  header does not contain date and time.\n");
    else printf("found '%s' at ii=%d\n", hdr->item[ii].value, ii);
  }

  /* Sample data should start after this index */
  /* Allocate memory for TAC data */
  int ret;
  ret=tacAllocate(tac, hdr->keyNr-ii, 7);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->tacNr=7;
  /* Set basic header information */
  tac->format=format;
  tac->isframe=1; // Data always contains frame start and end times
  tac->tunit=UNIT_SEC;
  tac->cunit=UNIT_COUNTS;
  /* Set TAC names into TAC struct */
  strcpy(tac->c[0].name, "Detector1-Coinc");
  strcpy(tac->c[1].name, "Detector1-Singl1");
  strcpy(tac->c[2].name, "Detector1-Singl2");
  strcpy(tac->c[3].name, "Detector2-Coinc");
  strcpy(tac->c[4].name, "Detector2-Singl1");
  strcpy(tac->c[5].name, "Detector2-Singl2");
  strcpy(tac->c[6].name, "AUX-counts");

  /* Go through the lines, extracting sample data, and deleting data lines
     from the header */
  if(verbose>2) printf("  reading data lines...\n");
  ii++;
  int i=0, stopped=0;
  double time_of_day, first_tod=0.0, time_since_start, meas_interval;
  while(ii<hdr->keyNr && i<tac->_sampleNr) {
    if(stopped) {iftDelete(hdr, ii); continue;}
    if(strnlen(hdr->item[ii].key, 3)>0 || hdr->item[ii].value==NULL) {
      iftDelete(hdr, ii); continue;}
    if(hdr->item[ii].value[0]=='#') {iftDelete(hdr, ii); continue;}
    //printf("ii=%d '%s'\n", ii, hdr->item[ii].value);
    n=sscanf(hdr->item[ii].value, "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf\n", 
         &time_of_day, &time_since_start, &meas_interval, 
         &tac->c[0].y[i], &tac->c[1].y[i], &tac->c[2].y[i], &tac->c[3].y[i],
         &tac->c[4].y[i], &tac->c[5].y[i], &tac->c[6].y[i]);
    //printf("  n=%d\n", n);
    if(n!=10) {
      /* first line may contain the protocol; keep it as comment */
      if(i==0) {ii++; continue;}
      /* Later occurrence is considered as an error */
      stopped=1; iftDelete(hdr, ii); continue;
    } 
    if(time_of_day==0.0 && meas_interval==0.0) {
      stopped=1; iftDelete(hdr, ii); continue;}
    if(i==0) first_tod=time_of_day;
    tac->x1[i]=time_since_start;
    tac->x2[i]=time_since_start+meas_interval;
    i++;
    iftDelete(hdr, ii);
  }
  tac->sampleNr=i;
  if(verbose>2) printf("  %d data line(s) read.\n", i);
  if(i<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }
  if(verbose>4) printf("  first_tod := %.1lf\n", first_tod);
  
  /* Calculate frame mid times */
  for(i=0; i<tac->sampleNr; i++) tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);

  /* Read scan start date and time from the first sample */
  char scan_start_time[32], sampler_start_time[32];
  time_t tim=(time_t)first_tod;
  struct tm tmtim;
  if(localtime_r(&tim, &tmtim)==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  /* Check that also date is here; if not, this is actually GEMS data */
  if(tmtim.tm_year<73) {
    if(verbose>0) printf("data is not Scanditronics but GEMS format\n");
    if(verbose>1 && 
       strftime(scan_start_time, 32, "%Y-%m-%d %H:%M:%S", &tmtim)!=0) {
      printf("first_sample_time := %s\n", scan_start_time);
    }
    tacFree(tac);
    return(tacReadGEMS(tac, hdr, status));
  }
  /* Set scan_start_time based on the first sample */
  if(strftime(scan_start_time, 32, "%Y-%m-%d %H:%M:%S", &tmtim)==0) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  /* Set sampler_start_time based on the header, or first sample, if header
     does not contain the time */
  if(start_time.tm_mday==0) { // header did not contain date and time
    strcpy(sampler_start_time, scan_start_time);
  } else {
    /* If header contains just the date but not the time, then take the time
       from samples */
    if(start_time.tm_hour==0 && start_time.tm_min==0 && start_time.tm_sec==0) {
      start_time.tm_hour=tmtim.tm_hour;
      start_time.tm_min=tmtim.tm_min;
      start_time.tm_sec=tmtim.tm_sec;
    }
    if(strftime(sampler_start_time, 32, "%Y-%m-%d %H:%M:%S", &start_time)==0)
      strcpy(sampler_start_time, scan_start_time);
  }


  /* Set all header lines to comments */
  for(ii=0; ii<hdr->keyNr; ii++) hdr->item[ii].comment=1;
  /* Copy the header contents to TAC (deletes previous contents) */
  (void)iftDuplicate(hdr, &tac->h);
  tac->h.type=3;
  /* Set the scan start time in the TAC header */
  (void)tacSetHeaderScanstarttime(&tac->h, scan_start_time);
  /* Set the sampler start time in the TAC header */
  if(sampler_start_time[0])
    (void)iftPut(&tac->h, "sampler_start_time", sampler_start_time, 1, NULL);
  /* Copy study number to the TAC header */
  ii=iftFindKey(hdr, "Patient", 0); 
  if(ii>=0 && strnlen(hdr->item[ii].value, 5)>0)
   (void)tacSetHeaderStudynr(&tac->h, hdr->item[ii].value);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
/** Read GEMS ABSS data from IFT struct into TAC struct.

    Two date-time strings will be written in header, scan_start_time containing
    the time of the first sample, and sampler_start_time containing the time
    given in ABSS file header.

    GEMS ABSS format is almost identical to Scanditronics, but the headers
    and times are processed differently, hence separate functions.
    If function finds that provided data is actually from Scanditronics, it 
    calls tacReadScanditronics.

    @return enum tpcerror (TPCERROR_OK when successful).
    @author Vesa Oikonen
    @sa tacInit, tacRead, tacReadScanditronics, tacReadAllogg
 */
int tacReadGEMS(
  /** Pointer to TAC struct, contents of which are to be written. */
  TAC *tac,
  /** Pointer to IFT to read data from. */
  IFT *hdr,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(tac==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return TPCERROR_FAIL;
  }
  tacFree(tac);
  if(hdr==NULL || hdr->keyNr<2) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return TPCERROR_NO_DATA;
  }  
  if(verbose>0) {printf("%s()\n", __func__); fflush(stdout);}

  /* Check whether IFT contains GEMS (or Scanditronics) data */
  int format;
  if(iftSearchValue(hdr, "AUX", 0)<0 || 
     iftSearchValue(hdr, "1st detector pair", 0)<0)
  {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_INVALID_FORMAT);
    return TPCERROR_INVALID_FORMAT;
  }
  if(iftSearchValue(hdr, "Scanditronics", 0)>=0) {
    if(verbose>0) printf("data is GEMS but Scanditronics format\n");
    return(tacReadScanditronics(tac, hdr, status));
  } else {
    format=TAC_FORMAT_ABSS_GEMS;
  }

  /* Find the measurement start date, possibly with time;
     at least the date must be there for calibration. */
  struct tm start_time;
  int ii, n;
  for(ii=0; ii<hdr->keyNr; ii++) {
    if(strnlen(hdr->item[ii].key, 3)>0) continue;
    n=strnlen(hdr->item[ii].value, 22);
    if(n<10 || n>19) continue;
    //printf("ii=%d '%s' '%s'\n", ii, hdr->item[ii].key, hdr->item[ii].value);
    if(strDateTimeRead(hdr->item[ii].value, &start_time)!=0) {
      /* Date and time not found, try only date */
      if(strDateRead(hdr->item[ii].value, &start_time)!=0) continue; 
    }
    break;
  }
  if(ii==hdr->keyNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  //printf("found '%s' at ii=%d\n", hdr->item[ii].value, ii);

  /* Sample data should start after this index */
  /* Allocate memory for TAC data */
  int ret;
  ret=tacAllocate(tac, hdr->keyNr-ii, 7);
  statusSet(status, __func__, __FILE__, __LINE__, ret);
  if(ret!=TPCERROR_OK) return(ret);
  tac->tacNr=7;
  /* Set basic header information */
  tac->format=format;
  tac->isframe=1; // Data always contains frame start and end times
  tac->tunit=UNIT_SEC;
  tac->cunit=UNIT_COUNTS;
  /* Set TAC names into TAC struct */
  strcpy(tac->c[0].name, "Detector1-Coinc");
  strcpy(tac->c[1].name, "Detector1-Singl1");
  strcpy(tac->c[2].name, "Detector1-Singl2");
  strcpy(tac->c[3].name, "Detector2-Coinc");
  strcpy(tac->c[4].name, "Detector2-Singl1");
  strcpy(tac->c[5].name, "Detector2-Singl2");
  strcpy(tac->c[6].name, "AUX-counts");

  /* Go through the lines, extracting sample data, and deleting data lines
     from the header */
  if(verbose>2) printf("  reading data lines...\n");
  ii++;
  int i=0, stopped=0;
  double time_of_day, first_tod=0.0, time_since_start, meas_interval;
  while(ii<hdr->keyNr && i<tac->_sampleNr) {
    if(stopped) {iftDelete(hdr, ii); continue;}
    if(strnlen(hdr->item[ii].key, 3)>0 || hdr->item[ii].value==NULL) {
      iftDelete(hdr, ii); continue;}
    if(hdr->item[ii].value[0]=='#') {iftDelete(hdr, ii); continue;}
    //printf("ii=%d '%s'\n", ii, hdr->item[ii].value);
    n=sscanf(hdr->item[ii].value, "%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf\n", 
         &time_of_day, &time_since_start, &meas_interval, 
         &tac->c[0].y[i], &tac->c[1].y[i], &tac->c[2].y[i], &tac->c[3].y[i],
         &tac->c[4].y[i], &tac->c[5].y[i], &tac->c[6].y[i]);
    //printf("  n=%d\n", n);
    if(n!=10) {
      /* first line may contain the protocol; keep it as comment */
      if(i==0) {ii++; continue;}
      /* Later occurrence is considered as an error */
      stopped=1; iftDelete(hdr, ii); continue;
    } 
    if(time_of_day==0.0 && meas_interval==0.0) {
      stopped=1; iftDelete(hdr, ii); continue;}
    if(i==0) first_tod=time_of_day;
    tac->x1[i]=time_since_start;
    tac->x2[i]=time_since_start+meas_interval;
    i++;
    iftDelete(hdr, ii);
  }
  tac->sampleNr=i;
  if(verbose>2) printf("  %d data line(s) read.\n", i);
  if(i<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return(TPCERROR_NO_DATA);
  }
  if(verbose>4) printf("  first_tod := %.1lf\n", first_tod);
  
  /* Calculate frame mid times */
  for(i=0; i<tac->sampleNr; i++) tac->x[i]=0.5*(tac->x1[i]+tac->x2[i]);

  /* Read scan start time from the first sample, which is used also as 
     sampler start time, if only date was given in the header */
  char scan_start_time[32], sampler_start_time[32];
  time_t tim=(time_t)first_tod;
  struct tm tmtim;
  if(gmtime_r(&tim, &tmtim)==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }
  /* Check whether also date is here; 
     if yes, this is actually Scanditronics data */
  if(tmtim.tm_year>71 || tmtim.tm_mday>1) {
    if(verbose>0) printf("data is not GEMS but Scanditronics format\n");
    if(verbose>1 && 
       strftime(scan_start_time, 32, "%Y-%m-%d %H:%M:%S", &tmtim)!=0) {
      printf("first_sample_time := %s\n", scan_start_time);
    }
    tacFree(tac);
    return(tacReadScanditronics(tac, hdr, status));
  }
  if(start_time.tm_hour==0 && start_time.tm_min==0 && start_time.tm_sec==0) {
    start_time.tm_hour=tmtim.tm_hour;
    start_time.tm_min=tmtim.tm_min;
    start_time.tm_sec=tmtim.tm_sec;
    n=strftime(sampler_start_time, 32, "%Y-%m-%d %H:%M:%S", &start_time);
    if(n==0) strcpy(sampler_start_time, "");
    strcpy(scan_start_time, sampler_start_time);
  } else {
    n=strftime(sampler_start_time, 32, "%Y-%m-%d %H:%M:%S", &start_time);
    if(n==0) strcpy(sampler_start_time, "");
    start_time.tm_hour=tmtim.tm_hour;
    start_time.tm_min=tmtim.tm_min;
    start_time.tm_sec=tmtim.tm_sec;
    n=strftime(scan_start_time, 32, "%Y-%m-%d %H:%M:%S", &start_time);
    if(n==0) strcpy(scan_start_time, "");
  }
  if(!scan_start_time[0]) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATETIME);
    return(TPCERROR_NO_DATETIME);
  }


  /* Set all header lines to comments */
  for(ii=0; ii<hdr->keyNr; ii++) hdr->item[ii].comment=1;
  /* Copy the header contents to TAC (deletes previous contents) */
  (void)iftDuplicate(hdr, &tac->h);
  tac->h.type=3;
  /* Set the scan start time in the TAC header */
  (void)tacSetHeaderScanstarttime(&tac->h, scan_start_time);
  /* Set the sampler start time in the TAC header */
  if(sampler_start_time[0])
    (void)iftPut(&tac->h, "sampler_start_time", sampler_start_time, 1, NULL);
  /* Copy study number to the TAC header */
  ii=iftFindKey(hdr, "Patient", 0); 
  if(ii>=0 && strnlen(hdr->item[ii].value, 5)>0)
   (void)tacSetHeaderStudynr(&tac->h, hdr->item[ii].value);

  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(TPCERROR_OK);
}
/*****************************************************************************/

/*****************************************************************************/
