/** @file taccomp.c
 *  @brief Comparison of TAC struct data.
 */
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcift.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
/*****************************************************************************/
#include "tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
/** Check whether time and concentration units are the same in two TAC data.
    @return 0 in case of match, 1 if y (concentration) unit is not matching,
    2 if x (time) unit is not matching, and 3 if neither is matching.
    @sa tacCompareConc, tacCompareTimes
    @author Vesa Oikonen
 */
int tacCompareUnit(
  /** Pointer to TAC structure. */
  TAC *d1,
  /** Pointer to TAC structure. */
  TAC *d2,
  /** Pointer to status data; enter NULL if not needed */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 3;
  }

  int ret=0;
  if(d1->cunit!=d2->cunit) ret+=1;
  if(d1->tunit!=d2->tunit) ret+=2;
  if(ret>0 && verbose>0) {
    printf("tac1.cunit := %s\n", unitName(d1->cunit));
    printf("tac2.cunit := %s\n", unitName(d2->cunit));
    printf("tac1.tunit := %s\n", unitName(d1->tunit));
    printf("tac2.tunit := %s\n", unitName(d2->tunit));
  }
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  return(ret);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC names are the same in two TAC data.

    Comparison is very strict and even case-sensitive, thus TAC names may need
    to be preprocessed if this is used for other purpose than SW testing.
    @return 0 in case of match, 1 if no match or error.
    @author Vesa Oikonen
    @sa tacSortByName, tacCompareUnit
 */
int tacCompareNames(
  /** Pointer to TAC structure. */
  TAC *d1,
  /** Pointer to TAC structure. */
  TAC *d2,
  /** TAC index [0..tacNr-1] to compare; enter <0 to verify all TACs. */
  const int i,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->tacNr<1 || d2->tacNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* If index is specified, then verify that that TAC is available */
  if(i>=0 && (i>=d1->tacNr || i>=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 1;
  }

  /* If index is not specified, then tacNr must match */
  if(i<0 && (d1->tacNr!=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different TAC nr.\n");
    return(1);
  }

  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int ri=0; ri<d1->tacNr; ri++) if(i<0 || i==ri) {
    if(strcmp(d1->c[ri].name, d2->c[ri].name)!=0) {
      if(verbose>0) {
        printf("tac1.c[%d].name := '%s'\n", ri, d1->c[ri].name);
        printf("tac2.c[%d].name := '%s'\n", ri, d2->c[ri].name);
      }
      return(1);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC concentrations (y values) are the same in two TAC data.
    Note that units are ignored here.

    If either absolute or relative difference is below the limit, the test is reported as passed.
    @sa tacCompareUnit, tacCompareTimes.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int tacCompareConc(
  /** Pointer to TAC structure. */
  TAC *d1,
  /** Pointer to TAC structure. */
  TAC *d2,
  /** TAC index [0..tacNr-1] to compare; enter <0 to verify all TACs. */
  const int i,
  /** Limit for accepted absolute difference; obligatory. */
  const double test_abs,
  /** Optional limit for accepted relative difference |2*(x1-x2)/(x1+x2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const double test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->tacNr<1 || d2->tacNr<1
     || d1->sampleNr<1 || d2->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }
  /* If index is specified, then verify that that TAC is available */
  if(i>=0 && (i>=d1->tacNr || i>=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_FAIL);
    return 1;
  }

  /* If index is not specified, then tacNr must match */
  if(i<0 && (d1->tacNr!=d2->tacNr)) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different TAC nr.\n");
    return(2);
  }

  /* Sample nr must match */
  if(d1->sampleNr!=d2->sampleNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different sample nr.\n");
    return(3);
  }
  
  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int ri=0; ri<d1->tacNr; ri++) if(i<0 || i==ri) {
    for(int fi=0; fi<d1->sampleNr; fi++) {
      if(doubleMatch(d1->c[ri].y[fi], d2->c[ri].y[fi], test_abs)==1)
        continue;
      if(test_rel>0.0 &&
         doubleMatchRel(d1->c[ri].y[fi], d2->c[ri].y[fi], test_rel)==1)
        continue;
      if(verbose>0) {
        double s;
        s=fabs(d1->c[ri].y[fi]-d2->c[ri].y[fi]);
        printf("tac1.c[%d].y[%d] := %g\n", ri, fi, d1->c[ri].y[fi]);
        printf("tac2.c[%d].y[%d] := %g\n", ri, fi, d2->c[ri].y[fi]);
        printf("|diff| := %g\n", s);
        printf("diff_limit := %g\n", test_abs);
        if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
      }
      return(10);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC sample times (x values) are the same in two TAC data.
    Note that units are ignored here.

    If either absolute or relative difference is below the limit, the test is reported as passed.
    @sa tacCompareUnit, tacCompareConc, tacSortByTime, tacVerifyTimeOrder.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int tacCompareTimes(
  /** Pointer to TAC structure. */
  TAC *d1,
  /** Pointer to TAC structure. */
  TAC *d2,
  /** Limit for accepted absolute difference. */
  const double test_abs,
  /** Optional limit for accepted relative difference |2*(x1-x2)/(x1+x2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const double test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->sampleNr<1 || d2->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }

  /* X type must match */
  if(d1->isframe!=d2->isframe) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different sampling type.\n"); 
    return(2);
  }
  
  /* Sample nr must match */
  if(d1->sampleNr!=d2->sampleNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different sample nr.\n");
    return(3);
  }
  
  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int fi=0; fi<d1->sampleNr; fi++) {
    if(d1->isframe==0) {
      if(doubleMatch(d1->x[fi], d2->x[fi], test_abs)==1)
        continue;
      if(test_rel>0.0 && doubleMatchRel(d1->x[fi], d2->x[fi], test_rel)==1)
        continue;
      if(verbose>0) {
        double s;
        s=fabs(d1->x[fi]-d2->x[fi]);
        printf("tac1.x[%d] := %g\n", fi, d1->x[fi]);
        printf("tac2.x[%d] := %g\n", fi, d2->x[fi]);
        printf("|diff| := %g\n", s);
        printf("diff_limit := %g\n", test_abs);
        if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
      }
      return(10);
    } else {
      if(doubleMatch(d1->x1[fi], d2->x1[fi], test_abs)==1 &&
         doubleMatch(d1->x2[fi], d2->x2[fi], test_abs)==1)
        continue;
      if(test_rel>0.0 &&
         doubleMatchRel(d1->x1[fi], d2->x1[fi], test_rel)==1 &&
         doubleMatchRel(d1->x2[fi], d2->x2[fi], test_rel)==1)
        continue;
      if(verbose>0) {
        printf("tac1.x1[%d] := %g\n", fi, d1->x1[fi]);
        printf("tac2.x1[%d] := %g\n", fi, d2->x1[fi]);
        printf("|diff| := %g\n", fabs(d1->x1[fi]-d2->x1[fi]));
        printf("tac1.x2[%d] := %g\n", fi, d1->x2[fi]);
        printf("tac2.x2[%d] := %g\n", fi, d2->x2[fi]);
        printf("|diff| := %g\n", fabs(d1->x2[fi]-d2->x2[fi]));
        printf("diff_limit := %g\n", test_abs);
        if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
      }
      return(11);
    }
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/** Check whether TAC weights are the same in two TAC data.

    If either absolute or relative difference is below the limit, the test is reported as passed.
    @sa tacCompareTimes, tacCompareConc, tacCompareNames.
    @return 0 in case of match, and >0 if no match or error.
    @author Vesa Oikonen
 */
int tacCompareWeights(
  /** Pointer to TAC structure. */
  TAC *d1,
  /** Pointer to TAC structure. */
  TAC *d2,
  /** Limit for accepted absolute difference. */
  const double test_abs,
  /** Optional limit for accepted relative difference |2*(w1-w2)/(w1+w2)| ;
      set to negative value to not test this; in case of zero mean, this test
      is assumed to fail, but test for absolute difference may still pass. */
  const double test_rel,
  /** Pointer to status data; enter NULL if not needed. */
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  if(verbose>1) printf("%s()\n", __func__);

  /* Check that required data exists */
  if(d1==NULL || d2==NULL || d1->sampleNr<1 || d2->sampleNr<1) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_NO_DATA);
    return 1;
  }

  /* Weight setting must match */
  if(d1->weighting!=d2->weighting) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different weighting setting.\n"); 
    return(2);
  }
  
  /* If no weights, then no checking either */
  if(d1->weighting==WEIGHTING_OFF && d2->weighting==WEIGHTING_OFF) return(0);
  if(d1->weighting==WEIGHTING_UNKNOWN && d2->weighting==WEIGHTING_UNKNOWN) return(0);
  
  /* Sample nr must match */
  if(d1->sampleNr!=d2->sampleNr) {
    statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
    if(verbose>0) printf("different sample nr.\n");
    return(3);
  }
  
  /* Compare */
  statusSet(status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  for(int fi=0; fi<d1->sampleNr; fi++) {
    if(doubleMatch(d1->w[fi], d2->w[fi], test_abs)==1) continue;
    if(test_rel>0.0 && doubleMatchRel(d1->w[fi], d2->w[fi], test_rel)==1) continue;
    if(verbose>0) {
      double s;
      s=fabs(d1->w[fi]-d2->w[fi]);
      printf("tac1.w[%d] := %g\n", fi, d1->w[fi]);
      printf("tac2.w[%d] := %g\n", fi, d2->w[fi]);
      printf("|diff| := %g\n", s);
      printf("diff_limit := %g\n", test_abs);
      if(test_rel>0.0) printf("rel_diff_limit := %g\n", test_rel);
    }
    return(10);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
