/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include "tpcextensions.h"
#include "test_tpctac.h"
/*****************************************************************************/

/*****************************************************************************/
int test_tacFormattxt(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  char *cptr;
  cptr=tacFormattxt(-1); if(cptr!=NULL) return 1;
  cptr=tacFormattxt(TAC_FORMAT_UNKNOWN); if(strcmp(cptr, "Unknown")) return 2;
  cptr=tacFormattxt(TAC_FORMAT_AMIDE); if(strcmp(cptr, "Amide")) return 3;
  cptr=tacFormattxt(9999); if(cptr!=NULL) return 4;
  if(verbose>1) {
    int i=0; printf("TAC file formats:\n");
    while((cptr=tacFormattxt(i))!=NULL) {printf("  %s\n", cptr); i++;}
  }

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacFormatIdentify(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  tacformat f;

  if(verbose>1) printf("\n testing with empty name\n");
  f=tacFormatIdentify(NULL);
  if(f!=TAC_FORMAT_UNKNOWN) return(1);
  f=tacFormatIdentify("");
  if(f!=TAC_FORMAT_UNKNOWN) return(2);
  
  if(verbose>1) printf("\n testing with valid format name\n");
  f=tacFormatIdentify("SIMPLE");
  if(f!=TAC_FORMAT_SIMPLE) return(11);
  f=tacFormatIdentify("ABSS-ALLOGG");
  if(f!=TAC_FORMAT_ABSS_ALLOGG) return(12);

  if(verbose>1) printf("\n testing with non-standard format name\n");
  f=tacFormatIdentify("DAT");
  if(f!=TAC_FORMAT_TSV_UK) return(21);
  f=tacFormatIdentify("TAC");
  if(f!=TAC_FORMAT_PMOD) return(22);

  if(verbose>1) printf("\n testing with invalid format names\n");
  f=tacFormatIdentify("ECAT");
  if(f!=TAC_FORMAT_UNKNOWN) return(41);
  f=tacFormatIdentify("DICOM");
  if(f!=TAC_FORMAT_UNKNOWN) return(42);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacFormatDetermine(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int format;

  if(verbose>1) printf("\nTrying with empty file name\n");
  format=tacFormatDetermine("", NULL);
  if(format!=TAC_FORMAT_UNKNOWN) return(1);

  if(verbose>1) printf("\nTrying with non-existing file\n");
  format=tacFormatDetermine("nonexisting.tac", NULL);
  if(format!=TAC_FORMAT_UNKNOWN) return(2);

  if(verbose>1) printf("\nTrying with Inveon file\n");
  format=tacFormatDetermine("P100670_Inveon.csv", NULL);
  if(format!=TAC_FORMAT_INVEON) return(11);

  if(verbose>1) printf("\nTrying with another Inveon file\n");
  format=tacFormatDetermine("ae392_inveon.csv", NULL);
  if(format!=TAC_FORMAT_INVEON) return(12);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacFormatWriteSupported(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  if(tacFormatWriteSupported(-1)) return(1);
  if(tacFormatWriteSupported(99999)) return(2);
  if(tacFormatWriteSupported(TAC_FORMAT_QVIEW)) return(3);
  if(tacFormatWriteSupported(TAC_FORMAT_INVEON)) return(4);
  if(!tacFormatWriteSupported(TAC_FORMAT_SIMPLE)) return(11);
  if(!tacFormatWriteSupported(TAC_FORMAT_PMOD)) return(12);

  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacWrite(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  FILE *fp;
  char fname[128];
  TAC tac; tacInit(&tac);
  
  if(verbose>1) printf("creating test data\n");
  ret=create_tac(&tac); if(ret!=TPCERROR_OK) return 1;

  /* Simple file format */
  if(verbose>1) printf("writing test data in simple format\n");
  strcpy(fname, "simple.dat");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac); return 11;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac, fp, TAC_FORMAT_SIMPLE, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 12;}

  /* DFT file format */
  if(verbose>1) printf("writing test data in DFT format\n");
  strcpy(fname, "data.dft");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac); return 21;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac, fp, TAC_FORMAT_DFT, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 22;}

  /* PMOD file format */
  if(verbose>1) printf("writing test data in PMOD format\n");
  strcpy(fname, "pmod.tac");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac); return 31;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac, fp, TAC_FORMAT_PMOD, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 32;}

  /* CSV file format */
  if(verbose>1) printf("writing test data in CSV format\n");
  strcpy(fname, "data.csv");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac); return 41;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac, fp, TAC_FORMAT_CSV_UK, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac); return 42;}

  tacFree(&tac);
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
int test_tacRead(
  TPCSTATUS *status
) {
  int verbose=0; if(status!=NULL) verbose=status->verbose;
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  if(verbose>0) {
    printf("\n=====================================\n");
    printf("\n%s\n", __func__);
    printf("\n=====================================\n");
  }

  int ret;
  FILE *fp;
  char fname[128];
  TAC tac1; tacInit(&tac1);
  TAC tac2; tacInit(&tac2);
  
  if(verbose>1) printf("\ncreating test data\n");
  ret=create_tac(&tac1); if(ret!=TPCERROR_OK) return 1;

  /* Simple file format */
  if(verbose>1) printf("\nwriting test data in simple format\n");
  strcpy(fname, "simple.dat");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 11;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_SIMPLE, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 12;}
  if(verbose>1) printf("\nreading test data in simple format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 13;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_SIMPLE) {tacFree(&tac1); tacFree(&tac2); return 14;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 15;}
  ret=0;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x[fi]-tac2.x[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 16;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac2);

  /* DFT file format */
  if(verbose>1) printf("\nwriting test data in DFT format\n");
  tac1.isframe=1;
  strcpy(fname, "test.dft");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 21;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_DFT, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 22;}
  if(verbose>1) printf("\nreading test data in DFT format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 23;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_DFT) {tacFree(&tac1); tacFree(&tac2); return 24;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 25;}
  ret=0;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x1[fi]-tac2.x1[fi])>1.0E-03) ret++;
    if(fabs(tac1.x2[fi]-tac2.x2[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 26;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 27;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 28;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac2);

  /* DFT file format with frame mid times */
  if(verbose>1) printf("\nwriting test data in DFT format with frame mid times\n");
  tac1.isframe=0;
  strcpy(fname, "test.dft");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 31;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_DFT, 1, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 32;}
  if(verbose>1) printf("\nreading test data in DFT format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 33;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_DFT) {tacFree(&tac1); tacFree(&tac2); return 34;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 35;}
  ret=0; if(tac1.isframe!=tac2.isframe) ret++;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x[fi]-tac2.x[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 36;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 37;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 38;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac2);

  /* PMOD tac file format */
  if(verbose>1) printf("\nwriting test data in PMOD tac format\n");
  tac1.isframe=1;
  strcpy(fname, "test.tac");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 41;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_PMOD, 0, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 42;}
  if(verbose>1) printf("\nreading test data in PMOD format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 43;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_PMOD) {tacFree(&tac1); tacFree(&tac2); return 44;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 45;}
  ret=0;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x1[fi]-tac2.x1[fi])>1.0E-03) ret++;
    if(fabs(tac1.x2[fi]-tac2.x2[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 46;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 47;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 48;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac2);

  /* PMOD bld file format */
  if(verbose>1) printf("\nwriting test data in PMOD bld format\n");
  tac1.isframe=0;
  strcpy(fname, "test.bld");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 51;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_PMOD, 0, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 52;}
  if(verbose>1) printf("\nreading test data in PMOD format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 53;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_PMOD) {tacFree(&tac1); tacFree(&tac2); return 54;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 55;}
  ret=0; if(tac1.isframe!=tac2.isframe) ret++;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x[fi]-tac2.x[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 56;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 57;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 58;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac2);

  /* CSV tac file format */
  if(verbose>1) printf("\nwriting test data in CSV tac format\n");
  tac1.isframe=1;
  for(int ri=0; ri<tac1.tacNr; ri++) {
    tac1.c[ri].size=10.0;
    for(int fi=0; fi<tac1.sampleNr; fi++) {
      tac1.c[ri].y[fi]*=1.001;
    }
  }
  strcpy(fname, "test.csv");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 61;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_CSV_INT, 0, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 62;}
  if(verbose>1) printf("\nreading test data in CSV format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 63;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_CSV_INT) {tacFree(&tac1); tacFree(&tac2); return 64;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 65;}
  ret=0;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x1[fi]-tac2.x1[fi])>1.0E-03) ret++;
    if(fabs(tac1.x2[fi]-tac2.x2[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 66;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 67;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 68;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac2);

  /* CSV blood file format */
  if(verbose>1) printf("\nwriting test data in CSV blood format\n");
  tac1.isframe=0;
  for(int ri=0; ri<tac1.tacNr; ri++) {
    tac1.c[ri].size=10.0;
    for(int fi=0; fi<tac1.sampleNr; fi++) {
      tac1.c[ri].y[fi]*=1.001;
    }
  }
  strcpy(fname, "test.tsv");
  fp=fopen(fname, "w"); if(fp==NULL) {tacFree(&tac1); return 71;}
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacWrite(&tac1, fp, TAC_FORMAT_TSV_UK, 0, status); fclose(fp);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); return 72;}
  if(verbose>1) printf("\nreading test data in CSV format\n");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 73;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_TSV_UK) {tacFree(&tac1); tacFree(&tac2); return 74;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 75;}
  ret=0;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x[fi]-tac2.x[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 76;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 77;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 78;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac1); tacFree(&tac2);



  /* Carimas txt file format */
  if(verbose>1) printf("\nreading existing test data in Carimas txt format\n");
  strcpy(fname, "carimas1.txt");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 81;}
  if(verbose>1) printf("reading existing comparison data\n");
  strcpy(fname, "carimas1.tac");
  ret=tacRead(&tac1, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 82;}
  if(verbose>1) printf("check contents\n");
  if(tac2.format!=TAC_FORMAT_CARIMAS_TXT) {
    tacFree(&tac1); tacFree(&tac2); return 83;}
  if(tac1.tacNr!=tac2.tacNr || tac1.sampleNr!=tac2.sampleNr) {
    tacFree(&tac1); tacFree(&tac2); return 84;}
  ret=0;
  for(int fi=0; fi<tac1.sampleNr; fi++) {
    if(fabs(tac1.x[fi]-tac2.x[fi])>1.0E-03) ret++;
    for(int ri=0; ri<tac1.tacNr; ri++) {
      if(fabs(tac1.c[ri].y[fi]-tac2.c[ri].y[fi])>1.0E-03) ret++;
    }
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 85;}
  if(verbose>5) {
    printf("tac1.cunit := %d\ntac2.cunit := %d\n", tac1.cunit, tac2.cunit);
    printf("tac1.tunit := %d\ntac2.tunit := %d\n", tac1.tunit, tac2.tunit);
  }
  if(tac1.cunit!=tac2.cunit) {tacFree(&tac1); tacFree(&tac2); return 86;}
  if(tac1.tunit!=tac2.tunit) {tacFree(&tac1); tacFree(&tac2); return 87;}
  ret=0;
  for(int ri=0; ri<tac1.tacNr; ri++) {
    if(strcmp(tac1.c[ri].name, tac2.c[ri].name)!=0) ret++;
  }
  if(ret>0) {tacFree(&tac1); tacFree(&tac2); return 88;}
  if(verbose>2) printf("  --> ok\n");
  tacFree(&tac1); tacFree(&tac2);



  /* Inveon CSV file format */
  if(verbose>1) printf("\nreading existing test data in Inveon csv format\n");
  strcpy(fname, "P100670_Inveon.csv");
  statusSet(status, __func__, __FILE__, __LINE__, 0);
  ret=tacRead(&tac2, fname, status);
  if(ret!=TPCERROR_OK) {tacFree(&tac1); tacFree(&tac2); return 91;}
  if(tac2.tacNr!=1 || tac2.sampleNr!=51) {
    tacFree(&tac1); tacFree(&tac2); return 92;}
  if(tac2.tunit!=UNIT_MIN) {tacFree(&tac1); tacFree(&tac2); return 93;}
  if(tac2.cunit!=UNIT_PID_PER_G) {tacFree(&tac1); tacFree(&tac2); return 94;}
  //printf("tacname='%s'\n", tac2.c[0].name);
  if(strcmp(tac2.c[0].name, "Spine")!=0) {tacFree(&tac1); tacFree(&tac2); return 95;}
  if(!doubleMatch(tac2.x[50], 55.0, 0.001)) {tacFree(&tac1); tacFree(&tac2); return 96;}
  //printf("%g\n", tac2.c[0].y[50]);
  if(!doubleMatch(tac2.c[0].y[50], 2.5124433, 1.0E-06)) {tacFree(&tac1); tacFree(&tac2); return 97;}
  if(verbose>2) printf("  --> ok\n");



  tacFree(&tac1); tacFree(&tac2);

  statusSet(status, __func__, __FILE__, __LINE__, 0);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
