/** @file parmatch.c
 *  @brief Verify that two PAR files have matching contents; for SW testing.
 *  @details Application name was previously resmatch. 
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 *  @test Add tests for all necessary file formats.
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpcpar.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Verify that the contents in two parameter files are similar.",
  "This can be used for automating software and analysis tool testing.",
  "By default exact match is required for numerical values, but this behaviour",
  "can be changed with options -rel and -abs (below); if both are set, then",
  "values are considered to match is either of the requirements is met.",
  "Header contents cannot be tested with this tool.",
  "Programs return code is 0, if files were matching, 10, if the files did",
  "not match, and 1-9 in case of error.",
  " ",
  "Usage: @P [options] filename1 filename2",
  " ",
  "Options:",
  " -p=<Y|n>",
  "     Parameter values are checked (y, default) or not checked (n).",
  " -sd=<y|N>",
  "     Parameter SD values are checked (y) or not checked (n, default).",
  " -cl=<y|N>",
  "     Parameter CL values are checked (y) or not checked (n, default).",
  " -tacnames=<y|N>",
  "     TAC names are checked (y) or not checked (n, default).",
  " -parnames=<Y|n>",
  "     Parameter names are checked (y, default) or not checked (n).",
  " -parunits=<y|N>",
  "     Parameter units are checked (y) or not checked (n, default).",
  " -wss=<y|N>",
  "     WSS values are checked (y) or not checked (n, default).",
  " -abs=<value>",
  "     Absolute differences must not exceed the specified limit.",
  " -rel=<value>",
  "     Relative differences must not exceed the specified percent limit.",
  " -par=<list>",
  "     List of parameter numbers to be tested; all by default.",
  " -tac=<list>",
  "     List of TAC numbers to be tested; all by default.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "Example 1:",
  "     @P -tac=1-3 -par=1,4 results1.csv results2.csv",
  "Example 2:",
  "     @P -abs=0.02 -rel=5 -parnames=n results1.csv results2.csv",
  " ",
  "See also: parformat, paradd, pardiff, iftmatch, iftisval, tacmatch",
  " ",
  "Keywords: parameter, tool, software testing",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  int ret, matchNr=0;
  char *cptr, parfile1[FILENAME_MAX], parfile2[FILENAME_MAX];
  double test_abs=-1.0;
  double test_rel=-1.0;
  /* Optional tests */
  int pvals=1;
  int sdvals=0;
  int clvals=0;
  int tacnames=0;
  int parnames=1;
  int parunits=0;
  int wssvals=0;
  INTLIST tlist, plist;

  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  intlistInit(&tlist); intlistInit(&plist);
  parfile1[0]=parfile2[0]=(char)0;
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "ABS=", 4)==0) {
      test_abs=atofVerified(cptr+4); if(!isnan(test_abs)) continue;
    } else if(strncasecmp(cptr, "REL=", 4)==0) {
      test_rel=atofVerified(cptr+4); if(!isnan(test_rel)) continue;
    } else if(strncasecmp(cptr, "P=", 2)==0) {
      if((pvals=tpcYesNo(cptr+2))>=0) continue;
    } else if(strncasecmp(cptr, "SD=", 3)==0) {
      if((sdvals=tpcYesNo(cptr+3))>=0) continue;
    } else if(strncasecmp(cptr, "CL=", 3)==0) {
      if((clvals=tpcYesNo(cptr+3))>=0) continue;
    } else if(strncasecmp(cptr, "WSS=", 4)==0) {
      if((wssvals=tpcYesNo(cptr+4))>=0) continue;
    } else if(strncasecmp(cptr, "TACNAMES=", 9)==0) {
      if((tacnames=tpcYesNo(cptr+9))>=0) continue;
    } else if(strncasecmp(cptr, "PARNAMES=", 9)==0) {
      if((parnames=tpcYesNo(cptr+9))>=0) continue;
    } else if(strncasecmp(cptr, "PARUNITS=", 9)==0) {
      if((parunits=tpcYesNo(cptr+9))>=0) continue;
    } else if(strncasecmp(cptr, "PAR=", 4)==0) {
      if(intlistExpandFromString(cptr+4, ", ", &plist, 1)>0) continue;
    } else if(strncasecmp(cptr, "TAC=", 4)==0) {
      if(intlistExpandFromString(cptr+4, ", ", &tlist, 1)>0) continue;
    }
    fprintf(stderr, "Error: invalid option '%s'.\n", argv[ai]);
    intlistFree(&tlist); intlistFree(&plist);
    return(1);
  } else break;

  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* Arguments */
  for(; ai<argc; ai++) {
    if(!parfile1[0]) {
      strlcpy(parfile1, argv[ai], FILENAME_MAX); continue;
    } else if(!parfile2[0]) {
      strlcpy(parfile2, argv[ai], FILENAME_MAX); continue;
    }
    fprintf(stderr, "Error: invalid argument '%s'.\n", argv[ai]);
    intlistFree(&tlist); intlistFree(&plist);
    return(1);
  }

  /* Is something missing? */
  if(!parfile2[0]) {tpcPrintUsage(argv[0], info, stdout); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    for(ai=0; ai<argc; ai++)
      printf("%s ", argv[ai]); 
    printf("\n");
    printf("parfile1 := %s\n", parfile1);
    printf("parfile2 := %s\n", parfile2);
    if(test_abs>=0.0) printf("test_abs := %g\n", test_abs);
    if(test_rel>=0.0) printf("test_rel := %g\n", test_rel);
    printf("pvals := %d\n", pvals);
    printf("sdvals := %d\n", sdvals);
    printf("clvals := %d\n", clvals);
    printf("wssvals := %d\n", wssvals);
    printf("tacnames := %d\n", tacnames);
    printf("parnames := %d\n", parnames);
    printf("parunits := %d\n", parunits);
    if(plist.nr>0) {
      printf("plist := %d", plist.i[0]);
      for(int i=1; i<plist.nr; i++) printf(", %d", plist.i[i]);
      printf("\n");
    }
    if(tlist.nr>0) {
      printf("tlist := %d", tlist.i[0]);
      for(int i=1; i<tlist.nr; i++) printf(", %d", tlist.i[i]);
      printf("\n");
    }
    fflush(stdout);
  }

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-1;

  /*
   *  Read PAR data
   */
  if(verbose>1) {printf("reading %s\n", parfile1); fflush(stdout);}
  PAR par1, par2; parInit(&par1); parInit(&par2);
  ret=parRead(&par1, parfile1, &status);
  if(ret) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), parfile1);
    parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
    return(2);
  }
  if(verbose>1) {printf("reading %s\n", parfile2); fflush(stdout);}
  ret=parRead(&par2, parfile2, &status);
  if(ret) {
    fprintf(stderr, "Error: %s (%s)\n", errorMsg(status.error), parfile2);
    parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
    return(3);
  }

  /*
   *  Check parameter names
   */
  if(parnames) {
    if(verbose>0) {printf("parameter names\n"); fflush(stdout);}
    if(plist.nr==0) {
      ret=parCompareParameterNames(&par1, &par2, -1, 1, &status);
    } else {
      ret=0;
      for(int pi=0; pi<plist.nr && !ret; pi++) {
        ret=parCompareParameterNames(&par1, &par2, plist.i[pi]-1, 1, &status);
        if(ret && verbose>1) printf("no match for parameter %d\n", plist.i[pi]);
      }
    }
    if(ret) {
      fprintf(stdout, "result: parameter names in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }  

  /*
   *  Check parameter units
   */
  if(parunits) {
    if(verbose>0) {printf("parameter units\n"); fflush(stdout);}
    if(plist.nr==0) {
      ret=parCompareParameterUnits(&par1, &par2, -1, &status);
    } else {
      ret=0;
      for(int pi=0; pi<plist.nr && !ret; pi++) {
        ret=parCompareParameterUnits(&par1, &par2, plist.i[pi]-1, &status);
        if(ret && verbose>1) printf("no match for parameter %d\n", plist.i[pi]);
      }
    }
    if(ret) {
      fprintf(stdout, "result: parameter units in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }  

  /*
   *  Check TAC ids
   */
  if(tacnames) {
    if(verbose>0) {printf("TAC names\n"); fflush(stdout);}
    if(tlist.nr==0) {
      ret=parCompareTacNames(&par1, &par2, -1, 1, &status);
    } else {
      ret=0;
      for(int ti=0; ti<tlist.nr && !ret; ti++) {
        ret=parCompareTacNames(&par1, &par2, tlist.i[ti]-1, 1, &status);
        if(ret && verbose>1) printf("no match for TAC %d\n", tlist.i[ti]);
      }
    }
    if(ret) {
      fprintf(stdout, "result: TAC names in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }  

  /*
   *  Check parameter values
   */
  if(pvals) {
    if(verbose>0) {printf("parameter values\n"); fflush(stdout);}
    if(plist.nr==0 && tlist.nr==0) {
      ret=parCompareParameters(&par1, &par2, -1, -1, 1, 0, 0, 
                               test_abs, 0.01*test_rel, &status);
    } else if(plist.nr==0) {
      ret=0;
      for(int ti=0; ti<tlist.nr && !ret; ti++) {
        ret=parCompareParameters(&par1, &par2, -1, tlist.i[ti]-1, 
                                 1, 0, 0, test_abs, 0.01*test_rel, &status);
        if(ret && verbose>1) 
          printf("no match for TAC %d\n", tlist.i[ti]);
      }
    } else if(tlist.nr==0) {
      ret=0;
      for(int pi=0; pi<plist.nr && !ret; pi++) {
        ret=parCompareParameters(&par1, &par2, plist.i[pi]-1, -1, 
                                 1, 0, 0, test_abs, 0.01*test_rel, &status);
        if(ret && verbose>1) 
          printf("no match for parameter %d\n", plist.i[pi]);
      }
    } else {
      ret=0; int ti, pi;
      for(ti=0; ti<tlist.nr && !ret; ti++) {
        for(pi=0; pi<plist.nr && !ret; pi++) {
          ret=parCompareParameters(&par1, &par2, plist.i[pi]-1, tlist.i[ti]-1, 
                                   1, 0, 0, test_abs, 0.01*test_rel, &status);
          if(ret && verbose>1) 
            printf("no match for parameter %d, TAC %d\n", 
                   plist.i[pi], tlist.i[ti]);
        }
      }
    }
    if(ret) {
      fprintf(stdout, "result: parameter values in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }  

  /*
   *  Check SD and CL values
   */
  if(sdvals) {
    if(verbose>0) {printf("parameter SD values\n"); fflush(stdout);}
    if(plist.nr==0 && tlist.nr==0) {
      ret=parCompareParameters(&par1, &par2, -1, -1, 0, 1, 0, 
                               test_abs, 0.01*test_rel, &status);
    } else if(plist.nr==0) {
      ret=0;
      for(int ti=0; ti<tlist.nr && !ret; ti++) {
        ret=parCompareParameters(&par1, &par2, -1, tlist.i[ti]-1, 
                                 0, 1, 0, test_abs, 0.01*test_rel, &status);
        if(ret && verbose>1) 
          printf("no match for TAC %d\n", tlist.i[ti]);
      }
    } else if(tlist.nr==0) {
      ret=0;
      for(int pi=0; pi<plist.nr && !ret; pi++) {
        ret=parCompareParameters(&par1, &par2, plist.i[pi]-1, -1, 
                                 0, 1, 0, test_abs, 0.01*test_rel, &status);
        if(ret && verbose>1) 
          printf("no match for parameter %d\n", plist.i[pi]);
      }
    } else {
      ret=0; int ti, pi;
      for(ti=0; ti<tlist.nr && !ret; ti++) {
        for(pi=0; pi<plist.nr && !ret; pi++) {
          ret=parCompareParameters(&par1, &par2, plist.i[pi]-1, tlist.i[ti]-1, 
                                   0, 1, 0, test_abs, 0.01*test_rel, &status);
          if(ret && verbose>1) 
            printf("no match for parameter %d, TAC %d\n", 
                   plist.i[pi], tlist.i[ti]);
        }
      }
    }
    if(ret) {
      fprintf(stdout, "result: parameter SD values in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }  
  if(clvals) {
    if(verbose>0) {printf("parameter CL values\n"); fflush(stdout);}
    if(plist.nr==0 && tlist.nr==0) {
      ret=parCompareParameters(&par1, &par2, -1, -1, 0, 0, 1, 
                               test_abs, 0.01*test_rel, &status);
    } else if(plist.nr==0) {
      ret=0;
      for(int ti=0; ti<tlist.nr && !ret; ti++) {
        ret=parCompareParameters(&par1, &par2, -1, tlist.i[ti]-1, 
                                 0, 0, 1, test_abs, 0.01*test_rel, &status);
        if(ret && verbose>1) 
          printf("no match for TAC %d\n", tlist.i[ti]);
      }
    } else if(tlist.nr==0) {
      ret=0;
      for(int pi=0; pi<plist.nr && !ret; pi++) {
        ret=parCompareParameters(&par1, &par2, plist.i[pi]-1, -1, 
                                 0, 0, 1, test_abs, 0.01*test_rel, &status);
        if(ret && verbose>1) 
          printf("no match for parameter %d\n", plist.i[pi]);
      }
    } else {
      ret=0; int ti, pi;
      for(ti=0; ti<tlist.nr && !ret; ti++) {
        for(pi=0; pi<plist.nr && !ret; pi++) {
          ret=parCompareParameters(&par1, &par2, plist.i[pi]-1, tlist.i[ti]-1, 
                                   0, 0, 1, test_abs, 0.01*test_rel, &status);
          if(ret && verbose>1) 
            printf("no match for parameter %d, TAC %d\n", 
                   plist.i[pi], tlist.i[ti]);
        }
      }
    }
    if(ret) {
      fprintf(stdout, "result: parameter CL values in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }  

  /*
   *  Check WSS values
   */
  if(wssvals) {
    if(verbose>0) {printf("WSS values\n"); fflush(stdout);}
    if(plist.nr==0) {
      ret=parCompareWSS(&par1, &par2, test_abs, -1, 0.01*test_rel, &status);
    } else {
      ret=0;
      for(int pi=0; pi<plist.nr && !ret; pi++) {
        ret=parCompareWSS(&par1, &par2, test_abs, plist.i[pi]-1, 
                          0.01*test_rel, &status);
        if(ret && verbose>1) printf("no match for parameter %d\n", plist.i[pi]);
      }
    }
    if(ret) {
      fprintf(stdout, "result: WSS values in %s and %s do not match.\n",
        parfile1, parfile2);
      parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);
      return(10);
    }
    matchNr++;
  }

  parFree(&par1); parFree(&par2); intlistFree(&tlist); intlistFree(&plist);

  if(matchNr==0) {
    fprintf(stderr, "Error: no tests applied.\n"); fflush(stderr);
    return(1);
  } else if(verbose>=0) {
    fprintf(stdout, "result: match was found.\n"); fflush(stdout);
  }
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
