#!/bin/bash
#: Title      : test_pardiff
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../pardiff$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"


if [ ! -f plasma.bld ]; then
  printf "Failed: required data file does not exist.\n"
  exit 1
fi

if [ ! -f correct_100.csv ] || [ ! -f correct_110.csv ]; then
  printf "Failed: required data file does not exist.\n"
  exit 1
fi
if [ ! -f correct_130.csv ] || [ ! -f correct_131.csv ]; then
  printf "Failed: required data file does not exist.\n"
  exit 1
fi


printf "\n make test data \n"

if [ ! -f blood.bld ] || [ ! -f frames_long.dat ]; then
  rm -f pre.tac post.tac
  p2blood FDG plasma.bld 0.42 blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f pre.tac ]; then
  printf "\n Simulate TTACs for pre study \n"
  rm -f pre*.csv
  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.00 0.0 0 0 0 3 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 revers L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.2 0.1  0.05 0.0 0 0 0 4 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 refer L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.10 0.0 0 0 0 5 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 reg1 L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.1 0.05 0.10 0.0 0 0 0 4 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 reg2 L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.2 0.2  0.10 0.0 0 0 0 5 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 reg3 L1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp.tac frames_long.dat pre.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


if [ ! -f post.tac ]; then
  printf "\n Simulate TTACs for post study \n"
  rm -f post*.csv
  p2t_v3c -nosub plasma.bld blood.bld 0.3 0.15 0.000 0.0 0 0 0 3 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 revers L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.2 0.1  0.025 0.0 0 0 0 4 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 refer L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.3  0.15 0.05 0.0 0 0 0 5 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 reg1 L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.1  0.05 0.05 0.0 0 0 0 4 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 reg2 L2
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  p2t_v3c -nosub plasma.bld blood.bld 0.15 0.15 0.06 0.0 0 0 0 5 30 temp2.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacren temp2.tac 0 reg3 L1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd temp.tac temp2.tac 1
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  simframe temp.tac frames_long.dat post.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f prek3.csv ] || [ ! -f prek3.res ]; then
  printf "\n 2TCM fit for pre study... \n"
  fitk3 -svg=prek3.svg -sd=no -cl=no plasma.bld blood.bld pre.tac 60 prek3.res
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  parformat -f=csv-uk prek3.res prek3.csv
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f postk3.csv ] || [ ! -f postk3.res ]; then
  printf "\n 2TCM fit for post study... \n"
  fitk3 -svg=postk3.svg -sd=n -cl=n plasma.bld blood.bld post.tac 60 postk3.res
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  parformat -f=csv-uk postk3.res postk3.csv
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f prek2.csv ] || [ ! -f prek2.res ]; then
  printf "\n 1TCM fit for pre study... \n"
  fitk2 -svg=prek2.svg -sd=no -cl=no plasma.bld blood.bld pre.tac 60 prek2.res
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  parformat -f=csv-uk prek2.res prek2.csv
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f postk2.csv ] || [ ! -f postk2.res ]; then
  printf "\n 1TCM fit for post study... \n"
  fitk2 -svg=postk2.svg -sd=n -cl=n plasma.bld blood.bld post.tac 60 postk2.res
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  parformat -f=csv-uk postk2.res postk2.csv
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f preneg.csv ] || [ ! -f postneg.csv ] || [ ! -f correct_neg.csv ]; then
  printf "\n Making test data with negative parameters \n"

  printf "parameters;P1;P2\n" > preneg.csv
  printf "reg1;10;-5\n"      >> preneg.csv
  printf "reg2;10;-5\n"      >> preneg.csv
  printf "reg3;10;0\n"       >> preneg.csv

  printf "parameters;P1;P2\n"  > postneg.csv
  printf "reg1;-5;-10\n"      >> postneg.csv
  printf "reg2;5;10\n"        >> postneg.csv
  printf "reg3;0;10\n"        >> postneg.csv

  printf "parameters;P1[%%];P2[%%]\n"  > correct_neg.csv
  printf "reg1;-150;-100\n"           >> correct_neg.csv
  printf "reg2;-50;300\n"             >> correct_neg.csv
  printf "reg3;-100;0\n"              >> correct_neg.csv

  printf "parameters;P1[%%];P2[%%]\n"  > correct_negocc.csv
  printf "reg1;150;-100\n"            >> correct_negocc.csv
  printf "reg2;50;300\n"              >> correct_negocc.csv
  printf "reg3;100;0\n"               >> correct_negocc.csv
fi


if [ ! -f presdcl.csv ] || [ ! -f postsdcl.csv ] || [ ! -f correct_sdcl.csv ]; then
  printf "\n Making test data with SD and CL of parameters \n"

  printf "parameters;P1[1/min];95%%CL1;95%%CL2;P2[mL/mL];SD\n" > presdcl.csv
  printf "reg1;1;0,9;1,1;2;0,1\n"                             >> presdcl.csv
  printf "reg2;2;1,9;2,1;3;0,2\n"                             >> presdcl.csv
  printf "reg3;3;2,9;3,1;4;0,3\n"                             >> presdcl.csv

  printf "parameters;P1[1/min];95%%CL1;95%%CL2;P2[mL/mL];SD\n" > postsdcl.csv
  printf "reg1;0,9;0,8;1;1,8;0,11\n"                          >> postsdcl.csv
  printf "reg2;1,8;1,6;2;2,7;0,22\n"                          >> postsdcl.csv
  printf "reg3;2,7;2,6;3;3,0;0,33\n"                          >> postsdcl.csv

  printf "parameters;P1[%%];P2[%%]\n"  > correct_sdcl.csv
  printf "reg1;10;10\n"               >> correct_sdcl.csv
  printf "reg2;10;10\n"               >> correct_sdcl.csv
  printf "reg3;10;25\n"               >> correct_sdcl.csv
fi


if [ ! -f tacname1.csv ] || [ ! -f tacname2.csv ]; then
  printf "\n Making test data with different TAC names \n"

  printf "parameters;P1;P2;P3\n" > tacname1.csv
  printf "reg1;1;2;3\n"         >> tacname1.csv
  printf "reg2;2;3;4\n"         >> tacname1.csv
  printf "reg3;3;4;5\n"         >> tacname1.csv

  printf "parameters;P1;P2;P3\n" > tacname2.csv
  printf "reg1;1;2;3\n"         >> tacname2.csv
  printf "reg2;2;3;4\n"         >> tacname2.csv
  printf "reg4;4;5;6\n"         >> tacname2.csv
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Default difference calculation. \n"
printf " Test case 2: Option to save difference percentages without decimals. \n"
printf " Expected result: differences are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -round prek3.csv postk3.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.001 -rel=0.01 correct_100.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.1 \n"
printf " Test case: The same with RES format. \n"
printf " Expected result: differences are calculated and saved. \n\n"

rm -f output.res
$PROGRAM -round prek3.res postk3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.001 -rel=0.01 correct_100.csv output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.0.2 \n"
printf " Test case: Differencies of negative parameters. \n"
printf " Expected result: differences are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM preneg.csv postneg.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -abs=0.001 -rel=0.01 correct_neg.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n------------------------------------------------------------------\n"


printf "\n 1.1.0 \n"
printf " Test case: Occupancy calculation. \n"
printf " Expected result: occupancies are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -occ prek3.csv postk3.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.1 -rel=0.1 correct_110.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: The same with RES format. \n"
printf " Expected result: occupancies are calculated and saved. \n\n"

rm -f output.res
$PROGRAM -occupance prek3.res postk3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.1 -rel=0.1 correct_110.csv output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.1.2 \n"
printf " Test case: Occupancies of negative parameters. \n"
printf " Expected result: occupancies are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -OCC preneg.csv postneg.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -abs=0.001 -rel=0.01 correct_negocc.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n------------------------------------------------------------------\n"


printf "\n 1.2.0 \n"
printf " Test case: Inhibition calculation. \n"
printf " Expected result: inhibitions are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -inh prek3.csv postk3.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.1 -rel=0.1 correct_110.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.1 \n"
printf " Test case: The same with RES format. \n"
printf " Expected result: Inhibition are calculated and saved. \n\n"

rm -f output.res
$PROGRAM -inhibition prek3.res postk3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.1 -rel=0.1 correct_110.csv output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.2.2 \n"
printf " Test case: Inhibition of negative parameters. \n"
printf " Expected result: Inhibition are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -INH preneg.csv postneg.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -abs=0.001 -rel=0.01 correct_negocc.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n------------------------------------------------------------------\n"

printf "\n 1.3.0 \n"
printf " Test case: Optional difference calculation, no percentage. \n"
printf " Expected result: differences are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -dif prek3.csv postk3.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.001 -rel=0.01 correct_130.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 1.3.1 \n"
printf " Test case: The same with RES format. \n"
printf " Expected result: differences are calculated and saved. \n\n"

rm -f output.res
$PROGRAM -difference prek3.res postk3.res output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -par=1-7 -tacnames=y -abs=0.001 -rel=0.01 correct_130.csv output.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n------------------------------------------------------------------\n"


printf "\n 1.4.0 \n"
printf " Test case: Data with parameter SD and CL. \n"
printf " Expected result: occupancies are calculated and saved. \n\n"

rm -f output.csv
$PROGRAM -occ presdcl.csv postsdcl.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -abs=0.01 -rel=0.1 correct_sdcl.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   Testing a few error situations \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption presdcl.csv postsdcl.csv output.csv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: Nonexisting input filename. \n"
printf " Expected result: Error. \n\n"

$PROGRAM nonexisting.csv postsdcl.csv output.csv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open file (nonexisting.csv)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM presdcl.csv nonexisting.csv output.csv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "cannot open file (nonexisting.csv)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.2.0 \n"
printf " Test case: Wrong input file format. \n"
printf " Expected result: Error. \n\n"

printf "# stupid format \n\n" > stupid.csv
printf "parameters,meh,duh\n" >> stupid.csv
printf "0,2,3\n" >> stupid.csv
printf "1,2\n" >> stupid.csv
printf "2,3,4\n" >> stupid.csv

$PROGRAM stupid.csv postsdcl.csv output.csv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid file format (stupid.csv)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM presdcl.csv stupid.csv output.csv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid file format (stupid.csv)"
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n------------------------------------------------------------------\n"

printf "\n 2.3.0 \n"
printf " Test case: different TAC nr in file1 and file2. \n"
printf " Expected result: Error. \n\n"

cp -f prek3.res less.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
resdel less.res reg1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

rm -f output.res
$PROGRAM less.res prek3.res output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different TAC number in parameter files."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res
$PROGRAM prek3.res less.res output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different TAC number in parameter files."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.3.1 \n"
printf " Test case: different TAC names in file1 and file2. \n"
printf " Expected result: Error. \n\n"

rm -f output.csv
$PROGRAM tacname1.csv tacname2.csv output.csv 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different TAC names in parameter files."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n------------------------------------------------------------------\n"

printf "\n 2.4.0 \n"
printf " Test case: different parameter nr in file1 and file2. \n"
printf " Expected result: Error. \n\n"

rm -f output.res
$PROGRAM prek2.res postk3.res output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different parameters in parameter files."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res
$PROGRAM prek3.res postk2.res output.res 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "different parameters in parameter files."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.4.1 \n"
printf " Test case 1: different parameter nr in file1 and file2. \n"
printf " Test case 2: option --force. \n"
printf " Expected result: No error. \n\n"

rm -f output.res
$PROGRAM --force prek2.res postk3.res output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

rm -f output.res
$PROGRAM -f prek3.res postk2.res output.res 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n------------------------------------------------------------------\n"

printf "\n 2.5.0 \n"
printf " Test case: output filename not given. \n"
printf " Expected result: output is written in stdout. \n\n"

rm -f output.csv
$PROGRAM --silent prek2.csv postk2.csv > stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
$PROGRAM prek2.csv postk2.csv output.csv
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -tacnames=y -abs=0.01 -rel=0.1 output.csv stdout.txt
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

