/** @file hist2svg.c
 *  @brief Plot histogram from data given in TAC format.
 *  @copyright (c) Turku PET Centre
 *  @author Vesa Oikonen
 */
/// @cond
/*****************************************************************************/
#include "tpcclibConfig.h"
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
/*****************************************************************************/
#include "tpcextensions.h"
#include "tpcift.h"
#include "tpctac.h"
#include "tpctacmod.h"
/*****************************************************************************/

/*****************************************************************************/
static char *info[] = {
  "Plot histogram from data given in TAC format.",
  " ",
  "Usage: @P [options] datafile svgfile",
  " ",
  "Options:",
  " -c=<Number>",
  "     The dataset in file to use, in case there are several; by default 1.",
  " -x1=<start of x axis>",
  " -x2=<end of x axis>",
  " -y1=<start of y axis>",
  " -y2=<end of y axis>",
  " -mt=\"<main title>\"",
  "     User-specified string to set as main title.",
  " -stdoptions", // List standard options like --help, -v, etc
  " ",
  "See also: tac2svg, imghist",
  " ",
  "Keywords: TAC, tool",
  0};
/*****************************************************************************/

/*****************************************************************************/
/* Turn on the globbing of the command line, since it is disabled by default in
   mingw-w64 (_dowildcard=0); in MinGW32 define _CRT_glob instead, if necessary;
   In Unix&Linux wildcard command line processing is enabled by default. */
/*
#undef _CRT_glob
#define _CRT_glob -1
*/
int _dowildcard = -1;
/*****************************************************************************/

/*****************************************************************************/
/**
 *  Main
 */
int main(int argc, char **argv)
{
  int ai, help=0, version=0, verbose=1;
  char tacfile[FILENAME_MAX], svgfile[FILENAME_MAX];
  char main_title[64];
  double preset_xmin, preset_xmax, preset_ymin, preset_ymax;
  int ci=0;

  
  /*
   *  Get arguments
   */
  if(argc==1) {tpcPrintUsage(argv[0], info, stderr); return(1);}
  tacfile[0]=svgfile[0]=(char)0;
  main_title[0]=(char)0;
  preset_xmin=preset_xmax=preset_ymin=preset_ymax=nan("");
  /* Options */
  for(ai=1; ai<argc; ai++) if(*argv[ai]=='-') {
    if(tpcProcessStdOptions(argv[ai], &help, &version, &verbose)==0) continue;
    char *cptr=argv[ai]+1; if(*cptr=='-') cptr++; if(!*cptr) continue;
    if(strncasecmp(cptr, "MT=", 3)==0) {
      if(strncpyClean(main_title, cptr+3, 64)) continue;
    } else if(strncasecmp(cptr, "X1=", 3)==0) {
      if(atofCheck(cptr+3, &preset_xmin)==0) continue;
    } else if(strncasecmp(cptr, "X2=", 3)==0) {
      if(atofCheck(cptr+3, &preset_xmax)==0) continue;
    } else if(strncasecmp(cptr, "Y1=", 3)==0) {
      if(atofCheck(cptr+3, &preset_ymin)==0) continue;
    } else if(strncasecmp(cptr, "Y2=", 3)==0) {
      if(atofCheck(cptr+3, &preset_ymax)==0) continue;
    } else if(strncasecmp(cptr, "C=", 2)==0) {
      if(atoiCheck(cptr+2, &ci)==0 && ci>0) {ci--; continue;}
    }
    fprintf(stderr, "Error: invalid option '%s'\n", argv[ai]);
    return(1);
  } else break; // tac name argument may start with '-'

  TPCSTATUS status; statusInit(&status);
  statusSet(&status, __func__, __FILE__, __LINE__, TPCERROR_OK);
  status.verbose=verbose-3;
  
  /* Print help or version? */
  if(help==2) {tpcHtmlUsage(argv[0], info, ""); return(0);}
  if(help) {tpcPrintUsage(argv[0], info, stdout); return(0);}
  if(version) {tpcPrintBuild(argv[0], stdout); return(0);}

  /* The first argument (non-option) is the filename */
  if(ai<argc) {strlcpy(tacfile, argv[ai], FILENAME_MAX); ai++;}
  if(ai<argc) {strlcpy(svgfile, argv[ai], FILENAME_MAX); ai++;}
  else {fprintf(stderr, "Error: missing filename.\n"); return(1);}
  if(ai<argc) {fprintf(stderr, "Error: extra command-line argument.\n"); return(1);}
  
  if(preset_xmin>=preset_xmax || preset_ymin>=preset_ymax) {
    fprintf(stderr, "Error: invalid axis range settings.\n"); return(1);}

  /* In verbose mode print arguments and options */
  if(verbose>1) {
    printf("tacfile := %s\n", tacfile);
    printf("svgfile := %s\n", svgfile);
    if(main_title[0]) printf("main_title := %s\n", main_title);
    if(isfinite(preset_xmin)) printf("preset_xmin := %g\n", preset_xmin);
    if(isfinite(preset_xmax)) printf("preset_xmax := %g\n", preset_xmax);
    if(isfinite(preset_ymin)) printf("preset_ymin := %g\n", preset_ymin);
    if(isfinite(preset_ymax)) printf("preset_ymax := %g\n", preset_ymax);
  }

  /* Read the file */
  if(verbose>1) printf("reading %s\n", tacfile);
  TAC tac; tacInit(&tac);
  tacRead(&tac, tacfile, &status);
  if(status.error!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(2);
  }
  if(verbose>2) {
    printf("fileformat := %s\n", tacFormattxt(tac.format));
    printf("tacNr := %d\n", tac.tacNr);
    printf("sampleNr := %d\n", tac.sampleNr);
    printf("frames := %d\n", tac.isframe);
    if(tac.tacNr>1) printf("tac := %s\n", tac.c[ci].name);
  }
  if(tac.isframe) tacSetX(&tac, NULL);
  if(ci>=tac.tacNr) {
    fprintf(stderr, "Error: data file does not contain data set %d.\n", 1+ci);
    tacFree(&tac); return(2);
  }
  /* Delete other datasets than the selected one */
  if(tac.tacNr>1) {
    for(int i=0; i<tac.tacNr; i++) if(i==ci) tac.c[i].sw=0; else tac.c[i].sw=1;
    int i=tac.tacNr-1;
    while(i>=0) {
      if(tac.c[i].sw && tacDeleteTACC(&tac, i)!=TPCERROR_OK) {
        fprintf(stderr, "Error: invalid data file.\n");
        tacFree(&tac); return(2);
      }
      i--;
    }
  }

  /* Determine the plot min and max values */
  double xmin=nan(""), xmax=nan(""), ymin=nan(""), ymax=nan("");
  if(tacXRange(&tac, &xmin, &xmax)) {
    fprintf(stderr, "Error: invalid data x range.\n");
    tacFree(&tac); return(2);
  }
  if(verbose>3) printf("data_xrange := %g %g\n", xmin, xmax);
  if((isfinite(preset_xmin) && preset_xmin>=xmax) || (isfinite(preset_xmax) && preset_xmax<=xmin)) {
    fprintf(stderr, "Error: invalid x axis range settings.\n");
    tacFree(&tac); return(1);
  }
  if(isfinite(preset_xmin)) xmin=preset_xmin;
  if(isfinite(preset_xmax)) xmax=preset_xmax;
  /* Determine the plot min and max y values inside the x range */
  if(tacYRangeInXRange(&tac, 0, xmin, xmax, &ymin, &ymax, NULL, NULL, NULL, NULL)) {
    fprintf(stderr, "Error: invalid data y range.\n");
    tacFree(&tac); return(2);
  }
  if(verbose>3) printf("data_yrange := %g %g\n", ymin, ymax);
  if((isfinite(preset_ymin) && preset_ymin>=ymax) || (isfinite(preset_ymax) && preset_ymax<=ymin)) {
    fprintf(stderr, "Error: invalid y axis range settings.\n");
    tacFree(&tac); return(1);
  }
  if(ymin>0.0) ymin=0.0;
  else if(ymax<0.0) ymax=0.0;
  if(isfinite(preset_ymin)) ymin=preset_ymin;
  if(isfinite(preset_ymax)) ymax=preset_ymax;
  if(verbose>3) {
    printf("plotted_xrange := %g %g\n", xmin, xmax);
    printf("plotted_yrange := %g %g\n", ymin, ymax);
  }


  /* Plot */
  tacPlotHistogramSVG(&tac, main_title, xmin, xmax, ymin, ymax, svgfile, &status);
  if(status.error!=TPCERROR_OK) {
    fprintf(stderr, "Error: %s\n", errorMsg(status.error));
    tacFree(&tac); return(10);
  }


  tacFree(&tac);
  return(0);
}
/*****************************************************************************/

/*****************************************************************************/
/// @endcond
