#!/bin/bash
#: Title      : test_conv1tcm
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../conv1tcm$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f input.dat ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f input1s.dat ] || [ ! -f inputs.svg ]; then
  interpol -c=0,60,1 input.dat input1s.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=n -mt="Input TACs" inputs.svg -s input.dat -l input1s.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f input01s.dat ]; then
  interpol -c=0,60,0.1 input.dat input01s.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case: Input data has sample frequency of 0.1 and starts from zero. \n"
printf " Expected result: Simulated TAC is the same as with Adams-Moulton method. \n\n"


rm -f output01s_am.dat output01s_conv.dat

$PROGRAM -d3 input01s.dat 0.1 0.05 output01s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

p2t_3c -nosub input01s.dat 0.1 0.05 0 0 0 0 output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Sample frequency of 0.1" sim01s.svg -l input1s.dat output01s_am.dat output01s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.3 -rel=1 output01s_am.dat output01s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Input data has sample frequency of one and starts from zero. \n"
printf " Expected result: Simulated TAC is not too far from Adams-Moulton method. \n\n"

rm -f output1s_am.dat output1s_conv.dat

$PROGRAM input1s.dat 0.1 0.05 output1s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

p2t_3c -nosub input1s.dat 0.1 0.05 0 0 0 0 output1s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Sample frequency of one" sim1s.svg -l input1s.dat output1s_am.dat output1s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=2.0 -rel=10 output1s_am.dat output1s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 1.0.2 \n"
printf " Test case: Input data has variable sample frequency but starts from zero. \n"
printf " Expected result: Simulated TAC is not too far from Adams-Moulton method. \n\n"

rm -f outputVs_am.dat outputVs_conv.dat

$PROGRAM input.dat 0.1 0.05 outputVs_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

p2t_3c -nosub input.dat 0.1 0.05 0 0 0 0 outputVs_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Variable sample frequency" simVs.svg input.dat outputVs_am.dat outputVs_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=2.0 -rel=10 outputVs_am.dat outputVs_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n----------------------------------------------------------------\n"



printf "\n 1.1.0 \n"
printf " Test case 1: Input data has variable sample frequency but starts from zero. \n"
printf " Test case 2: Sample interval is shortened with option -i. \n"
printf " Expected result: The shorter the sample interval, the closer the simulated TAC is to the Adams-Moulton method. \n\n"

p2t_3c -nosub input01s.dat 0.1 0.05 0 0 0 0 truth_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -f=input.dat truth_am.dat output_am.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacstudy --force output_am.tac simulation
tacren output_am.tac 1 AM__

$PROGRAM -i=0.01 input.dat 0.1 0.05 output1_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -i=0.1 input.dat 0.1 0.05 output2_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -i=0.5 input.dat 0.1 0.05 output3_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -i=1.0 input.dat 0.1 0.05 output4_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM -i=2.0 input.dat 0.1 0.05 output5_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacadd -ovr output.tac output1_conv.dat output2_conv.dat output3_conv.dat output4_conv.dat output5_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tacstudy --force output.tac simulation
tacren output.tac 1 Convol_001_
tacren output.tac 2 Convol_010_
tacren output.tac 3 Convol_050_
tacren output.tac 4 Convol_100_
tacren output.tac 5 Convol_200_
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -legend=y -mt="Effect of option -i" optioni.svg -s output_am.tac -l output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

auc0=$(interpol -i -x=20 -header=no output_am.tac)
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
auc1=$(interpol -i -x=20 -header=no output1_conv.dat)
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
auc2=$(interpol -i -x=20 -header=no output2_conv.dat)
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
auc3=$(interpol -i -x=20 -header=no output3_conv.dat)
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
auc4=$(interpol -i -x=20 -header=no output4_conv.dat)
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
auc5=$(interpol -i -x=20 -header=no output5_conv.dat)
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

printf "0.00\t%s\n" $auc0 >  auc.dat
printf "0.01\t%s\n" $auc1 >> auc.dat
printf "0.10\t%s\n" $auc2 >> auc.dat
printf "0.50\t%s\n" $auc3 >> auc.dat
printf "1.00\t%s\n" $auc4 >> auc.dat
# Leave out to better see the plot
#printf "2.00\t%s\n" $auc5 >> auc.dat

fit_line -svg=auc.svg auc.dat 0.001 1.1 auc.res
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift auc.res auc.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.01 auc.ift r 1.0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
#iftlist auc.ift Intercept
iftisval -abs=0.05 auc.ift Intercept $auc0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.1.1 \n"
printf " Test case 1: Input data has variable sample frequency but starts from zero. \n"
printf " Test case 2: Shortened sample with option -i. \n"
printf " Expected result: Simulated TAC is close to Adams-Moulton method from original frames. \n\n"

rm -f outputVsi_conv.dat

$PROGRAM -i=0.005 input.dat 0.1 0.05 outputVsi_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.25 -rel=0.1 outputVs_am.dat outputVsi_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   Test some errors \n"
printf "===================================================================\n"


printf "\n 2.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption input.dat 1.0 2.0 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.0 \n"
printf " Test case: Extra argument when simulating. \n"
printf " Expected result: Error. \n\n"

$PROGRAM input.dat 1.0 2.0 output.dat extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

printf "\n 2.2.0 \n"
printf " Test case: Missing parameter. \n"
printf " Expected result: Error. \n\n"

$PROGRAM input.dat output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

