#!/bin/bash
#: Title      : test_sim_h2o
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../sim_h2o$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f rvlv.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f rvlv.tac ] || [ ! -f rv.tac ] || [ ! -f lv.tac ]; then
  printf "\n calculate input data for simulations \n"
  fit2dat -c=0,420,1 rvlv.fit rvlv.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
  tacadd -ovr rv.tac rvlv.tac RV
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tacadd -ovr lv.tac rvlv.tac LV
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f ss.bld ]; then
  printf "\n make SS input data for simulations \n"
  printf "time[seconds]	AIF[kBq/mL]\n" > temp.bld
  printf "0	0\n" >> temp.bld
  printf "1	0\n" >> temp.bld
  printf "21	100\n" >> temp.bld
  printf "1201	100\n" >> temp.bld

  interpol -c=0,1200,1 temp.bld ss.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi



printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Only PAR file name given as argument. \n"
printf " Expected result 1: Parameter file saved. \n"
printf " Expected result 2: Error if file exists. \n\n"

rm -f output.par
$PROGRAM output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat output.par 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt format TSV-UK
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.par model radiowater2
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


$PROGRAM output.par 1>stdout2.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout2.txt Error "missing file names."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Template parameter file used with decent input. \n"
printf " Expected result: Simulated TTACs saved. \n\n"

rm -f output.tac output101.svg
$PROGRAM output.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="Test 1.0.1" -legend=y output101.svg -l lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Simulate TTACs without dispersion. \n"
printf " Expected result: Fitting simulated TTACs give correct parameter values. \n\n"

rm -f test.tac output110.par output110.svg

printf "# model := radiowater2\n"  > test.par
printf "Parameters	f[mL/(mL*min)]	p[mL/mL]	PTF[mL/mL]	Va[mL/mL]	deltaT[sec]	tau[sec]\n" >> test.par
printf "tac1	2	1	0.5	0.0	5	0\n" >> test.par
printf "tac2	2	1	0.4	0.1	5	0\n" >> test.par

printf "Parameters	Flow[mL/(mL*min)]	pWater[mL/mL]	Va[mL/mL]	delayT[sec]	k2[1/min]	WSS\n" > correct.par
printf "tac1	1	0.5	0.0	5	2	0\n" >> correct.par
printf "tac2	0.88889	0.44444	0.1	5	2	0\n" >> correct.par

$PROGRAM test.par lv.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="Test 1.1.0" -legend=y output110.svg -l lv.tac test.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit_h2o -delay=5 -fpt -ml -k2 lv.tac test.tac 420 output110.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -p=y -tacnames=n -parnames=y -parunits=n -wss=n -abs=0.001 -rel=1 correct.par output110.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Simulate TTACs with dispersion. \n"
printf " Expected result: Fitting simulated TTACs using pre-dispersed input give correct parameter values. \n\n"

rm -f test2.tac output111.par output111.svg

printf "# model := radiowater2\n"  > test2.par
printf "Parameters	f[mL/(mL*min)]	p[mL/mL]	PTF[mL/mL]	Va[mL/mL]	deltaT[sec]	tau[sec]\n" >> test2.par
printf "tac1	2	1	0.5	0.0	5	6\n" >> test2.par
printf "tac2	2	1	0.4	0.1	5	6\n" >> test2.par

$PROGRAM test2.par lv.tac test2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="Test 1.1.1" -legend=y output111.svg -l lv.tac test2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

disp4dft ON lv.tac 6 tempinput.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

fit_h2o -delay=5 -fpt -ml -k2 tempinput.tac test2.tac 420 output111.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parmatch -p=y -tacnames=n -parnames=y -parunits=n -wss=n -abs=0.001 -rel=1 correct.par output111.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.1.2 \n"
printf " Test case: Simulate TTACs with dispersion in steady state. \n"
printf " Expected result: Correct activity level. \n\n"

rm -f test3.tac output112.par output112.svg

printf "# model := radiowater2\n"  > test3.par
printf "Parameters	f[mL/(mL*min)]	p[mL/mL]	PTF[mL/mL]	Va[mL/mL]	deltaT[sec]	tau[sec]\n" >> test3.par
printf "tac1	1	0.8	1	0	5	10\n" >> test3.par
printf "tac2	2	0.8	1	0	10	4\n" >> test3.par

$PROGRAM test3.par ss.bld test3.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="Test 1.1.2" -legend=y output112.svg -l ss.bld test3.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccut -ls test3.tac 0 0 temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -nv -ift temp.tac > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.1 temp.ift value[1][1] 80
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval -abs=0.1 temp.ift value[2][1] 80
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

