#!/bin/bash
#: Title      : test_sim_wliv
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi
PROGRAM=../../sim_wliv$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f rvlv.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f lv.tac ]; then
  printf "\n calculate input data for simulations \n"
  fit2dat -c=0,420,1 rvlv.fit rvlv.tac
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
  tacadd -ovr lv.tac rvlv.tac LV
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi

if [ ! -f ss.bld ]; then
  printf "\n make SS input data for simulations \n"
  printf "time[seconds]	AIF[kBq/mL]\n" > temp.bld
  printf "0	0\n" >> temp.bld
  printf "1	0\n" >> temp.bld
  printf "21	100\n" >> temp.bld
  printf "1201	100\n" >> temp.bld

  interpol -c=0,1200,1 temp.bld ss.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  printf "\n ok \n"
fi


printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"



printf "\n 1.0.0 \n"
printf " Test case: Only PAR file name given as argument. \n"
printf " Expected result 1: Parameter file saved. \n"
printf " Expected result 2: Error if file exists. \n\n"

rm -f output.par output.ift
$PROGRAM output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat output.par 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt format TSV-UK
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.par model radiowater-liver
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift output.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval output.ift K1a
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift kGI
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output.ift beta
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


$PROGRAM output.par 1>stdout2.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout2.txt Error "parameter template exists."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.1 \n"
printf " Test case: Template parameter file used with decent input. \n"
printf " Expected result: Simulated TTACs saved. \n\n"

rm -f output.tac output101.svg
$PROGRAM output.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
tac2svg -mt="Test 1.0.1" -legend=y output101.svg -l lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.0 \n"
printf " Test case: Only PAR file name and option -irf given as argument. \n"
printf " Expected result: Parameter file with beta saved. \n\n"

rm -f output2.par output2.ift
$PROGRAM -ttm output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat output2.par 1>stdout.txt 2>&1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt format TSV-UK
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.par model radiowater-liver-TTM
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
parformat -f=ift output2.par
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.ift beta
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.ift PdT
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval output2.ift kGI
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


#printf "\n 1.1.1 \n"
#printf " Test case: Template parameter file used with decent input. \n"
#printf " Expected result: Simulated TTACs saved. \n\n"

#rm -f output2.tac output111.svg
#$PROGRAM output2.par lv.tac output2.tac
#if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
#printf "\n ok \n"
#tac2svg -mt="Test 1.1.1" -legend=y output111.svg -l lv.tac output2.tac
#if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
#printf "\n passed. \n\n"




printf "\n -----------------------------------------------------------------------\n"

printf "\n 2.0.0 \n"
printf " Test case: Vb is the only effective parameter. \n"
printf " Expected result: Correct output. \n\n"

rm -f output.tac

printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0	0	0.8	0.1	0	0	0\n" >> temp.par

taccalc lv.tac x 0.1 temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM temp.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.1 -rel=0.1 temp.tac output.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.1 \n"
printf " Test case: Vb and delay are the only effective parameters. \n"
printf " Expected result: Correct output. \n\n"

rm -f output.tac

printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0	0	0.8	0.1	10	0	0\n" >> temp.par

tactime -nogap temp.tac +10 temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -f=lv.tac temp2.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM temp.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.1 -rel=0.1 temp.tac output.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.0.2 \n"
printf " Test case: Vb, delay, and dispersion are the only effective parameters. \n"
printf " Expected result: Correct output. \n\n"

rm -f output.tac

printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0	1.0E-08	1	0.1	10	0	1\n" >> temp.par

simdisp lv.tac 60 10 temp2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taccalc temp2.tac x 0.1 temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

$PROGRAM temp.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg -mt="test 2.0.2" output202.svg -l temp.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=1 temp.tac output.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0	1.0E-08	1	0.1	0	10	1\n" >> temp.par

rm -f output2.tac
$PROGRAM temp.par lv.tac output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="test 2.0.2b" output202b.svg -l temp.tac output2.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=1 temp.tac output2.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0	1.0E-08	1	0.1	4	6	1\n" >> temp.par

rm -f output3.tac
$PROGRAM temp.par lv.tac output3.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="test 2.0.2c" output202c.svg -l temp.tac output3.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=1 temp.tac output3.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.1.0 \n"
printf " Test case: Only arterial flow. \n"
printf " Expected result: Correct output. \n\n"

rm -f output.tac

printf "# model := radiowater2\n" > h2o.par
printf "Parameters	f	p	PTF	Va	deltaT	tau\n" >> h2o.par
printf "tac1	1	0.5	1	0	0	0\n" >> h2o.par

printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	1	0	0.5	0	0	0	0\n" >> temp.par

sim_h2o h2o.par lv.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM temp.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tacmatch -abs=0.1 -rel=0.1 temp.tac output.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 2.1.1 \n"
printf " Test case: Only portal flow with delay and dispersion. \n"
printf " Expected result: Correct output. \n\n"

rm -f output.tac

printf "# model := radiowater2\n" > h2o.par
printf "Parameters	f	p	PTF	Va	deltaT	tau\n" >> h2o.par
printf "tac1	1	0.8	1	0	0	0\n" >> h2o.par

simdisp lv.tac 60 10 tempinp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi

printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0	1	0.8	0	10	0	1\n" >> temp.par

sim_h2o h2o.par tempinp.tac temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM temp.par lv.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg -mt="test 2.1.1" output211.svg -l temp.tac output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
tacmatch -abs=10 -rel=1 temp.tac output.tac 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n 2.2.0 \n"
printf " Test case: Arterial and portal flow with steady state input. \n"
printf " Expected result: Correct output level. \n\n"

printf "# model := radiowater-liver\n" > temp.par
printf "Parameters	K1a	K1p	p	Vb	LdT	PdT	kGI\n" >> temp.par
printf "tac1	0.5	1	0.8	0	10	10	1\n" >> temp.par

rm -f output.tac 

$PROGRAM temp.par ss.bld output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

tac2svg -mt="test 2.2.0" output220.svg -l ss.bld output.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

taccut -ls output.tac 0 0 temp.tac
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
taclist -nv -ift temp.tac > temp.ift
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
iftisval -abs=0.1 temp.ift value[1][1] 80
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

