#!/bin/bash
#: Title      : test_simdisp
#: Date       : 2023-06-29
#: Author     : "Vesa Oikonen" <vesa.oikonen@utu.fi>
#: Options    : None

# Set the name of executable to test
if [ -n "${OS+1}" ] && [ "$OS" = "Windows_NT" ]; then
EXT=.exe;
else
EXT=;
fi

PROGRAM=../../simdisp$EXT;
if [ ! -f $PROGRAM ]; then
  printf "Failed: executable does not exist.\n"
  exit 1
fi

printf "=====================================================================\n"
printf "creating test data for %s\n" $PROGRAM
printf "=====================================================================\n"


printf "\n verifying that required data exists \n"

if [ ! -f input.dat ] || [ ! -f blood.fit ]; then
  printf "Failed: required data does not exist.\n"
  exit 1
fi

if [ ! -f input1s.dat ] || [ ! -f inputs.svg ]; then
  interpol -c=0,60,1 input.dat input1s.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
  tac2svg -legend=n -mt="Input TACs" inputs.svg -s input.dat -l input1s.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f input01s.dat ]; then
  interpol -c=0,60,0.1 input.dat input01s.dat
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi

if [ ! -f blood.bld ]; then
  fit2dat -c=0,60,0.02 blood.fit blood.bld
  if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
fi




printf "\n\n"
printf "=====================================================================\n"
printf "testing %s\n" $PROGRAM
printf "=====================================================================\n"

printf "\n 1.0.0 \n"
printf " Test case 1: Simulating only dispersion. \n"
printf " Test case 2: Input data has sample frequency of 0.1 and starts from zero. \n"
printf " Test case 3: Correcting dispersion. \n"
printf " Expected result 1: Simulated TAC is the same as with convolution method. \n"
printf " Expected result 2: Dispersion corrected TAC is close to original. \n\n"


rm -f output01s_am.dat output01s_conv.dat

conv1tcm input01s.dat 0.4 0.4 output01s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM input01s.dat 2.5 0 output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Sample frequency of 0.1" sim01s.svg -l input1s.dat output01s_conv.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=1 output01s_conv.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM output01s_am.dat -2.5 0 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Dispersion correction" cor01s.svg -l input1s.dat output01s_conv.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=1 input01s.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"




printf "\n 1.0.1 \n"
printf " Test case 1: Simulating dispersion and delay. \n"
printf " Test case 2: Input data has sample frequency of 0.1 and starts from zero. \n"
printf " Test case 3: Correcting dispersion and delay. \n"
printf " Expected result 1: Simulated TAC is the same as with convolution method. \n"
printf " Expected result 2: Corrected TAC is close to original. \n\n"


rm -f output01s_am.dat

tactime output01s_conv.dat 1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -f=output01s_conv.dat temp.dat output01s_conv2.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM input01s.dat 2.5 1 output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Dispersion and delay added" sim01s2.svg -l input01s.dat output01s_conv2.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=1 output01s_conv2.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM  output01s_am.dat -2.5 -1
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Dispersion and delay correction" cor01s2.svg -l input01s.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=1 input01s.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.0.2 \n"
printf " Test case 1: Simulating and correcting only delay. \n"
printf " Test case 2: Input data has sample frequency of 0.1 and starts from zero. \n"
printf " Expected result 1: Simulated TAC is the same as with convolution method. \n"
printf " Expected result 2: Corrected TAC is close to original. \n\n"


rm -f output01s_am.dat

tactime input01s.dat 1 temp.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
interpol -f=input01s.dat temp.dat output01s.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM input01s.dat 0 1 output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.1 -rel=1 output01s.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"

$PROGRAM output01s_am.dat 0 -1 
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.1 -rel=1 input01s.dat output01s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n----------------------------------------------------------------\n"


printf "\n 1.1.0 \n"
printf " Test case 1: Simulating only dispersion. \n"
printf " Test case 2: Input data has sample frequency of one and starts from zero. \n"
printf " Expected result: Simulated TAC is not too far from the convolution method. \n\n"

rm -f output1s_am.dat output1s_conv.dat

conv1tcm -i=0.1 input1s.dat 0.4 0.4 output1s_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM input1s.dat 2.5 0 output1s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Sample frequency of 1" sim1s.svg -l input1s.dat output1s_conv.dat output1s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=0.5 -rel=1 output1s_conv.dat output1s_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.1.1 \n"
printf " Test case: Input data has variable sample frequency but starts from zero. \n"
printf " Expected result: Simulated TAC is not too far from convolution method. \n\n"

rm -f outputVs_am.dat outputVs_conv.dat

conv1tcm -i=0.1 input.dat 0.4 0.4 outputVs_conv.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM input.dat 2.5 0 outputVs_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Variable sample frequency" simVs.svg input.dat outputVs_conv.dat outputVs_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=2.0 -rel=10 outputVs_conv.dat outputVs_am.dat
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n----------------------------------------------------------------\n"

printf "\n 1.2.0 \n"
printf " Test case: Simulating only dispersion. \n"
printf " Expected result: Same results as with disp4dft. \n\n"

# dispersion constant 0.5 min
disp4dft ON blood.bld 0.5 output_disp_trad.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM blood.bld 0.5 0 output_disp_new.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tac2svg -mt="Dispersion" -x2=10 sim_disp.svg -l blood.bld output_disp_trad.bld output_disp_new.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch -abs=1 -rel=1 output_disp_trad.bld output_disp_new.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 1.2.1 \n"
printf " Test case: Name for output not given. \n"
printf " Expected result: Input file overwritten with simulated data. \n\n"

cp -f blood.bld test.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

$PROGRAM test.bld 0.5 0
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"

tacmatch output_disp_new.bld test.bld
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"





printf "\n===================================================================\n"
printf "   Test some errors \n"
printf "===================================================================\n"


printf "\n 3.0.0 \n"
printf " Test case: Invalid option. \n"
printf " Expected result: Error. \n\n"

$PROGRAM -stupidoption input.dat 1.0 2.0 output.dat 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid option '-stupidoption'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.1.0 \n"
printf " Test case: Extra argument. \n"
printf " Expected result: Error. \n\n"

$PROGRAM input.dat 1.0 2.0 output.dat extra 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n ok \n"
iftisval stdout.txt Error "invalid argument 'extra'."
if [ $? -ne 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"


printf "\n 3.2.0 \n"
printf " Test case: Missing parameter. \n"
printf " Expected result: Error. \n\n"

$PROGRAM input.dat 1.0 1>stdout.txt 2>&1
if [ $? -eq 0 ] ; then printf "Failed!\n" ; exit 1 ; fi
printf "\n passed. \n\n"



printf "\n===================================================================\n"
printf "   All passed! \n"
printf "===================================================================\n"
exit 0

